<?php
/**
 * @title OAuth Server
 * @desc OAuth Authorization Server Implementation
 */

use Defuse\Crypto\Key;
use League\OAuth2\Server\Grant\PasswordGrant;
use League\OAuth2\Server\Grant\ClientCredentialsGrant;
use League\OAuth2\Server\Grant\RefreshTokenGrant;
use League\OAuth2\Server\ResourceServer;
use League\OAuth2\Server\Exception;

class Bootstrap_Oauth extends Am_Module
{
    private $authorizationServer;
    private $encryptionKey;

    /**
     * @var OauthScopeTable
     */
    private $scopeRepository;
    private $resourceServer;

    const CONFIG_ACCESS_TOKEN_DEFAULT_LIFETIME = '1H';
    const CONFIG_REFRESH_TOKEN_DEFAULT_LIFETIME = '30D';
    const ADMIN_PERMISSION_CLIENTS = 'oauth-clients';
    const ADMIN_PERMISSION_TOKENS = 'oauth-tokens';

    /**
     * Add own scopes.
     */
    const EVENT_GET_SCOPES = 'oauthGetScopes';

    /**
     * Load resources, api controllers will be registered at this point
     * resource server is passed through this event
     */
    const EVENT_LOAD_RESOURCES = 'oauthLoadResources';
    
    const EVENT_CREATE_AUTHORIZATION_SERVER = 'oauthCreateAuthServer';
    
    /**
     * GRANT types
     */
    const GRANT_PASSWORD = 'password';
    const GRANT_CLIENT_CREDENTIALS = 'client_credentials';
    const GRANT_REFRESH_TOKEN = 'refresh_token';
    const GRANT_AUTHORIZATION_CODE = 'authorization_code';
    const GRANT_IMPLICIT = 'implicit';

    const GRANT_AMEMBER_DEVICE = 'amember_device';

    const DEVICE_COOKIE = 'am-oauth-device-id';

    function onAdminWarnings(\Am_Event $event)
    {
        if (version_compare(PHP_VERSION, '7.0.0', '<')) {
            $event->addReturn(___('At least php 7.0 required to run oauth server. Your PHP version is: ' . PHP_VERSION));
        }
    }

    function init()
    {
        $this->getDi()->router->addRoute('api2', new Am_Mvc_Router_Route(
            'api2/*', [
                'module' => 'oauth',
                'controller' => 'api',
                'action' => 'index',
            ]
        ));
    }

    function isConfigured()
    {
        return $this->getConfig('public_key') && $this->getConfig('private_key') && version_compare(PHP_VERSION,
                '7.0.0', '>=');
    }

    function onGetPermissionsList(\Am_Event $e)
    {
        $e->addReturn(___("OAuth2 server: Can add/delete/edit clients"), self::ADMIN_PERMISSION_CLIENTS);
        $e->addReturn(___("OAuth2 server: Can Add/remove access tokens"), self::ADMIN_PERMISSION_TOKENS);
    }

    function getEnabledGrantTypesOptions()
    {
        $options = [];
        foreach ($this->getAuthorizationServer()->getEnabledGrantTypes() as $grant) {
            $options[$grant->getIdentifier()] = $grant->getIdentifier();
        }
        return $options;
    }

    function getAvailableScopes()
    {
        return $this->getScopeRepository()->getScopeOptions();
    }

    /**
     * @return OauthScopeTable;
     */
    function getScopeRepository()
    {
        return $this->getResourceServer()->getScopeRepository();
    }

    function getRefreshTokenLifetimeInterval($returnSeconds = false)
    {
        $dateInterval = $this->configValueToDateInterval($this->getConfig('refresh_token_lifetime',
            self::CONFIG_REFRESH_TOKEN_DEFAULT_LIFETIME), 'D');
        if ($returnSeconds) {
            $t = new DateTime('now');
            return -($t->getTimestamp() - $t->add($dateInterval)->getTimestamp());
        } else {
            return $dateInterval;
        }
    }

    function getAccessTokenLifetimeInterval($returnSeconds = false)
    {
        $dateInterval = $this->configValueToDateInterval($this->getConfig('access_token_lifetime',
            self::CONFIG_ACCESS_TOKEN_DEFAULT_LIFETIME), 'H');
        if ($returnSeconds) {
            $t = new DateTime('now');
            return -($t->getTimestamp() - $t->add($dateInterval)->getTimestamp());
        } else {
            return $dateInterval;
        }
    }

    protected function configValueToDateInterval($string, $defaultInterval = 'H')
    {
        $string = strtoupper(trim($string));
        if (!preg_match('#^PT*(\d+)([YMDWHMS])#', $string, $regs)) {
            $string = intval($string) . $defaultInterval;
            $string = preg_match('#[HMS]#', $string) ? "PT$string" : "P$string";
        }

        return new DateInterval($string);
    }

    function createPasswordGrantType()
    {
        $passwordGrant = new Am_Oauth_Grant_Password($this->getDi()->oauthUserTable,
            $this->getDi()->oauthRefreshTokenTable);
        $passwordGrant->setRefreshTokenTTL($this->getRefreshTokenLifetimeInterval());
        return $passwordGrant;
    }

    function createRefreshTokenGrantType()
    {
        $refreshTokenGrant = new Am_Oauth_Grant_RefreshToken($this->getDi()->oauthRefreshTokenTable);
        $refreshTokenGrant->setRefreshTokenTTL($this->getRefreshTokenLifetimeInterval());
        return $refreshTokenGrant;
    }

    /**
     * @return Am_Oauth_Grant_AuthCode
     */
    function createAuthCodeGrantType()
    {
        $authCodeGrant = new Am_Oauth_Grant_AuthCode(
            $this->getDi()->oauthAuthCodeTable, $this->getDi()->oauthRefreshTokenTable, new \DateInterval('PT10M')
        ); // 10 mins expiration

        $authCodeGrant->setRefreshTokenTTL($this->getRefreshTokenLifetimeInterval());

        if ($this->getConfig('enable_pkce')) {
            $authCodeGrant->enableCodeExchangeProof();
        }

        return $authCodeGrant;
    }

    function createImplicitGrantType()
    {
        $implicitGrant = new Am_Oauth_Grant_Implicit(
            new \DateInterval('PT1H')
        );

        return $implicitGrant;
    }

    /**
     * @return Am_Oauth_Authorization_Server $server;
     */
    function getAuthorizationServer()
    {
        if (is_null($this->authorizationServer)) {
            $_ = new Am_Oauth_Server_Authorization(
                $this->getDi()->oauthClientTable,
                $this->getDi()->oauthAccessTokenTable,
                $this->getScopeRepository(),
                $this->getConfig('private_key'),
                $this->getEncryptionKey(),
                new Am_Oauth_Server_ResponseType_Bearer()
            );

            $_->enableGrantType(
                $this->createPasswordGrantType(), $this->getAccessTokenLifetimeInterval()
            );

            $_->enableGrantType(
                new \Am_Oauth_Grant_ClientCredentials, $this->getAccessTokenLifetimeInterval()
            );

            if ($this->getConfig('require_device')) {
                $_->enableGrantType(
                    new \Am_Oauth_Grant_AmemberDevice(), $this->getAccessTokenLifetimeInterval()
                );
            }

            $_->enableGrantType(
                $this->createRefreshTokenGrantType(), $this->getAccessTokenLifetimeInterval()
            );

            $_->enableGrantType(
                $this->createAuthCodeGrantType(), $this->getAccessTokenLifetimeInterval()
            );

            $_->enableGrantType(
                $this->createImplicitGrantType(), new DateInterval('PT1H')
            );
            
            $this->authorizationServer = $this->getDi()->hook->filter($_, Bootstrap_Oauth::EVENT_CREATE_AUTHORIZATION_SERVER);
        }

        return $this->authorizationServer;
    }

    function getEncryptionKey()
    {
        if (is_null($this->encryptionKey)) {
            $store_key = $this->getDi()->store->getBlob('oauth-encryption-key');

            if (!$store_key) {
                $key = Key::createNewRandomKey();
                $this->getDi()->store->setBlob('oauth-encryption-key', $key->saveToAsciiSafeString());
            } else {
                $key = Key::loadFromAsciiSafeString($store_key);
            }
            $this->encryptionKey = $key;
        }
        return $this->encryptionKey;
    }

    function onAdminMenu(Am_Event $event)
    {
        $event->getMenu()
            ->addPage([
                'id' => 'oauth',
                'label' => ___('Oauth 2 Server'),
                'uri' => 'javascript:',
                'resource' => self::ADMIN_PERMISSION_CLIENTS,
                'controller' => 'admin-clients',
                'module' => 'oauth',
                'pages' => [
                    [
                        'id' => 'oauth-clients',
                        'controller' => 'admin-clients',
                        'module' => 'oauth',
                        'label' => ___('Clients'),
                        'resource' => self::ADMIN_PERMISSION_CLIENTS,
                    ],
                    [
                        'id' => 'oauth-tokens',
                        'controller' => 'admin-tokens',
                        'module' => 'oauth',
                        'label' => ___("Access Tokens"),
                        'resource' => self::ADMIN_PERMISSION_TOKENS,
                    ],
                ]
            ]);
    }

    function onDaily(Am_Event $event)
    {
        $this->getDi()->oauthAccessTokenTable->cleanUp();
        $this->getDi()->oauthRefreshTokenTable->cleanUp();
        $this->getDi()->oauthAuthCodeTable->cleanUp();
    }

    /**
     * Initiailize and return instance of resource server;
     * @return Am_Oauth_Server_Resource
     */
    function getResourceServer()
    {
        if (is_null($this->resourceServer)) {
            $resourceServer = new Am_Oauth_Server_Resource(
                $this->getDi()->oauthAccessTokenTable, $this->getConfig('public_key')
            );

            $resourceServer->setDi($this->getDi());
            $resourceServer->bootstrap();

            $this->resourceServer = $this->getDi()->hook->filter($resourceServer, new Am_Oauth_Event_Resource());
            
        }
        return $this->resourceServer;
    }

    function onApiCheckPermissions(Am_Event $event)
    {
        $request = $event->getRequest();

        if (!$request->getHeader('Authorization')) {
            return;
        }

        $alias = $event->getAlias();
        $method = $event->getMethod();
        try {
            $request = $this->getResourceServer()->validateAuthenticatedRequest(Am_Mvc_Oauth_Request::fromGlobals());
        } catch (\Exception $exception) {
            throw new Am_Exception_InputError($exception->getMessage());
        }
        $allowedScopes = $request->getAttribute('oauth_scopes');

        $scopeId = implode('.', ['rest', $alias, $method]);
        if (in_array($scopeId, (array)$allowedScopes)) {
            return $event->addReturn(true);
        }

        throw new Am_Exception_InputError(___("Access Denied. You have no permissions to request %s %s", $alias,
            $method));
    }

    function onSavedFormTypes(Am_Event $event)
    {
        $event->getTable()->addTypeDef([
            'type' => 'oauth-register',
            'class' => 'Am_Oauth_Form_Signup',
            'title' => ___('Api2 Registration Form'),
            'defaultTitle' => ___('Api2  Registration Form'),
            'defaultComment' => ___('Form will be used to validate user registration request through /api2/register endpoint'),
            'generateCode' => false,
            'urlTemplate' => ['Am_Oauth_Form_Signup', 'getSavedFormUrl'],
            'isSingle' => true,
            'isSignup' => true,
            'noDelete' => true,
        ]);

        $event->getTable()->addTypeDef([
            'type' => 'oauth-profile',
            'class' => 'Am_Oauth_Form_Profile',
            'title' => ___('Api2 User Profile Form'),
            'defaultTitle' => ___('Api2 User Profile Form'),
            'defaultComment' => ___('Form will be used to validate user profile update request through /api2/user/profile endpoint'),
            'generateCode' => false,
            'urlTemplate' => ['Am_Oauth_Form_Profile', 'getSavedFormUrl'],
            'isSingle' => true,
            'isSignup' => false,
            'noDelete' => true,
        ]);
    }

    protected static
    function setUpFormIfNotExist(
        DbSimple_Interface $db
    ) {
        $tbl = Am_Di::getInstance()->savedFormTable->getName();
        if (!$db->selectCell("SELECT COUNT(*) FROM {$tbl} WHERE type=?", 'oauth-register')) {
            $max = $db->selectCell("SELECT MAX(sort_order) FROM {$tbl}");
            $db->query("INSERT INTO {$tbl} (title, comment, type, fields, sort_order)
                VALUE (?a)", [
                'Api2 Registration Form',
                'Form will be used to validate user registration request through /api2/register endpoint',
                'oauth-register',
                '[{"id":"name","class":"name","hide":"1"},{"id":"email","class":"email","hide":true},{"id":"login","class":"login","hide":true},{"id":"password","class":"password","hide":true},{"id":"address","class":"address","hide":"1","config":{"fields":{"street":1,"city":1,"country":1,"state":1,"zip":1}}}]',
                ++$max
            ]);
        }
        if (!$db->selectCell("SELECT COUNT(*) FROM {$tbl} WHERE type=?", 'oauth-profile')) {
            $max = $db->selectCell("SELECT MAX(sort_order) FROM {$tbl}");
            $db->query("INSERT INTO {$tbl} (title, comment, type, fields, sort_order)
                VALUE (?a)", [
                'Api2 User Profile  Form',
                'Form will be used to validate user profile update request through /api2/user/profile endpoint',
                'oauth-profile',
                '[{"id":"name","class":"name","hide":1},{"id":"email","class":"email","hide":true},{"id":"new-password","class":"new-password","hide":1,"config":{"do_not_ask_current_pass":"1","do_not_confirm":"1"}}]',
                ++$max
            ]);
        }
    }

    function onDbUpgrade(Am_Event $e)
    {
        echo "Set Up API Registration Form...";
        $this->setUpFormIfNotExist($this->getDi()->db);
        echo "Done<br>\n";
    }

    function onUserAfterDelete(Am_Event $e)
    {
        $user = $e->getUser();
        foreach (['?_oauth_access_token', '?_oauth_auth_code', '?_oauth_refresh_token'] as $tbl) {
            $this->getDi()->db->query("update {$tbl} set is_revoked=1 where user_id=?", $user->pk());
        }
    }

    function onDeletePersonalData(Am_Event $e)
    {
        $user = $e->getUser();
        foreach (['?_oauth_access_token', '?_oauth_auth_code', '?_oauth_refresh_token'] as $tbl) {
            $this->getDi()->db->query("update {$tbl} set is_revoked=1 where user_id=?", $user->pk());
        }
    }

    function setDeviceCookie($device_id)
    {
        Am_Cookie::set(self::DEVICE_COOKIE, $device_id, time() + 3600 * 24 * 365 * 10, '/',
            $this->getDi()->request->getHttpHost());
    }

    function getDeviceCookie()
    {
        return !empty($_COOKIE[self::DEVICE_COOKIE]) ? $_COOKIE[self::DEVICE_COOKIE] : null;
    }

    function sendConfirmationEmail(User $user)
    {
        $code = $this->getDi()->security->randomString(6, '0123456789');

        $data = [
            'security_code' => $code,
            'email' => $user->email
        ];

        $this->getDi()->store->setBlob(
            'member-verify-email-oauth-' . $user->user_id,
            serialize($data),
            sqlTime($this->getDi()->time + 48 * 3600)
        );

        $tpl = Am_Mail_Template::load('oauth.confirm_email', get_first($user->lang,
            $this->getDi()->app->getDefaultLocale(false)), true);

        $tpl->setUser($user);
        $tpl->setCode($code);

        $tpl->setUrl(
            $this->getDi()->url('oauth/confirm-email',
                ['em' => $this->getDi()->security->obfuscate($user->pk()) . '-' . $code]
                , false, true)
        );
        $tpl->send($user);
    }

    function confirmEmailAddress(User $user, $code)
    {
        if ($user->email_confirmed) {
            return true;
        }

        $data = $this->getDi()->store->getBlob('member-verify-email-oauth-' . $user->user_id);
        if (empty($data)) {
            return false;
        }

        $data = @unserialize($data);

        if (!is_array($data)) {
            return false;
        }

        if ($data['security_code'] == $code && $data['email'] == $user->email) {
            // $user->setEmailAddressConfirmed();
            /**
             * @todo Remove this before release;
             */
            $user->email_confirmed = true;
            $user->email_confirmation_date = $this->getDi()->sqlDateTime;
            $user->save();
            return true;
        }

        return false;
    }
}