<?php


abstract class Am_Oauth_Api_Resource_Object extends Am_Oauth_Api_Resource_Abstract
{
    /** @var array parsed class dockblock to get variables */
    private $_docBlock = [];
    /**
     *
     * @param type $endpoint
     * @param type $scopeId
     */
    function __construct()
    {
        $this->setEndpoint($this->getBaseEndpoint());

        $this->method = [];

        foreach ($this->possibleMethods() as $method)
        {
            if ($this->isImplemented($method))
            {
                $this->method[] = $method;
            }
        }
        $this->setupScopes();
    }

    function isEnabled()
    {
        return true;
    }


    function setupScopes()
    {
        $this->scopes[] = OauthScope::create(
            $this->getBaseScopeId(),
            $this->getDescription(),
            $this->getAllowedGrantTypes()
        );

        if (count($this->method) > 1) {
            foreach ($this->method as $method) {
                $this->scopes[] = OauthScope::create(
                    $this->getBaseScopeId() . '.' . $method,
                    $this->getDescription() . " Method: " . $method,
                    $this->getAllowedGrantTypes()
                );
            }
        }
    }

    private function parseDockBlock()
    {
        if (!$this->_docBlock)
        {
            $rc = new ReflectionClass($this);
            $docText = $rc->getDocComment();
            $descFinished = false;
            $desc = [];
            foreach (explode("\n", $docText) as $line) {
                $line = rtrim(ltrim($line, "*\t "));
                if (strpos($line, '/**') === 0) continue; // skip first line

                if (preg_match('#^@\w+#', $line))
                    $descFinished = true;
                elseif (!$descFinished)
                    $desc[] = $line;
                if (preg_match('#^@(Api.+?)\s+(.+)$#', $line, $regs)) {
                    $this->_docBlock[$regs[1]] = $regs[2];
                }
            }
            $this->_docBlock['ClassDescription'] = implode("\n", $desc);
        }
        return $this->_docBlock;
    }

    private function fetchDocBlockVar($var, $isCommaDelimitedArray=false, $throwException = true)
    {
        $docBlock = $this->parseDockBlock();
        $_ = empty($docBlock[$var]) ? null : $docBlock[$var];
        if (empty($_) && $throwException)
            throw new Am_Exception_InternalError("API Resource definition incorrect - no @$var specified at " . get_class($this));
        if ($isCommaDelimitedArray)
            $_ = array_filter(array_map('trim', explode(',', $_)));
        return $_;
    }

    /**
     * Short description about resource. Description will be used when scopes will be created.
     *
     */
    function getDescription()
    {
        return $this->fetchDocBlockVar('ApiDescription');
    }

    /**
     * return resource endpoint;
     */

    function getBaseEndpoint()
    {
        return $this->fetchDocBlockVar('ApiBaseEndpoint');
    }

    /**
     * scope for resource.
     * From that scope resource will create separate scope for each method.
     * For example if base scope is 'test'
     * Depends on implemented methods these scopes will be created:
     * test.post, test.get, test.delete, test.put
     * @return string
     */
    function getBaseScopeId()
    {
        return $this->fetchDocBlockVar('ApiBaseScope');
    }


    /**
     * Should return array of allowed grant types;
     * Please  think about how controller will be used.
     * If you want to use it for actions on user behalf, allow only
     * authorisation_code or password grant types.
     * If that controller should be used to run admin actions,
     * allow only client_credentials grant type
     */
    function getAllowedGrantTypes()
    {
        static $_oapiGrantConst = [];
        if (!$_oapiGrantConst)
        {
            $rc = new ReflectionClass('Bootstrap_Oauth');
            foreach ($consts = $rc->getConstants() as $k => $v)
                if (strpos($k, 'GRANT_')===0)
                    $_oapiGrantConst[$k] = $v;
        }
        return array_map(function ($_) use ($_oapiGrantConst) { // replace uppercase strings to constant from Bootstrap_Oauth
                if ((strtoupper($_) == $_) && array_key_exists($_, $_oapiGrantConst))
                    return $_oapiGrantConst[$_];
                else
                    return $_;
            },
            $this->fetchDocBlockVar('ApiAllowedGrantTypes', true));
    }

    /**
     * Override this method if resource should handle post (insert requests)
     * @param Am_Oauth_Api_Request $request Incoming request;
     * @param Am_Oauth_Api_Response $response Am_Oauth_Api_Response object
     * @param type $args
     * @return \Am_Oauth_Api_Response_NotImplemented
     */

    function post(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        return new Am_Oauth_Api_Response_NotImplemented;
    }


    /**
    * Override this method if resource should handle get requests
    * @param Am_Oauth_Api_Request $request
    * @param Am_Oauth_Api_Response $response
    * @param type $args
    * @return \Am_Oauth_Api_Response_NotImplemented
    */
    function get(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        return new Am_Oauth_Api_Response_NotImplemented;
    }

    /**
     * Override this method if resource should handle put (update requests)
     * @param Am_Oauth_Api_Request $request
     * @param Am_Oauth_Api_Response $response
     * @param type $args
     * @return \Am_Oauth_Api_Response_NotImplemented
     */
    function put(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        return new Am_Oauth_Api_Response_NotImplemented;
    }

    /**
     * Override this method if resource should handle delete requests.
     * @param Am_Oauth_Api_Request $request
     * @param Am_Oauth_Api_Response $response
     * @param type $args
     * @return \Am_Oauth_Api_Response_NotImplemented
     */
    function delete(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        return new Am_Oauth_Api_Response_NotImplemented;
    }

    function isImplemented($method)
    {
        $ref = new \ReflectionMethod($this, $method);
        return ($ref->getDeclaringClass()->getName() === get_class($this));
    }

    public function __invoke(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        foreach ($this->possibleMethods() as $method)
        {
            if ($request->getMethod() == strtoupper($method))
            {
                return $this->{$method}($request, $response, $args);
            }
        }

        return new Am_Oauth_Api_Response_NotImplemented();
    }

    function getValidateScopesCallback($method)
    {
        return new Am_Oauth_Api_Middleware_Scope([$this->getBaseScopeId(), $this->getBaseScopeId() . '.' . $method]);
    }

}
