<?php

/**
 * Post user signup form
 *
 * @ApiAllowedGrantTypes GRANT_CLIENT_CREDENTIALS, GRANT_AMEMBER_DEVICE
 * @ApiBaseEndpoint /register
 */
class Am_Oauth_Api_Resource_Register extends Am_Oauth_Api_Resource_Object
{
    use \Am_Mvc_Controller_User_Create;
    use  \Am_Oauth_Traits_GetFormErrors;

    function getDescription()
    {
        return ___("Create user account");
    }
    
    function getBaseScopeId()
    {
        return "register";
    }
    
    public function setupScopes()
    {
        $this->scopes = [
            new OauthScope(
                'register',
                $this->getDescription(),
                [
                    Bootstrap_Oauth::GRANT_CLIENT_CREDENTIALS,
                    Bootstrap_Oauth::GRANT_AMEMBER_DEVICE,
                ]),
            new OauthScope(
                'register.post',
                $this->getDescription()." Method: post",
                [
                    Bootstrap_Oauth::GRANT_CLIENT_CREDENTIALS,
                    Bootstrap_Oauth::GRANT_AMEMBER_DEVICE,
                ])
        ];
    }
    

    function createSignupForm(Am_Oauth_Api_Request $request)
    {
        $form = new \Am_Form('oauth-api-register');

        if (!class_exists('Am_Form_Brick', false)) {
            class_exists('Am_Form_Brick', true);
            \Am_Di::getInstance()->hook->call(\Am_Event::LOAD_BRICKS);
        }
        $savedForm = $this->getDi()->savedFormTable->getByType('oauth-register');

        if (!$savedForm) {
            throw new Am_Exception_InputError(___('Form is not configured'));
        }

        foreach ($savedForm->getBricks() as $brick) {
            if (($brick->getId() == 'email') && $brick->getConfig('validate')) {
                $form->addHidden('_confirm_email')->setValue(1);
            }
            $brick->insertBrick($form);
        }

        $form->addRule(
            'callback',
            '--form error--',
            ['callback' => [$this, 'validate'], 'arguments' => [$form, $savedForm]]
        );

        $form->removeElementByName('_pass');
        $form->addHidden('saved_form_id')->setValue($savedForm->pk());
        $form->addHidden('remote_addr')->setValue($request->getAttribute('ip_address'));
        $form->addHidden('user_agent')->setValue($request->getServerParams()['HTTP_USER_AGENT']??"");

        return $form;
    }

    function validate($vars, $form, $savedForm)
    {
        $event = new Am_Event_ValidateSavedForm($vars, $form, $savedForm);
        $this->getDi()->hook->call($event);
        if ($errors = $event->getErrors())
        {
            $form->setError($errors[0]);
            return false;
        }
        return true;
    }

    public function post(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {

        $form = $this->createSignupForm($request);

        $form->addDataSource($request);

        if(!$form->validate())
        {
            return new \Am_Oauth_Api_Response_ValidationError(
                ___('An error happened when we try to validate your request. Please check hints for more details'),
                $this->getFormErrors($form)
                );
        }

        $vars = $form->getValue();

        unset($vars['user_id']); // No games with user_id;
        $vars['email_confirmed'] = false;
        unset($vars['email_confirmation_date']);

        $user = $this->createUser($vars);
        if($vars['_confirm_email'])
        {
            $this->getModule()->sendConfirmationEmail($user);
        }

        $user->pass = $user->getPlaintextPass();

        $client_id = $request->getOauthClientId();

        $client = $this->getDi()->oauthClientTable->findFirstBy(['client_id' => $client_id]);

        if($client && $client->isAllowedGrantType(Bootstrap_Oauth::GRANT_PASSWORD))
        {
            $scopes = $client->getAllowedScopes(Bootstrap_Oauth::GRANT_PASSWORD);
            $user_id = $user->pk();

            $passwordGrant = $this->getModule()->getAuthorizationServer()->getGrantTypeById(Bootstrap_Oauth::GRANT_PASSWORD);

            $accessToken = $passwordGrant->userIssueAccessToken(
                $this->getModule()->getAccessTokenLifetimeInterval(), $client, $user_id, $this->getModule()->getScopeRepository()->getScopeEntityByIdentifier($scopes)
            );

        $responseParams = [
            'token_type'   => 'Bearer',
            'expires_in'   => $this->getModule()->getAccessTokenLifetimeInterval(3600),
            'access_token' => (string)$accessToken,
            'scopes' => implode(' ', $scopes)
        ];

        }

        return (!empty($responseParams) ? $response->withDataAndNewToken($user, 201, $responseParams) : $response->withSuccessData($user, 201));
    }

}