<?php

/**
 * @ApiAllowedGrantTypes GRANT_AUTHORIZATION_CODE, GRANT_PASSWORD, GRANT_IMPLICIT
 * @ApiBaseEndpoint /user/invoice[/{invoice_id}]
 * @ApiBaseScope user.invoice
 */
class Am_Oauth_Api_Resource_UserInvoice extends Am_Oauth_Api_Resource_Object
{

    function getDescription()
    {
        return ___("Manage user's invoices");
    }

    public function get(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        $user_id = $request->getUserId();

        $req = [
            'user_id' => $user_id,
        ];
        if(isset($args['invoice_id']))
        {
            $req['invoice_id'] =  $args['invoice_id'];
        }

        $invoices = $request->getDi()->invoiceTable->findBy($req);

        if($args['invoice_id'] && !$invoices)
        {
            return new Am_Oauth_Api_Response_NotFound();
        }


        foreach($invoices as $key => $invoice)
        {
            if($invoice->status == Invoice::PENDING)
            {
                $invoices[$key] = $this->createPaymentLink($invoice);
            }
        }

        return $response->withSuccessInvoice($invoices);

    }


    function createPaymentLink(Invoice $invoice)
    {
        if(empty($invoice->due_date))
        {
            $invoice->due_date= sqlDate('+7 days');
            $invoice->updateSelectedFields(['due_date']);
        }

        if($invoice->due_date >= $this->getDi()->sqlDate)
        {
            $invoice->paymentLink = $this->getDi()->surl("pay/{$invoice->getSecureId('payment-link')}");
        }
        return $invoice;

    }

    function addProductsToInvoice(array $products, Invoice $invoice)
    {
        foreach($products as $prConfig)
        {
            if(!is_array($prConfig))
            {
                $prConfig = ['product_id' => intval($prConfig)];
            }

            $product = $this->getDi()->productTable->load($prConfig['product_id']);

            if(isset($prConfig['billing_plan_id']))
            {
                $product->setBillingPlan($prConfig['billing_plan_id']);
            }

            $qty = 1;
            if(isset($prConfig['qty']))
            {
                $qty = intval($prConfig['qty']);
                if(!$product->getBillingPlan()->variable_qty || ($qty < 0))
                {
                    $qty = 1;
                }
            }

            $invoice->add($product, $qty);
        }

        return $invoice;
    }


    function processInvoice(Invoice $invoice, Am_Mvc_Request_Interface $request)
    {
        $this->getDi()->hook->call(Am_Event::INVOICE_BEFORE_PAYMENT,
            [
                'invoice' => $invoice
            ]);

        $plugin = $this->getDi()->plugins_payment->loadGet($invoice->paysys_id);

        $result = new Am_Paysystem_Result();
        $plugin->processInvoice($invoice, $request, $result);

        if (($result->isSuccess() || $result->isFailure()) && ($transaction = $result->getTransaction()))
        {
            $transaction->setInvoice($invoice);
            $transaction->process();

        }

        return $result;
    }

    public function post(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        $user_id = $request->getUserId();

        if(empty($user_id))
        {
            return new Am_Oauth_Api_Response_Forbidden();
        }

        $products = $request->getParam('products');

        if(empty($products))
        {
            return new Am_Oauth_Api_Response_BadRequest(___('Products are missing in incoming request'));
        }

        if(!is_array($products))
        {
            $products = [$products];
        }

        $invoice = $request->getDi()->invoiceRecord;

        $invoice->setUser($request->getDi()->userTable->load($user_id));

        $invoice = $this->addProductsToInvoice($products, $invoice);

        $coupon = $request->getParam('coupon');

        if(!empty($coupon))
        {
            $invoice->setCouponCode($coupon);
            $errors = $invoice->validateCoupon();
            if($errors)
            {
                return new Am_Oauth_Api_Response_ValidationError (___('Unable to validation coupon code'), $errors);
            }

        }
        $paysys_id = $request->getParam('paysys_id');
        if(empty($paysys_id))
        {
            $paysys_id = 'free';
        }

        $invoice->calculate();

        $invoice->setPaysystem($paysys_id);

        $err = $invoice->validate();

        if(!empty($err))
        {
            return new Am_Oauth_Api_Response_ValidationError(___('Unable to validate invoice record'), $err);
        }

        $invoice->insert();

        try
        {
            $result = $this->processInvoice($invoice, $request);
        }
        catch (Am_Exception_Redirect $e)
        {
            $invoice->refresh();
            if (!$invoice->isCompleted())
            {
                $invoice->paymentLink = $e->url;
            }

        }

        if($result->isFailure())
        {
            return new Am_Oauth_Api_Response_BadRequest($result->getErrorMessages());
        }
        else if($result->isAction())
        {
            $action = $result->getAction();

            if($action instanceof Am_Paysystem_Action_Redirect)
            {
                $invoice->paymentLink = $action->getUrl();
            }
            else
            {
                // Unable to handle response here, let's do this via default pay controller.
                $invoice = $this->createPaymentLink($invoice);
            }
        }

        return $response->withSuccessInvoice($invoice);
    }

    function delete(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        $user = $request->getUser();

        $invoice_id = @$args['invoice_id'];
        if(empty($invoice_id))
        {
            return new Am_Oauth_Api_Response_BadRequest ('Invoice ID is empty in incoming request.');
        }
        $invoice = $request->getDi()->invoiceTable->load($invoice_id);

        if($invoice->user_id != $user->pk())
        {
            new Am_Oauth_Api_Response_BadRequest ('Unable to cancel invoice related to another user');
        }

        $result = new Am_Paysystem_Result();
        try
        {
            $paysys = $invoice->getPaysystem();
            if(!is_null($paysys))
                $paysys->cancelAction($invoice, 'cancel-admin', $result);
        }
        catch (Exception $e)
        {
            Am_Di::getInstance()->logger->error("oauth: unable to cancel invoice ", ["exception" => $e]);
            return new Am_Oauth_Api_Response_ValidationError(___('Unable to cancel invoice #%s: %s', $invoice->pk(), $e->getMessage()));
        }

        if($result->isAction() && ($result->getAction() instanceof Am_Paysystem_Action_Redirect))
        {
            return new Am_Oauth_Api_Response_Redirect($result->getAction()->getUrl());

        }else if ($result->getStatus() != Am_Paysystem_Result::SUCCESS)
        {
            return new Am_Oauth_Api_Response_ValidationError(___('Unable to cancel invoice #%s: %s', $invoice->pk(), $statusMsg));
        }
        // Success
        return $this->get($request, $response, $args);
    }

    function put(Am_Oauth_Api_Request $request, Am_Oauth_Api_Response $response, $args)
    {
        $user = $request->getUser();

        $invoice_id = @$args['invoice_id'];

        if(empty($invoice_id))
        {
            return new Am_Oauth_Api_Response_BadRequest ('Invoice ID is empty in incoming request');
        }

        $invoice = $request->getDi()->invoiceTable->load($invoice_id);

        if($invoice->isCompleted())
        {
            return new Am_Oauth_Api_Response_BadRequest('Unable to update completed invoice');
        }


        if($invoice->user_id != $user->pk())
        {
            new Am_Oauth_Api_Response_BadRequest ('Unable to cancel invoice related to another user');
        }

        $products = $request->getParam('products');


        if(!empty($products) && !is_array($products))
        {
            $products = [$products];
        }

        if(!empty($products))
        {
            foreach($invoice->getItems() as $item)
            {
                $invoice->deleteItem($item);
            }

            $invoice = $this->addProductsToInvoice($products, $invoice);
        }


        $coupon = $request->getParam('coupon');

        if(!empty($coupon))
        {
            $invoice->setCouponCode($coupon);
            $errorors = $invoice->validateCoupon();
            if($errors)
            {
                return new Am_Oauth_Api_Response_ValidationError (___('Unable to validation coupon code'), $errors);
            }

        }
        $paysys_id = $request->getParam('paysys_id');

        if(empty($paysys_id))
        {
            $paysys_id = 'free';
        }


        $invoice->calculate();

        $invoice->setPaysystem($paysys_id);

        $err = $invoice->validate();

        if(!empty($err))
        {
            return new Am_Oauth_Api_Response_ValidationError(___('Unable to validate invoice record'), $err);
        }

        $invoice->update();

        try
        {
            $result = $this->processInvoice($invoice, $request);
        }
        catch (Am_Exception_Redirect $e)
        {
            $invoice->refresh();
            if (!$invoice->isCompleted())
            {
                $invoice->paymentLink = $e->url;
            }

        }

        if($result->isFailure())
        {
            return new Am_Oauth_Api_Response_BadRequest($result->getErrorMessages());
        }
        else if($result->isAction())
        {
            $action = $result->getAction();

            if($action instanceof Am_Paysystem_Action_Redirect)
            {
                $invoice->paymentLink = $action->getUrl();
            }
            else
            {
                // Unable to handle response here, let's do this via default pay controller.
                $invoice = $this->createPaymentLink($invoice);
            }
        }
        return $response->withSuccessInvoice($invoice);
    }

}


