<?php

use Slim\App;
use Slim\Factory\AppFactory;
use Slim\Factory\ServerRequestCreatorFactory;
use League\OAuth2\Server\ResourceServer;
use League\OAuth2\Server\Middleware\ResourceServerMiddleware;
use RKA\Middleware\IpAddress;

class Am_Oauth_Server_Resource extends ResourceServer
{

    protected
        $resourses = [],
        $_di,
        $scopeRepository,
        $router;
    
    

    function setDi($di)
    {
        $this->_di = $di;
    }

    /**
     * 
     * @return Am_Di
     */
    function getDi()
    {
        return $this->_di;
    }

    function setScopeRepository($scopeRepository)
    {
        $this->scopeRepository = $scopeRepository;
    }
    
    /**
     * 
     * @return Slim\App $router;
     */
    function getRouter()
    {
        return $this->router;
    }
    /**
     * 
     * @return OauthScopeTable
     */
    function getScopeRepository()
    {
        return $this->scopeRepository;
    }

    function addResource($endpoint, $method=null, $scopes = null, $callback = null)
    {
        if ($endpoint instanceof Am_Oauth_Api_Resource_Abstract)
        {
            $resource = $endpoint;
            $resource->setDi($this->getDi());
            $scopes = $resource->getScopes();
        }
        else
        {
            $resource = new Am_Oauth_Api_Resource_Callback();
            $resource
                ->setMethod($method)
                ->setEndpoint($endpoint)
                ->setScopes($scopes)
                ->setCallback($callback)
                ->setDi($this->getDi());
                
        }
        
        $this->getScopeRepository()->addScope($scopes);
        
        $this->resourses[] = $resource;
        
        return $this;
    }

    /**
     * Setup hooks
     */
    function bootstrap()
    {
        /*
         * Init Scope repository
         * 
         */
        $scopeRepository = new \OauthScopeTable;
        
        $this->scopeRepository = $this->getDi()->hook->filter($scopeRepository, Bootstrap_Oauth::EVENT_GET_SCOPES);
    
        /**
         * Load Resource config
         */
        $this->loadResourcesList();

        /**
         * Init router
         */
        
        ServerRequestCreatorFactory::setServerRequestCreator(new Am_Oauth_Api_Request_Factory);
        
        $this->router = AppFactory::create(new Am_Oauth_Api_Response_Factory());
        
        $this->router->setBasePath(REL_ROOT_URL);
        
        $this->router->add(new IpAddress());
        
    }

    /**
     * Function loads available resources
     */
    function loadResourcesList()
    {
        $resDir = dirname(__DIR__) . '/Api/Resource'; // application/oauth/library/Am/Oauth/Resource
        $it = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($resDir));
        $hook = $this->getDi()->hook;
        foreach ($it as $item)
        {
            if ($item->getExtension() == 'php')
            {
                $fn = substr($item->getPathname(), strlen($resDir)+1);
                if (in_array($fn, ['Callback.php', 'Abstract.php', 'Object.php']))
                    continue;
                // temp. hack
                // skip including newsletter resources if module is not enabled. normally we should put these 2 classes into newsletter module to avoid this hacking
                if((strpos($fn, 'UserNewsletter') === 0) && !$this->getDi()->modules->isEnabled('newsletter'))
                    continue;
                //
                $class = 'Am_Oauth_Api_Resource_' . preg_replace('#\.php$#', '', $fn);
                if(!class_exists($class, false)) {
                    
                    include_once($item->getPathname());
                    $hook->add(\Bootstrap_Oauth::EVENT_LOAD_RESOURCES,
                        function (\Am_Event $event) use ($class)
                        {
                            $resource = new $class;
                            if($resource->isEnabled()) {
                                $event->getReturn()->addResource($resource);
                            }
                        });
                }
            }
        }
    }
    
    function initRoutes()
    {
        foreach($this->resourses as $resource)
        {
            foreach($resource->getMethods() as $method)
            {
                $this->router
                    ->map([strtoupper($method)], $resource->getEndpoint(), $resource)
                    ->add($resource->getValidateScopesCallback($method));
            }
        }
        if($this->getDi()->modules->loadGet('oauth')->getConfig('require_device'))
        {
            $this->router->add(new Am_Oauth_Api_Middleware_Device());
        }
        
        $this->router->add(new Am_Oauth_Server_Resource_Middleware($this));
        $this->router->addRoutingMiddleware();
    
        $errorMiddleware = $this->router->addErrorMiddleware(defined('AM_DEBUG') && AM_DEBUG, true, true);
        $errorMiddleware->setDefaultErrorHandler(function(Am_Oauth_Api_Request $request, Throwable $ex)
        {
            return new Am_Oauth_Api_Response_FatalError($ex->getMessage());
        });
    
    }
    
    function run()
    {
        
        try
        {
            return $this->router->run();
        }
        catch(Exception $ex)
        {
            return new Am_Oauth_Api_Response_FatalError($ex->getMessage());
        }
    }
    

}
