package co.pixelbeard.theanfieldwrap.downloads;


import android.content.Context;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import butterknife.BindView;
import butterknife.ButterKnife;
import co.pixelbeard.theanfieldwrap.R;
import co.pixelbeard.theanfieldwrap.data.Podcast;
import co.pixelbeard.theanfieldwrap.data.RealmPodcast;
import co.pixelbeard.theanfieldwrap.data.source.DataRepository;
import co.pixelbeard.theanfieldwrap.data.source.local.LocalRepository;
import co.pixelbeard.theanfieldwrap.data.source.remote.RemoteRepository;
import co.pixelbeard.theanfieldwrap.home.HomeFragmentListener;
import co.pixelbeard.theanfieldwrap.playerService.MediaPlayerInterface;
import co.pixelbeard.theanfieldwrap.utils.BaseFragment;
import co.pixelbeard.theanfieldwrap.utils.ConnectionUtils;
import co.pixelbeard.theanfieldwrap.utils.FontUtils;
import co.pixelbeard.theanfieldwrap.utils.TawController;
import io.realm.Realm;
import io.realm.RealmResults;

public class DownloadsFragment extends BaseFragment implements DownloadsContract.View {

    private static final String TAG = DownloadsFragment.class.getSimpleName();

    @BindView(R.id.ll_back)
    LinearLayout llBack;

    @BindView(R.id.txt_downloads_title)
    TextView txtDownloadsTitle;
    @BindView(R.id.rv_downloads)
    RecyclerView rvDownloads;
    @BindView(R.id.txt_no_downloads)
    TextView txtNoDownloads;


    private Context mContext;
    private HomeFragmentListener mListener;
    private MediaPlayerInterface mediaPlayerInterface;

    private DownloadsContract.Presenter mPresenter;

    private DownloadsAdapter adapter;

    public DownloadsFragment() {
        // Required empty public constructor
    }

    public static DownloadsFragment newInstance() {
        DownloadsFragment fragment = new DownloadsFragment();
        Bundle args = new Bundle();
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);

        mContext = context;
        mListener = (HomeFragmentListener) context;
        mediaPlayerInterface = (MediaPlayerInterface) context;
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (getArguments() != null) {
        }

        new DownloadsPresenter(this, new DataRepository(new LocalRepository(Realm.getDefaultInstance()), new RemoteRepository()));
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_downloads, container, false);
        ButterKnife.bind(this, view);
        return view;
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        setTextViewFonts();
        setClickListener();
        setLayoutManager();

        if (adapter == null) {
            mPresenter.getDownloadedPodcasts();
        } else {
            rvDownloads.setAdapter(adapter);
            if (adapter.getItemCount() > 0) {
                txtNoDownloads.setVisibility(View.GONE);
            }
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        TawController.setCurrentPage(TAG);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mPresenter.unsubscribe();
    }

    private void setTextViewFonts() {
        txtDownloadsTitle.setTypeface(FontUtils.getInstance().getDinProBlack());
        txtNoDownloads.setTypeface(FontUtils.getInstance().getDinAlternateBold());
    }

    private void setClickListener() {
        llBack.setOnClickListener(v -> mListener.goBack());
    }

    private void setLayoutManager() {
        rvDownloads.setLayoutManager(new LinearLayoutManager(mContext, RecyclerView.VERTICAL, false));
    }

    @Override
    public void setAdapter(RealmResults<RealmPodcast> podcasts) {
        adapter = new DownloadsAdapter(podcasts, new OnDownloadedPodcastSelectedListener() {
            @Override
            public void onPodcastSelected(Podcast podcast) {
                mListener.setPodcastDetailsFrag(podcast);
            }

            @Override
            public void deletePodcast(RealmPodcast podcast, int position) {
                mPresenter.deletePodcast(podcast, position);
                adapter.remove(position);
                if (adapter.getItemCount() == 0) {
                    txtNoDownloads.setVisibility(View.VISIBLE);
                }


            }
        }, mediaPlayerInterface);


        if (isAdded() && !isDetached()) {
            rvDownloads.setAdapter(adapter);

            if (podcasts.size() == 0) {
                txtNoDownloads.setVisibility(View.VISIBLE);
            } else {
                txtNoDownloads.setVisibility(View.GONE);
            }
        }
    }

    @Override
    public void setPresenter(DownloadsContract.Presenter presenter) {
        if (presenter != null) {
            this.mPresenter = presenter;
        } else {
            throw new RuntimeException("Presenter is null");
        }
    }

    @Override
    public void goBack() {
        mListener.goBack();
    }

    @Override
    public void showLoader(String text) {
        //Not needed
    }

    @Override
    public void hideLoader() {
        //Not needed
    }

    @Override
    public void onUnknownError(String error) {
        //Not needed
    }

    @Override
    public void onTimeout() {
        //Not needed
    }

    @Override
    public void onNetworkError() {
        //Not needed
    }

    @Override
    public void onConnectionError() {
        //Not needed
    }

    @Override
    public void logoutUser() {
        //Not needed
    }

    @Override
    public boolean checkConnection() {
        return ConnectionUtils.userHasConnection(mContext);
    }
}
