package co.pixelbeard.theanfieldwrap.utils;


import org.junit.Before;
import org.junit.Test;

import java.util.ArrayList;
import java.util.List;

import co.pixelbeard.theanfieldwrap.data.Podcast;
import co.pixelbeard.theanfieldwrap.data.Subscription;
import co.pixelbeard.theanfieldwrap.data.Video;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

public class SubscriptionUtilsTest {


    /**
     * User sub levels:
     * 1 = Skeleton
     * 7 = Full Video
     * 8 = Full Audio
     * 9 = Premium (Full audio & full video)
     * <p>
     * Pod sub levels
     * 0 = Free
     * 1 = Skeleton
     * 2 = Full
     * <p>
     * Require User Subs for pod subs
     * Pod sub level 0 = no sub required if premium = false else user sub 1 and above can access
     * Pod sub level 1 = 1, 8 or 9 required
     * Pod sub level 2 = 8 or 9 required
     */

    Podcast mPodcast;
    Video mVideo;


    @Before
    public void setup() {
        mPodcast = new Podcast();
        mVideo = new Video();
    }


    @Test
    public void purchasedPodAccessTest() {
        mPodcast.setPurchased(true);

        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(1), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 8, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(7, 8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getQuadrupleSubs(), mPodcast));
    }


    @Test
    public void podLevel0AccessNotPremiumTest() {
        mPodcast.setSubLevel(0);
        mPodcast.setPremium(false);


        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(1), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 8, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(7, 8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getQuadrupleSubs(), mPodcast));

    }

    @Test
    public void podLevel0AccessPremiumTest() {
        mPodcast.setSubLevel(0);
        mPodcast.setPremium(true);


        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(1), mPodcast));
        assertFalse(SubscriptionUtils.userCanAccessAudio(getSingleSubs(7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(9), mPodcast));

        assertFalse(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 8, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(7, 8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getQuadrupleSubs(), mPodcast));

    }

    @Test
    public void podLevel1AccessTest() {
        mPodcast.setSubLevel(1);
        mPodcast.setPremium(true);

        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(1), mPodcast));
        assertFalse(SubscriptionUtils.userCanAccessAudio(getSingleSubs(7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(9), mPodcast));

        assertFalse(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 8, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(7, 8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getQuadrupleSubs(), mPodcast));
    }

    @Test
    public void podLevel2AccessTest() {
        mPodcast.setSubLevel(2);
        mPodcast.setPremium(true);

        assertFalse(SubscriptionUtils.userCanAccessAudio(getSingleSubs(1), mPodcast));
        assertFalse(SubscriptionUtils.userCanAccessAudio(getSingleSubs(7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getSingleSubs(9), mPodcast));

        assertFalse(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 7), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(1, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getDoubleSubs(8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 8), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 8, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(1, 7, 9), mPodcast));
        assertTrue(SubscriptionUtils.userCanAccessAudio(getTripleSubs(7, 8, 9), mPodcast));

        assertTrue(SubscriptionUtils.userCanAccessAudio(getQuadrupleSubs(), mPodcast));

    }

    @Test
    public void userCanAccessPremiumVideo() {

        assertFalse(SubscriptionUtils.userCanAccessVideo(getSingleSubs(1)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getSingleSubs(7)));
        assertFalse(SubscriptionUtils.userCanAccessVideo(getSingleSubs(8)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getSingleSubs(9)));

        assertTrue(SubscriptionUtils.userCanAccessVideo(getDoubleSubs(1, 7)));
        assertFalse(SubscriptionUtils.userCanAccessVideo(getDoubleSubs(1, 8)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getDoubleSubs(1, 9)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getDoubleSubs(7, 8)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getDoubleSubs(7, 9)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getDoubleSubs(8, 9)));

        assertTrue(SubscriptionUtils.userCanAccessVideo(getTripleSubs(1, 7, 8)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getTripleSubs(1, 8, 9)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getTripleSubs(1, 7, 9)));
        assertTrue(SubscriptionUtils.userCanAccessVideo(getTripleSubs(7, 8, 9)));

        assertTrue(SubscriptionUtils.userCanAccessVideo(getQuadrupleSubs()));
    }


    private List<Subscription> getSingleSubs(int authLevel) {

        List<Subscription> subscriptions = new ArrayList<>();

        subscriptions.add(new Subscription(authLevel, "","",""));


        return subscriptions;
    }


    private List<Subscription> getDoubleSubs(int authLevel1, int authLevel2) {

        List<Subscription> subscriptions = new ArrayList<>();

        subscriptions.add(new Subscription(authLevel1, "","",""));
        subscriptions.add(new Subscription(authLevel2, "","",""));


        return subscriptions;
    }

    private List<Subscription> getTripleSubs(int authLevel1, int authLevel2, int authLevel3) {

        List<Subscription> subscriptions = new ArrayList<>();

        subscriptions.add(new Subscription(authLevel1, "","",""));
        subscriptions.add(new Subscription(authLevel2, "","",""));
        subscriptions.add(new Subscription(authLevel3, "","",""));


        return subscriptions;
    }

    private List<Subscription> getQuadrupleSubs() {

        List<Subscription> subscriptions = new ArrayList<>();

        subscriptions.add(new Subscription(1, "","",""));
        subscriptions.add(new Subscription(7, "","",""));
        subscriptions.add(new Subscription(8, "","",""));
        subscriptions.add(new Subscription(9, "","",""));


        return subscriptions;
    }

}
