//
//  TAWLoginViewController.swift
//  TAW
//
//  Created by Andrew Steven on 17/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Crashlytics
import OneSignal
import TransitionButton

// MARK: - FieldType enum -

public enum FieldType {
    case standard
    case email
    case password
    case code
}

class TAWLoginViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var emailTextField: PBTextField!
    @IBOutlet weak var passwordTextField: PBTextField!
    @IBOutlet weak var loginButton: TransitionButton!
    @IBOutlet weak var forgotPasswordButton: UIButton!
    @IBOutlet weak var skipLoginButton: UIButton!
    
    @IBOutlet weak var loginButtonHeight: NSLayoutConstraint!
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        emailTextField.theTextFieldLabel.text = "EMAIL"
        emailTextField.setType(type: .email, validation_type: .email)
        emailTextField.theTextField.addTarget(self,
                                              action: #selector(textFieldDidChange(textField:)),
                                              for: .editingChanged)
        
        passwordTextField.theTextFieldLabel.text = "PASSWORD"
        passwordTextField.setType(type: .password, validation_type: .password)
        passwordTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)

        self.loginButton.backgroundColor = TAW.Colors.greyButtonColor

        let tapGesture = UITapGestureRecognizer(target: self, action: #selector(dismissKeyboard))
        tapGesture.delegate = self
        self.view.addGestureRecognizer(tapGesture)
    }
    
    override func viewWillAppear(_ animated: Bool) {
        // This is true when being pushed
        print("Is moving to parent: \(self.isMovingToParent)")
        self.animations = self.isMovingToParent ? [.slide(.left, .slightly), .fadeIn] : [.slide(.right, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        self.setupView()
    }
    
    private func setupView() {
        if UIDevice.isSmallDevice {
            self.view.layoutIfNeeded()
            self.loginButtonHeight.constant = 45
            self.view.layoutIfNeeded()
        }
        self.skipLoginButton.isHidden = LocalStorage.shared.loggedIn
    }
    
    // MARK: - UITextField -
    
    @objc func textFieldDidChange(textField: UITextField){
        
        let email = self.emailTextField.theTextField.text
        let password = self.passwordTextField.theTextField.text

        let validEmail = self.isValidField(email ?? "", type: .email)
        let validPassword = self.isValidField(password ?? "", type: .password)
        
        if validEmail && validPassword {
            self.loginButton.backgroundColor = TAW.Colors.redButtonColor
        } else {
            self.loginButton.backgroundColor = TAW.Colors.greyButtonColor
        }
    }
    
    @objc func dismissKeyboard() {
        self.view.endEditing(true)
    }
    
    // MARK: - Validation functions -
    
    func isValidField(_ text: String, type: FieldType) -> Bool {
        
        if text.count == 0 {
            return false
        } else {
            // check validation
            switch type {
            case .email:
                return self.isEmailValid(text)
            case .password, .standard:
                return self.isCharacterLengthValid(text)
            case .code:
                if text.count > 5 {
                    return true
                } else {
                    return false
                }
            }
        }
    }
    
    func isEmailValid(_ text: String) -> Bool {
        let theEmailRegEX = "[A-Z0-9a-z._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,64}"
        let email = NSPredicate(format:"SELF MATCHES %@", theEmailRegEX)
        if email.evaluate(with: text) {
            return true
        } else {
            return false
        }
    }
    
    func isCharacterLengthValid(_ text: String) -> Bool {
        if text.count >= 2 { // minimum character limit
            return true
        } else {
            return false
        }
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        if self.navigationController?.viewControllers.first == self {
            self.dismiss(animated: true, completion: nil)
        }
        else {
            self.navigationController?.popViewController(animated: true)
        }
    }
    
    private func showDeviceLimitModal(_ devices: [Device]) {
        if let tawVC = instantiateVC(identifier: "TAWDeviceLimitViewController") as? TAWDeviceLimitViewController {
            tawVC.deviceWarning = .limitReached
            tawVC.devices = devices
            let nav = TAWNavigationController(rootViewController: tawVC)
            nav.navigationBar.isHidden = true
            nav.modalPresentationStyle = UIModalPresentationStyle.overCurrentContext
            self.navigationController?.present(nav, animated: false, completion: nil)
        }
    }
    
    @IBAction func unwindToLoginView(segue:UIStoryboardSegue) {
        self.login()
    }
    
    // MARK: - API functions -
    
    @IBAction func login() {
        
        self.view.endEditing(true)
        
        let validEmail = self.emailTextField.checkValidation(required: true, type: .email)
        if !validEmail { return }
        
        let validPassword = self.passwordTextField.checkValidation(required: true, type: .password)
        if !validPassword { return }
        
        if !Connectivity.isConnectedToInternet {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        } else {
            self.loginButton.startAnimation()

            var oneSignalId = LocalStorage.shared.oneSignalId
            if oneSignalId == nil {
                if let existingId = OneSignal.getPermissionSubscriptionState().subscriptionStatus.userId {
                    LocalStorage.shared.oneSignalId = existingId
                    oneSignalId = existingId
                } else {
                    oneSignalId = ""
                }
            }
            
            let params = [
                "email": self.emailTextField.theTextField.text!,
                "passwd": self.passwordTextField.theTextField.text!,
                "device_token": oneSignalId!,
                "device_type": "iOS"
            ]
            
            APIClient.loginUser(parameters: params) { (jsonResponse, error) in
                if let userJson = jsonResponse["user"] as? [String: Any] {
                    let user = User(JSON: userJson)
                    user?.authToken = jsonResponse["auth_token"] as? String
                    LocalStorage.shared.user = user
                    LocalStorage.shared.loggedIn = true
                    
                    Crashlytics.sharedInstance().setUserName(user!.username)
                    Crashlytics.sharedInstance().setUserIdentifier("Auth Token: \(user!.authToken!)")
                    Crashlytics.sharedInstance().setUserEmail(user!.guest ? "Guest" : user!.email)
                    
                    if oneSignalId!.count > 0 {
                        self.associateOneSignalWithDevice(oneSignalId!)
                    }

                    if LocalStorage.shared.previousUserId != user?.userId {
                        Utilities().deletePreviousUsersDownloads()
                        Utilities().deleteOtherUsersPodcastPositions()
                    }

                    self.loginButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                        isSubscriptionPopupClose = false
                        let notificationCenter = NotificationCenter.default
                        notificationCenter.post(name: Notification.Name(rawValue: CheckFreeSubscription),
                                                object: self)
                        self.navigationController?.dismiss(animated: true, completion: nil)
                    })
                } else {
                    if let devicesJson = jsonResponse["devices"] as? [[String: Any]] {
                        var devices: [Device] = []
                        for dev in devicesJson {
                            if let device = Device(JSON: dev) {
                                devices.append(device)
                            }
                        }
                        self.loginButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                            self.showDeviceLimitModal(devices)
                        })
                    } else {
                        self.loginButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                            self.showAlertForAPIError(jsonResponse["message"] as! String)
                        })
                    }
                }
            }
        }
    }
    
//    private func checkForPodcastData() {
//        let userId = LocalStorage.shared.user?.userId ?? ""
//        do {
//            let realm = try Realm()
//            let podcastPositions = realm.objects(PodcastPosition.self).filter("userId != %@", userId)
//
//            // Delete any podcast position data that is not the current users
//            try realm.write {
//                realm.delete(podcastPositions)
//            }
//
//        } catch let error {
//            print("Error fetching podcast positions data: \(error.localizedDescription)")
//        }

//    }
    
    @IBAction func skipLogin(_ sender: Any) {
        
        if !Connectivity.isConnectedToInternet {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        } else {
            PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!,
                                                withTitle: "Signing in guest user")
            
            var oneSignalId = LocalStorage.shared.oneSignalId
            if oneSignalId == nil {
                if let existingId = OneSignal.getPermissionSubscriptionState().subscriptionStatus.userId {
                    LocalStorage.shared.oneSignalId = existingId
                    oneSignalId = existingId
                } else {
                    oneSignalId = ""
                }
            }
            
            APIClient.registerGuest { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    let user = User(authToken: jsonResponse["user_api_token"] as! String)
                    LocalStorage.shared.user = user
                    LocalStorage.shared.loggedIn = true
                    
                    Crashlytics.sharedInstance().setUserName(user.username)
                    Crashlytics.sharedInstance().setUserIdentifier("Auth Token: \(user.authToken!)")
                    Crashlytics.sharedInstance().setUserEmail((user.guest) ? "Guest" : user.email)
                    
                    if oneSignalId!.count > 0 {
                        self.associateOneSignalWithDevice(oneSignalId!)
                    }
                    
                    Utilities().deleteOtherUsersPodcastPositions()
                    
                    PBLoadingOverlay.shared.hideOverlay {
                        self.navigationController?.dismiss(animated: true, completion: nil)
                    }
                } else {
                    PBLoadingOverlay.shared.hideOverlay {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    }
                }
            }
        }
    }
    
    private func associateOneSignalWithDevice(_ oneSignalId: String) {
        
        let params = [
            "device_token": oneSignalId
        ]
        
        APIClient.associateDeviceToken(parameters: params) { (jsonResponse, error) in
            print("Associate Device Token Response: \(jsonResponse["message"] as! String)")
        }
    }
}

// MARK: - UIViewController extensions -

extension UIViewController {
    
    func showAlertForAPIError(_ error: String) {
        let alertController = UIAlertController(title: "Error", message: error, preferredStyle: .alert)
        alertController.addAction(UIAlertAction(title: "Ok", style: .default, handler: nil))
        self.navigationController?.present(alertController, animated: true, completion: nil)
    }
    
    func showAlertWithTitle(_ title: String, message: String) {
        let alertController = UIAlertController(title: title, message: message, preferredStyle: .alert)
        alertController.addAction(UIAlertAction(title: "Ok", style: .default, handler: nil))
        self.navigationController?.present(alertController, animated: true, completion: nil)
    }
    
    var isModal: Bool {
        
        let presentingIsModal = presentingViewController != nil
        let presentingIsNavigation = navigationController?.presentingViewController?.presentedViewController == navigationController
        let presentingIsTabBar = tabBarController?.presentingViewController is UITabBarController
        
        return presentingIsModal || presentingIsNavigation || presentingIsTabBar || false
    }
}
