//
//  TAWResetPasswordViewController.swift
//  TAW
//
//  Created by Andrew Steven on 17/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import TransitionButton

class TAWResetPasswordViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -
    
    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var stepLabel: UILabel!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var nextButton: TransitionButton!
    
    @IBOutlet weak var scrollView: UIScrollView!
    @IBOutlet weak var contentView: UIView!
    
    @IBOutlet weak var step1: UIView!
    @IBOutlet weak var emailTextField: PBTextField!
    
    @IBOutlet weak var step2: UIView!
    @IBOutlet weak var authCodeTextField: PBTextField!
    
    @IBOutlet weak var step3: UIView!
    @IBOutlet weak var newPasswordTextField: PBTextField!
    @IBOutlet weak var confirmPasswordTextField: PBTextField!
    
    @IBOutlet weak var nextButtonHeight: NSLayoutConstraint!
    
    // MARK: - Variables -
    
    var currentPage: CGFloat = 0
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()

        emailTextField.theTextFieldLabel.text = "EMAIL"
        emailTextField.setType(type: .email, validation_type: .email)
        emailTextField.theTextField.addTarget(self,
                                              action: #selector(textFieldDidChange(textField:)),
                                              for: .editingChanged)
        
        authCodeTextField.theTextFieldLabel.text = "AUTHORISATION CODE"
        authCodeTextField.setType(type: .code, validation_type: .code)
        authCodeTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)
        
        newPasswordTextField.theTextFieldLabel.text = "NEW PASSWORD"
        newPasswordTextField.setType(type: .password, validation_type: .password)
        newPasswordTextField.theTextField.addTarget(self,
                                                    action: #selector(textFieldDidChange(textField:)),
                                                    for: .editingChanged)
        
        confirmPasswordTextField.theTextFieldLabel.text = "CONFIRM PASSWORD"
        confirmPasswordTextField.setType(type: .password, validation_type: .password)
        confirmPasswordTextField.theTextField.addTarget(self,
                                                        action: #selector(textFieldDidChange(textField:)),
                                                        for: .editingChanged)
        
        self.nextButton.backgroundColor = TAW.Colors.greyButtonColor

        let tapGesture = UITapGestureRecognizer(target: self, action: #selector(dismissKeyboard))
        tapGesture.delegate = self
        self.view.addGestureRecognizer(tapGesture)
    }
    
    override func viewWillAppear(_ animated: Bool) {
        self.animations = [.slide(.left, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        self.setupView()
    }
    
    private func setupView() {
        if UIDevice.isSmallDevice {
            self.view.layoutIfNeeded()
            self.nextButtonHeight.constant = 45
            self.view.layoutIfNeeded()
        }
    }
    
    // MARK: - UITextField -
    
    @objc func textFieldDidChange(textField: UITextField){
        self.invisibleCheck()
    }
    
    @objc func dismissKeyboard() {
        self.view.endEditing(true)
    }
    
    // MARK: - Validation functions -
    
    private func invisibleCheck() {
        
        var isEnabled = false
        if currentPage == 0 {
            let validEmail = self.emailTextField.invisibleCheckValidation(type: .email)
            if validEmail {
                isEnabled = true
            }
        } else if currentPage == 1 {
            let validAuthCode = self.authCodeTextField.invisibleCheckValidation(type: .code)
            if validAuthCode {
                isEnabled = true
            }
        } else {
            let validPassword = self.newPasswordTextField.invisibleCheckValidation(type: .password)
            let validPassword2 = self.confirmPasswordTextField.invisibleCheckValidation(type: .password)
            if validPassword && validPassword2 {
                isEnabled = true
            }
        }
        
        if isEnabled {
            self.nextButton.backgroundColor = TAW.Colors.redButtonColor
        } else {
            self.nextButton.backgroundColor = TAW.Colors.greyButtonColor
        }
    }
    
    private func valid() -> Bool {
        if currentPage == 0 {
            let validEmail = self.emailTextField.checkValidation(required: true, type: .email)
            if !validEmail { return false }
            return true
        } else if currentPage == 1 {
            let validAuthCode = self.authCodeTextField.checkValidation(required: true, type: .code)
            if !validAuthCode { return false }
            return true
        } else {
            let validPassword = self.newPasswordTextField.checkValidation(required: true, type: .password)
            if !validPassword { return false }
            
            let validPassword2 = self.confirmPasswordTextField.checkValidation(required: true, type: .password)
            if !validPassword2 { return false }
            
            if self.newPasswordTextField.theTextField.text != self.confirmPasswordTextField.theTextField.text! {
                self.confirmPasswordTextField.overrideFailValidation(message: "Passwords do not match")
                return false
            }
            return true
        }
    }

    // MARK: - Button actions -
    
    @IBAction func goBack() {
        
        if currentPage == 0 {
            self.navigationController?.popViewController(animated: true)
        } else {
            currentPage -= 1
            scrollToStep()
        }
    }
    
    @IBAction func goNext() {
        
        if valid() {
            if currentPage == 0 {
                self.sendActivationCode()
            } else if currentPage == 2 {
                self.resetPassword()
            } else {
                currentPage += 1
                scrollToStep()
            }
        }
    }
    
    func scrollToStep() {
        
        let pageWidth: CGFloat = scrollView.frame.width
        let slideToX = currentPage * pageWidth
        self.scrollView.scrollRectToVisible(CGRect(x: slideToX,
                                                   y: 0, width: pageWidth,
                                                   height: self.scrollView.frame.height),
                                            animated: true)
        
        self.nextButton.setTitle(self.currentPage < 2 ? "NEXT" : "FINISH", for: .normal)
        self.stepLabel.text = "STEP \(Int(self.currentPage + 1))/3"
        self.invisibleCheck()
    }
    
    // MARK: - API functions -
    
    private func sendActivationCode() {
        
        if !Connectivity.isConnectedToInternet {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection", style: .noInternet)
        } else {
            self.nextButton.startAnimation()

            let params = [
                "email": self.emailTextField.theTextField.text!
            ]
            
            APIClient.recoverPassword(parameters: params) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    self.nextButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                        self.currentPage += 1
                        self.scrollToStep()
                    })
                }
                else {
                    self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    })
                }
            }
        }
    }
    
    private func resetPassword() {
        
        if !Connectivity.isConnectedToInternet {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        } else {
            self.nextButton.startAnimation()
            
            let params = [
                "recovery_code": self.authCodeTextField.theTextField.text!,
                "email": self.emailTextField.theTextField.text!,
                "password": self.newPasswordTextField.theTextField.text!
            ]
            
            APIClient.resetPassword(parameters: params) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    self.nextButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                        self.navigationController?.popViewController(animated: true)
                    })
                }
                else {
                    self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    })
                }
            }
        }
    }
}
