//
//  TAWPodcastDetailViewController.swift
//  TAW
//
//  Created by Andrew Steven on 21/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Alamofire
import RealmSwift
import UICircularProgressRing
import AVFoundation
import MediaPlayer
import Kingfisher
import ImageIO

let podcastPurchasedNotification = "PodcastPurchasedNotification"

class TAWPodcastDetailViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var downloadButton: UIButton!
    @IBOutlet weak var podcastImageView: UIImageView!
    @IBOutlet weak var lockImageView: UIImageView!
    @IBOutlet weak var premiumImageView: UIImageView!
    @IBOutlet weak var premiumImageViewOffsetX: NSLayoutConstraint!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var titleLabelOffsetY: NSLayoutConstraint!
    @IBOutlet weak var dateLabel: UILabel!
    @IBOutlet weak var durationLabel: UILabel!
    @IBOutlet weak var playButton: PBButton!
    
    @IBOutlet weak var downloadProgressRing: UICircularProgressRing!
    @IBOutlet weak var contentViewPaddingBottom: NSLayoutConstraint!
    @IBOutlet weak var unsubscribedView: UIView!
    @IBOutlet weak var unsubscribedViewHeight: NSLayoutConstraint!
    @IBOutlet weak var dynamicContentView: UIView!
    @IBOutlet weak var dynamicContentViewHeight: NSLayoutConstraint!
    @IBOutlet weak var previewView: UIView!
    @IBOutlet weak var tokensButton: UIButton!
    @IBOutlet weak var tokensButtonOffsetX: NSLayoutConstraint!
    @IBOutlet weak var tokensActivityIndicator: UIActivityIndicatorView!
    @IBOutlet weak var buyButton: PBButton!
    @IBOutlet weak var subscribeButton: PBButton!
    
    @IBOutlet weak var signUpButton: PBButton!
    @IBOutlet weak var loginButton: PBButton!
    
    @IBOutlet weak var progressContainer: UIView!
    @IBOutlet weak var progressBar: UIView!
    @IBOutlet weak var progressBarWidth: NSLayoutConstraint!
    
    // MARK: - Variables -
    
    var tokens: Int = 0
    var podcast: Podcast?
    var playerView: PBMiniPlayerView?
    var notificationCenter = NotificationCenter.default
    let downloadService = DownloadService.shared
    var progressTimer = Timer()
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        APIClient.recordAnalytic(.viewedPodcastPost, variable: self.podcast!.postId, secondaryVariable: "1")
        
        self.configureNotifications()
        
        // Prompt user to rate the App once they have listened 2, 5, or 10 podcasts/videos
        PBAppStoreReviewer.checkKeyActivityAndAskForReview()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        self.animations = self.isMovingToParent ? [.slide(.left, .slightly), .fadeIn] : [.fadeIn]
        super.viewWillAppear(animated)
        
        self.setupView()
        self.getWallet()
        self.startTimer()
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        
        self.view.layoutIfNeeded()
        if PBAudioPlayer.shared.podcast != nil {
            if self.playerView == nil {
                let window = UIApplication.shared.keyWindow
                let bottomPadding: CGFloat = window?.safeAreaInsets.bottom ?? 0
                
                let height: CGFloat =
                    self.navigationController?.tabBarController?.tabBar.isHidden ?? true ? 60 + bottomPadding : 60
                self.playerView = PBMiniPlayerView(frame: CGRect(x: 0,
                                                                 y: self.view.frame.size.height - height,
                                                                 width: self.view.frame.size.width,
                                                                 height: height))
                self.playerView!.delegate = self
                self.view.addSubview(self.playerView!)
            }
            self.playerView?.updateProgressView()
        }
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        self.stopTimer()
        if self.playerView != nil {
            self.playerView?.stopTimer()
            self.playerView?.removeFromSuperview()
            self.playerView = nil
        }
    }
    
    private func hideTabBar() {
        self.tabBarController?.tabBar.isHidden = true
    }
    
    private func setPodcastImage() {
        if Connectivity.isConnectedToInternet {
            if let image = self.podcast?.largeImage {
                self.podcastImageView.kf.setImage(with: URL(string: image),
                                                  placeholder: UIImage(named: "placeholder_podcasts"),
                                                  options: [.transition(.fade(0.2))])
            }
        } else {
            if let imageData = self.podcast?.imageData {
                if let image = UIImage(data: imageData) {
                    self.podcastImageView.image = image
                }
            }
        }
    }
    
    private func setupPremiumImageView() {
        guard let podcast = self.podcast else {
            return
        }

        DispatchQueue.main.async {
            self.lockImageView.isHidden = !podcast.isLocked
            self.premiumImageView.image = podcast.premiumImage
            self.premiumImageView.isHidden = !podcast.premium

            self.premiumImageViewOffsetX.constant = self.lockImageView.isHidden ? 27 : 52
            self.titleLabelOffsetY.constant = self.premiumImageView.isHidden ? 30 : 60
        }
    }
    
    private func setupView() {
        guard let podcast = self.podcast else {
            return
        }
        
        if podcast.isInvalidated {
            self.goBack()
            return
        }
        
        self.hideTabBar()
        self.setPodcastImage()
        self.setupPremiumImageView()
        
        let guest = LocalStorage.shared.user?.guest ?? true
        let locked = podcast.isLocked
        
        if !locked {
            self.buyButton.isHidden = true
//            self.subscribeButton.isHidden = true
        }
        else {
            self.buyButton.isHidden = guest
//            self.subscribeButton.isHidden = guest
        }
        
        self.unsubscribedViewHeight.constant = locked ? 100: 0
        self.tokensButtonOffsetX.constant = locked ? 20 : 70
        self.downloadButton.isHidden = locked
        self.tokensButton.isHidden = guest
        
        self.signUpButton.isHidden = !guest
        self.loginButton.isHidden = !guest
        
        self.previewView.isHidden = !locked
        self.tokensButton.titleLabel?.numberOfLines = 0
        
        self.buyButton.setTitle(self.podcast!.formattedCost, for: .normal)
        self.titleLabel.text = podcast.title
        
        self.dateLabel.text = podcast.formattedDate
        self.durationLabel.text = podcast.duration
        
        if podcast.isDownloaded {
            self.downloadButton.isHidden = true
            self.tokensButtonOffsetX.constant = 20
        }
        
        self.downloadProgressRing.font = TAW.Fonts.dinAlternate10!
        self.downloadProgressRing.ringStyle = .ontop
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 91 : 31
        
        if guest && !locked {
            // Need to show preview for guest users even if content is free
            self.previewView.isHidden = false
            self.buyButton.isHidden = true
//            self.subscribeButton.isHidden = true
            self.unsubscribedViewHeight.constant = 100
            self.downloadButton.isHidden = true
            self.tokensButton.isHidden = true
            
            self.signUpButton.isHidden = false
            self.loginButton.isHidden = false
        }
        
        if guest || locked {
            self.progressContainer.isHidden = true
        } else {
            self.progressContainer.isHidden = false
        }
        
        self.view.layoutIfNeeded()
        self.checkActiveDownloads()
        self.displayContent()
        
        self.updatePlayButtonState()
    }
    
    func startTimer() {
        self.progressTimer = Timer.scheduledTimer(timeInterval: 1.0, target: self, selector: #selector(updateProgressBar), userInfo: nil, repeats: true)
    }
    
    func stopTimer() {
        self.progressTimer.invalidate()
    }
    
    @objc func updateProgressBar() {
        
        guard let podcast = self.podcast else {
            return
        }
        
        if podcast.isInvalidated {
            self.goBack()
            return
        }
        
        if podcast.title == PBAudioPlayer.shared.podcast?.title {
            
            var elapsed = PBAudioPlayer.shared.elapsedTime
            var total = PBAudioPlayer.shared.totalDuration
            var width: CGFloat = 0
            
            if elapsed.isNaN {
                elapsed = 0
            }
            
            if total.isNaN {
                total = 0
            }
            
            if elapsed > 0 && total > 0 {
                width = CGFloat(elapsed/total) * self.progressContainer.frame.width
            }
            
            self.progressContainer.layoutIfNeeded()
            self.progressBarWidth.constant = width
            UIView.animate(withDuration: 0.5) {
                self.progressContainer.layoutIfNeeded()
            }
            
        } else {
            
            let realm = try! Realm()
            let podcastPosition = realm.objects(PodcastPosition.self).filter("postId == %@", podcast.postId)
            var elapsed = podcastPosition.first?.position ?? 0.0
            
            if let podcastMinutes = Double(podcast.podcastLength) {
                var total = podcastMinutes * 60
                var width: CGFloat = 0
                
                if elapsed.isNaN {
                    elapsed = 0
                }
                
                if total.isNaN {
                    total = 0
                }
                
                if elapsed > 0 && total > 0 {
                    width = CGFloat(elapsed/total) * self.progressContainer.frame.width
                }
                
                self.progressContainer.layoutIfNeeded()
                self.progressBarWidth.constant = width
                UIView.animate(withDuration: 0.3) {
                    self.progressContainer.layoutIfNeeded()
                }
            }
        }
    }
    
    private func checkActiveDownloads() {
        if let download = downloadService.activeDownloads[self.podcast!.safeURL] {
            self.downloadButton.isHidden = true
            self.tokensButtonOffsetX.constant = 70
            let value: CGFloat = CGFloat(download.progress)
            self.downloadProgressRing.startProgress(to: value, duration: 0)
            self.downloadProgressRing.isHidden = false
        }
    }
    
    private func displayContent() {
        
        let contents = Utilities().parseContent(self.podcast!.content)
        
        var yAxis: CGFloat = 0
        let width: CGFloat = UIScreen.main.bounds.width - (2*29)
        
        for component in contents {
            
            if component["type"] as? String ?? "" == "string" {
                let text = component["value"] as? String ?? ""
                let height: CGFloat = text.height(withConstrainedWidth: width,
                                                  font: TAW.Fonts.arialRegular16!,
                                                  lineSpacing: 1,
                                                  lineHeightMultiple: 1.2)
                
                let label = UILabel(frame: CGRect(x: 29, y: yAxis, width: width, height: height))
                label.text = text
                label.numberOfLines = 0
                label.font = TAW.Fonts.arialRegular16
                label.textColor = UIColor.black
                label.setLineSpacing(lineSpacing: 1, lineHeightMultiple: 1.2)
                self.dynamicContentView.addSubview(label)
                yAxis += height + 5
            } else {
                if let imageURL = component["value"] as? URL {
                    if let imageSource = CGImageSourceCreateWithURL(imageURL as CFURL, nil) {
                        if let imageProperties = CGImageSourceCopyPropertiesAtIndex(imageSource, 0, nil) as Dictionary? {
                            let pixelWidth = imageProperties[kCGImagePropertyPixelWidth] as! CGFloat
                            let pixelHeight = imageProperties[kCGImagePropertyPixelHeight] as! CGFloat
                            
                            let newHeight: CGFloat = Utilities().calculateHeightBasedOn(width: pixelWidth,
                                                                                        height: pixelHeight)
                            
                            let imageView = UIImageView(frame: CGRect(x: 0,
                                                                      y: yAxis,
                                                                      width: UIScreen.main.bounds.width,
                                                                      height: newHeight))
                            
                            imageView.kf.setImage(with: imageURL,
                                                  placeholder: UIImage(named: "placeholder_podcasts"),
                                                  options: [.transition(.fade(0.2))])
                            
                            imageView.contentMode = .scaleAspectFill
                            self.dynamicContentView.addSubview(imageView)
                            yAxis += imageView.frame.size.height + 5
                        }
                    }
                }
            }
        }

        self.view.layoutIfNeeded()
        self.dynamicContentViewHeight.constant = yAxis
        self.view.layoutIfNeeded()
    }
    
    @objc public func updatedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            
            if download.podcast == self.podcast {
                
                let value: CGFloat = CGFloat(download.progress)
                DispatchQueue.main.async {
                    self.downloadProgressRing.startProgress(to: value, duration: 0)
                }
            }
        }
    }
    
    @objc public func completedDownload(notification: NSNotification) {
        
        guard let podcast = self.podcast else {
            return
        }
        
        if let download = notification.userInfo?["download"] as? Download {
            if download.podcast.title == podcast.title {
                DispatchQueue.main.async {
                
                    self.downloadProgressRing.isHidden = true
                    self.tokensButtonOffsetX.constant = 20
                    UIView.animate(withDuration: 0.3, animations: {
                        self.view.layoutIfNeeded()
                    })
                }
            }
        }
    }
    
    @objc public func failedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            if download.podcast.title == self.podcast?.title {
                DispatchQueue.main.async {
                    self.downloadButton.isHidden = false
                    self.downloadProgressRing.isHidden = true
                    self.downloadProgressRing.startProgress(to: 0, duration: 0)
                    
                    self.showAlertWithTitle("Download Failed", message: "There was an error trying to download \(download.podcast.title). Try again later")
                }
            }
        }
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self,
                                            selector: #selector(podcastStateChanged),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged),
                                            object: nil)
        self.notificationCenter.addObserver(self,
                                            selector: #selector(podcastStateChanged),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged),
                                            object: nil)
        self.notificationCenter.addObserver(self,
                                            selector: #selector(updatedDownload(notification:)),
                                            name: NSNotification.Name(rawValue: downloadServiceUpdatedNotification),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(completedDownload(notification:)),
                                            name: NSNotification.Name(rawValue: downloadServiceCompletedNotification),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(displayPurchasedState),
                                            name: NSNotification.Name(rawValue: podcastPurchasedNotification),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(failedDownload(notification:)),
                                            name: NSNotification.Name(rawValue: downloadServiceFailedNotification),
                                            object: nil)
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    @objc func podcastStateChanged() {
        self.updatePlayButtonState()
    }
    
    func updatePlayButtonState() {
        
        if PBAudioPlayer.shared.podcast?.title == self.podcast?.title {
            if PBAudioPlayer.shared.isPlaying {
                playButton.setImage(UIImage(named: "pause_button"), for: .normal)
            } else {
                playButton.setImage(UIImage(named: "play_button"), for: .normal)
            }
        } else if PBAudioPlayer.shared.podcast == nil {
            playButton.setImage(UIImage(named: "play_button"), for: .normal)
            
            if self.playerView != nil {
                self.playerView?.stopTimer()
                self.playerView?.removeFromSuperview()
                self.playerView = nil
            }
        }
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 91 : 31
        self.view.layoutIfNeeded()
    }
    
    // MARK: - API functions -
    
    private func getWallet() {
        self.tokensButton.setTitle("Your Balance\n ", for: .normal)
        self.tokensActivityIndicator.startAnimating()

        if Connectivity.isConnectedToInternet {
            APIClient.getWallet { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    
                    if let wallet = jsonResponse["wallet"] as? [String: Any] {
                        if let tokens = wallet["balance"] as? Int {
                            self.tokens = tokens
                            self.tokensActivityIndicator.stopAnimating()
                            self.tokensButton.setTitle("Your Balance\n\(tokens) Tokens", for: .normal)
                        }
                    } else {
                        print("Error retrieving wallet")
                        self.tokensActivityIndicator.stopAnimating()
                        self.tokensButton.setTitle("Balance\nunavailable", for: .normal)
                    }
                } else {
                    self.tokensActivityIndicator.stopAnimating()
                    self.tokensButton.setTitle("Balance\nunavailable", for: .normal)
                    self.showAlertForAPIError(jsonResponse["message"] as? String ?? "")
                }
            }
        } else {
            self.tokensActivityIndicator.stopAnimating()
            self.tokensButton.setTitle("Balance\nunavailable", for: .normal)
        }
    }
    
    // MARK: - Button actions -
    
    @IBAction func unwindToPodcastDetails(segue: UIStoryboardSegue) {
        self.createAccount()
    }
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    @IBAction func download(_ sender: Any) {
        
        self.downloadButton.isHidden = true
        self.downloadProgressRing.startProgress(to: 0, duration: 0)
        self.downloadProgressRing.isHidden = false
        downloadService.startDownload(self.podcast!)
    }
    
    @IBAction func play(_ sender: Any) {
        
        guard let podcast = self.podcast else {
            return
        }
        
        // Check if user has access to premium audio or user has purchased this podcast
        if !podcast.isLocked {
            if PBAudioPlayer.shared.podcast?.title == podcast.title {
                if PBAudioPlayer.shared.isPlaying {
                    PBAudioPlayer.shared.pause()
                } else {
                    PBAudioPlayer.shared.play()
                }
            } else {
                if let tawVC = instantiateVC(identifier: "podcastPlayerVC") as? TAWPodcastPlayerViewController {
                    tawVC.newPodcast = podcast
                    self.navigationController?.present(tawVC, animated: true, completion: nil)
                }
            }
        } else {
            // User does not have access to this podcast, show 2 minute preview
            if let tawVC = instantiateVC(identifier: "podcastPlayerVC") as? TAWPodcastPlayerViewController {
                tawVC.newPodcast = podcast
                tawVC.isPreview = true
                let nav = TAWNavigationController(rootViewController: tawVC)
                nav.navigationBar.isHidden = true
                nav.modalPresentationStyle = UIModalPresentationStyle.overCurrentContext
                self.navigationController?.present(nav, animated: true, completion: nil)
            }
        }
    }
    
    @IBAction func buyEpisode(_ sender: Any) {
        guard let podcast = self.podcast else {
            return
        }
        
        if self.tokens < podcast.cost {
            self.showLowBalanceModal()
        } else {
            self.showPurchaseModal()
        }
    }
    
    private func showLowBalanceModal() {
        if let tawVC = instantiateVC(identifier: "alertVC") as? TAWAlertViewController {
            tawVC.set(title: "NOT ENOUGH TOKENS",
                      description: "Your token balance is too low to make this purchase. Buy tokens?",
                      type: .balance)
            tawVC.delegate = self
            self.navigationController?.present(tawVC, animated: false, completion: nil)
        }
    }
    
    private func showPurchaseModal() {
        guard let podcast = self.podcast else {
            return
        }
        
        if let tawVC = instantiateVC(identifier: "alertVC") as? TAWAlertViewController {
            tawVC.set(title: "ARE YOU SURE?",
                      description: "Are you sure you want to purchase \(podcast.title)",
                type: .purchase)
            tawVC.delegate = self
            self.navigationController?.present(tawVC, animated: false, completion: nil)
        }
    }
    
    private func makePurchase() {
        
        guard let podcast = self.podcast else {
            return
        }
        
        PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!, withTitle: "Purchasing podcast")
        
        let params = [
            "product_id": podcast.postId
        ]
        
        APIClient.purchaseItem(parameters: params) { (jsonResponse, _) in
            
            if jsonResponse["success"] as? Bool == true {
                print("Response: \(jsonResponse)")
                
                if let wallet = jsonResponse["wallet"] as? [String: Any] {
                    if let tokens = wallet["balance"] as? Int {
                        self.tokens = tokens
                        self.tokensButton.setTitle("Your Balance\n\(tokens) Tokens", for: .normal)
                    } else {
                        self.tokensButton.setTitle("Balance\nunavailable", for: .normal)
                    }
                }
                
                self.displayPurchasedState()
                PBLoadingOverlay.shared.hideOverlay()
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    self.showAlertWithTitle("Error",
                                            message: jsonResponse["message"] as? String ?? "")
                }
            }
        }
    }
    
    @objc private func displayPurchasedState() {
        
        if let podcast = self.podcast {
            do {
                // Get the default Realm
                let realm = try Realm()
                
                // Update the podcast object in realm
                try realm.write {
                    podcast.purchased = true
                }
            }
            catch {
                // Unable to update realm object
            }
        }

        self.progressContainer.isHidden = false
        self.downloadButton.isHidden = false
        self.tokensButtonOffsetX.constant = 70
        self.signUpButton.isHidden = true
        self.loginButton.isHidden = true
        self.playButton.isHidden = false
        self.buyButton.isHidden = true
        self.previewView.isHidden = true
        self.lockImageView.isHidden = true
        
        self.premiumImageView.image = UIImage(named: "purchased_icon")
        self.premiumImageViewOffsetX.constant = 27
        self.unsubscribedViewHeight.constant = 0
        self.view.layoutIfNeeded()
    }
    
    @IBAction func howToSubscribe(_ sender: Any) {
        isSubscriptionPopupClose = false
        self.showHowToSubscribe(alertDelegate: self)
    }
    
    @IBAction func login(_ sender: Any) {
        if let viewController = self.instantiateVC(identifier: "loginVC") as? TAWLoginViewController {
            let nav = TAWNavigationController(rootViewController: viewController)
            nav.navigationBar.isHidden = true
            self.navigationController?.present(nav, animated: true, completion: nil)
        }
    }
    
    @IBAction func createAccount() {
        self.showCreateAccount()
    }
    
    @IBAction func showWallet() {
        if Connectivity.isConnectedToInternet {
            self.showUserWallet()
        } else {
            let alert = UIAlertController(title: "Error", message: "Your wallet is not available whilst offline. Please try again when your device is back online", preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "Ok", style: .cancel, handler: nil))
            self.navigationController?.present(alert, animated: true, completion: nil)
        }
    }
}

// MARK: - PBMiniPlayerViewDelegate -

extension TAWPodcastDetailViewController: PBMiniPlayerViewDelegate {

    func openPlayer() {
        self.showPodcastPlayer()
    }
}

extension TAWPodcastDetailViewController: TAWAlertViewControllerDelegate {
    
    func didSelectOption(_ option: TAWAlertViewOption, type: TAWAlertViewType) {
        switch type {
        case .purchase:
            if option == .accept {
                self.makePurchase()
            }
        case .balance:
            if option == .accept {
                self.showPurchaseTokens()
            }
        }
    }
}
extension TAWPodcastDetailViewController: TAWAlertViewDelegate{
    func freeSubsExpired()
    {
       // showHowToSubscribe()
        self.showSubscriptionView(isFreeSubExpired: true)
    }

    func acceptedNotifications() {

    }

    func unlinkAccount() {

    }

    func accountCreated() {

    }

    func didTapLogout() {

    }

}

