//
//  TAWUnlockViewController.swift
//  TAW
//
//  Created by Andrew Steven on 11/02/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import UIKit
import RealmSwift

class TAWUnlockViewController: UIViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var containerView: UIView!
    @IBOutlet weak var buyButton: PBButton!
    @IBOutlet weak var buyButtonHeight: NSLayoutConstraint!
    
    // MARK: - Variables -
    
    var podcast: Podcast?
    var video: Video?
    var tokens: Int = 0
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        self.animateIn()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        self.getWallet()
        self.setupView()
    }
    
    private func setupView() {
        
        if UIDevice.isSmallDevice {
            self.view.layoutIfNeeded()
            self.buyButtonHeight.constant = 45
            self.view.layoutIfNeeded()
        }
        
        if LocalStorage.shared.user!.guest {
            self.buyButton.setImage(nil, for: .normal)
            self.buyButton.setTitle("CREATE FULL ACCOUNT", for: .normal)
        }
        else {
            self.buyButton.setImage(UIImage(named: "small_token_icon_2"), for: .normal)
            if let podcast = self.podcast {
                self.buyButton.setTitle(podcast.formattedCost, for: .normal)
            }
            
            if let video = self.video {
                self.buyButton.setTitle(video.formattedCost, for: .normal)
            }
        }
    }
    
    private func animateIn() {
        UIView.animate(withDuration: 0.3) {
            self.containerView.alpha = 1
        }
    }
    
    private func animateOut() {
        UIView.animate(withDuration: 0.3, animations: {
            self.containerView.alpha = 0
        }) { (_) in
            self.navigationController?.dismiss(animated: false, completion: nil)
        }
    }
    
    // MARK: - API functions -
    
    private func getWallet() {
        
        if Connectivity.isConnectedToInternet {
            APIClient.getWallet { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    
                    if let wallet = jsonResponse["wallet"] as? [String: Any] {
                        if let tokens = wallet["balance"] as? Int {
                            self.tokens = tokens
                        }
                    } else {
                        print("Error retrieving wallet")
                    }
                } else {
                    self.showAlertForAPIError(jsonResponse["message"] as? String ?? "")
                }
            }
        }
        
    }
    
    private func makePurchase() {
        
        var productId = "", message = ""
        if let podcast = self.podcast {
            productId = podcast.postId
            message = "Purchasing podcast"
        }
        else if let video = self.video {
            productId = video.postId
            message = "Purchasing video"
        }
        else {
            return
        }
        
        PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!, withTitle: message)
        
        let params = [
            "product_id": productId
        ]
        
        APIClient.purchaseItem(parameters: params) { (jsonResponse, _) in
            
            if jsonResponse["success"] as? Bool == true {
                print("Response: \(jsonResponse)")
                
                if let wallet = jsonResponse["wallet"] as? [String: Any] {
                    if let tokens = wallet["balance"] as? Int {
                        self.tokens = tokens
                    }
                }
                
                if let podcast = self.podcast {
                    
                    do {
                        // Get the default Realm
                        let realm = try Realm()
                        
                        // Update the podcast object in realm
                        try realm.write {
                            podcast.purchased = true
                        }
                    }
                    catch {
                        // Unable to update realm object
                    }
                    
                    NotificationCenter.default.post(name: Notification.Name(rawValue: podcastPurchasedNotification),
                                                 object: self)
                }
                else if let video = self.video {
                    video.purchased = true
                    NotificationCenter.default.post(name: Notification.Name(rawValue: videoPurchasedNotification),
                                                    object: self)
                }
                
                PBLoadingOverlay.shared.hideOverlay {
                    self.close()
                }
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    self.showAlertWithTitle("Error",
                                            message: jsonResponse["message"] as? String ?? "")
                }
            }
        }
    }
    
    // MARK: - Button actions -
    
    @IBAction func close() {
        self.animateOut()
    }

    @IBAction func howToSubscribe() {
        isSubscriptionPopupClose = false
        self.showHowToSubscribe(alertDelegate: self)
    }
    
    
    
    @IBAction func buyEpisode() {
        if LocalStorage.shared.user!.guest {
            // Unwind to Podcast Details View and show Complete Account
            self.performSegue(withIdentifier: "unwindToPodcastDetails", sender: nil)
        } else {
            // Buy Podcast or Video
            
            if let podcast = self.podcast {
                if self.tokens < podcast.cost {
                    self.showLowBalanceModal()
                } else {
                    self.showPurchaseModal()
                }
            }
            else if let video = self.video {
                if self.tokens < video.cost {
                    self.showLowBalanceModal()
                } else {
                    self.showPurchaseModal()
                }
            }
        }
    }
    
    private func createAccount() {
        self.showCreateAccount()
    }
    
    private func showLowBalanceModal() {
        if let tawVC = instantiateVC(identifier: "alertVC") as? TAWAlertViewController {
            tawVC.set(title: "NOT ENOUGH TOKENS",
                      description: "Your token balance is too low to make this purchase. Buy tokens?",
                      type: .balance)
            tawVC.delegate = self
            self.navigationController?.present(tawVC, animated: false, completion: nil)
        }
    }
    
    private func showPurchaseModal() {
        var title = ""
        if let podcast = self.podcast {
            title = podcast.title
        } else if let video = self.video {
            title = video.title
        } else {
            return
        }
        
        if let tawVC = instantiateVC(identifier: "alertVC") as? TAWAlertViewController {
            tawVC.set(title: "ARE YOU SURE?",
                      description: "Are you sure you want to purchase \(title)",
                type: .purchase)
            tawVC.delegate = self
            self.navigationController?.present(tawVC, animated: false, completion: nil)
        }
    }
}

extension TAWUnlockViewController: TAWAlertViewControllerDelegate {
    func didSelectOption(_ option: TAWAlertViewOption, type: TAWAlertViewType) {
        switch type {
        case .purchase:
            if option == .accept {
                self.makePurchase()
            }
        case .balance:
            if option == .accept {
                self.showPurchaseTokens()
            }
        }
    }
}
extension TAWUnlockViewController: TAWAlertViewDelegate{
    func freeSubsExpired()
    {
       // showHowToSubscribe()
        self.showSubscriptionView(isFreeSubExpired: true)
    }

    func acceptedNotifications() {

    }

    func unlinkAccount() {

    }

    func accountCreated() {

    }

    func didTapLogout() {

    }

}

