//
//  TAWDownloadsViewController.swift
//  TAW
//
//  Created by Andrew Steven on 17/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Spruce
import RealmSwift

class TAWDownloadsViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var tableView: UITableView!
    @IBOutlet weak var noDownloadsLabel: UILabel!
    
    // MARK: - Variables -
    
    var downloads: [Podcast] = []
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        self.tableView.tableFooterView = UIView()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        self.animations = [.slide(.left, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        getDownloads()
    }
    
    // MARK: - Data fetching -
    
    private func getDownloads() {
        
        self.downloads = []
        self.tableView.reloadData()
        self.tableView.spruce.prepare(with: self.tableViewAnimations)
        
        // Fetch downloaded podcasts
        let realm = try! Realm()
        let currentDownloads = realm.objects(Podcast.self).sorted(byKeyPath: "createdDate")
        let temp = Array(currentDownloads)
        self.downloads = temp.filter {$0.isDownloaded == true}
        
        if self.downloads.count == 0 {
            self.noDownloadsLabel.isHidden = false
        } else {
            self.noDownloadsLabel.isHidden = true
            self.tableView.reloadData()
            self.animateTableView(self.tableView)
        }
    }
    
    // MARK: - Button actions -

    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    private func showDeleteAlert(_ podcast: Podcast) {
        let alert = UIAlertController(title: "Are you sure?",
                                      message: "Are you sure you want to remove \(podcast.title) from your Downloads?",
                                      preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "NO", style: .cancel, handler: nil))
        alert.addAction(UIAlertAction(title: "YES", style: .destructive, handler: { (_) in
            self.remove(podcast)
        }))
        self.navigationController?.present(alert, animated: true, completion: nil)
    }
    
    private func remove(_ podcast: Podcast) {
        do {
            
            // Remove stored file from local directory
            let fileUrl: URL? = podcast.getSaveFileUrl()
            if fileUrl != nil {
                if FileManager.default.fileExists(atPath: fileUrl!.path) {
                    try FileManager.default.removeItem(atPath: fileUrl!.path)
                }
            }
            
            // Delete row from tableView
            if let index: Int = self.downloads.index(of: podcast) {
                self.downloads.remove(at: index)
                self.tableView.deleteRows(at: [IndexPath(row: index, section: 0)],
                                          with: .automatic)
            }
        }
        catch {
            // Failed to delete downloaded podcast
        }
    }
    
    // MARK: - Convenience
    
    private func notifyOnDeletion() {
        NotificationCenter.default.post(name: Notification.Name(rawValue: TAWDownloadRemovedNotification),
                                        object: self)
    }
}

// MARK: - UITableViewDelegate & UITableViewDataSource -

extension TAWDownloadsViewController: UITableViewDelegate, UITableViewDataSource {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return downloads.count
    }
    
    func tableView(_ tableView: UITableView, willDisplay cell: UITableViewCell, forRowAt indexPath: IndexPath) {
        
        cell.spruce.prepare(with: self.tableViewAnimations)
        var animation = SpringAnimation(duration: 0.7)
        animation.animationOptions = [UIView.AnimationOptions.allowUserInteraction]
        
        cell.spruce.animate(self.tableViewAnimations,
                            animationType: animation,
                            sortFunction: self.tableViewSortFunction!)
        cell.layoutSubviews()
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let podcast = self.downloads[indexPath.row]
        let cell: TAWPodcastTableViewCell =
            tableView.dequeueReusableCell(withIdentifier: "downloadsCell", for: indexPath) as! TAWPodcastTableViewCell
        cell.set(podcast: podcast, imageQuality: .medium, displayMode: .download, delegate: self, download: nil)
        return cell
    }
    
    func tableView(_ tableView: UITableView, canEditRowAt indexPath: IndexPath) -> Bool {
        if let current = PBAudioPlayer.shared.podcast {
            if current.isInvalidated {
                return false
            }
        }
        
        let download = self.downloads[indexPath.row]
        if download.isInvalidated {
            return false
        }
        
        if PBAudioPlayer.shared.podcast?.title == self.downloads[indexPath.row].title {
            return false
        }
        return true
    }
    
    func tableView(_ tableView: UITableView,
                   commit editingStyle: UITableViewCell.EditingStyle,
                   forRowAt indexPath: IndexPath) {
        
        if editingStyle == .delete {
            let podcast = self.downloads[indexPath.row]
            self.showDeleteAlert(podcast)
        }
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return 114
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        let podcast = self.downloads[indexPath.row]
        if let tawVC = instantiateVC(identifier: "podcastDetailVC") as? TAWPodcastDetailViewController {
            tawVC.podcast = podcast
            self.navigationController?.pushViewController(tawVC, animated: true)
        }
    }
}

extension TAWDownloadsViewController: PodcastCellDelegate {
    func downloadTapped(_ podcast: Podcast) {
        // Do nothing
    }

    func downloadFailed(_ podcast: Podcast) {
        // Do nothing
    }

    func podcastInvalidated() {
        self.getDownloads()
    }
    
    func didTapDeleteButton(_ podcast: Podcast) {
        self.showDeleteAlert(podcast)
    }
}
