//
//  TAWTokensViewController.swift
//  TAW
//
//  Created by Andrew Steven on 25/02/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import UIKit
import StoreKit
import Crashlytics

class TAWTokensViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -
    
    @IBOutlet weak var collectionView: UICollectionView!
    @IBOutlet weak var pageControl: UIPageControl!
    
    // MARK: - Variables -
    
    var cellWidth: CGFloat = 0
    var cellHeight: CGFloat = 0
    var products: [SKProduct] = []
    var notificationCenter = NotificationCenter.default
    
    // MARK: - Lifecycle -

    override func viewDidLoad() {
        super.viewDidLoad()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        self.animations = [.slide(.left, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        
        // Ratio 300 : 500 based on designs
        cellHeight = UIDevice.isSmallDevice ? UIScreen.main.bounds.height - 180 : UIScreen.main.bounds.height - 280
        cellWidth = (cellHeight / 500) * 300
        
        let marginSpacing = (UIScreen.main.bounds.width - cellWidth) / 2
        
        if let flowLayout = collectionView.collectionViewLayout as? UICollectionViewFlowLayout {
            flowLayout.minimumLineSpacing = 0
            flowLayout.minimumInteritemSpacing = 0
            flowLayout.sectionInset = UIEdgeInsets(top: 0, left: marginSpacing, bottom: 0, right: marginSpacing)
            flowLayout.itemSize = CGSize(width: cellWidth, height: cellHeight)
        }

        self.configureNotifications()
        self.getProducts()
    }

    //MARK: - Notifications

    func configureNotifications() {
        self.notificationCenter.addObserver(self, selector: #selector(getProducts), name: NSNotification.Name(rawValue: PBIAPTiersFetched), object: nil)
    }

    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    // MARK: - Data fetching -
    
    @objc private func getProducts() {
        
        self.products = IAPClient.shared.products.filter({$0.productIdentifier == "taw_iap_250_tokens" || $0.productIdentifier == "taw_iap_500_tokens"})
        print(self.products)
        self.pageControl.numberOfPages = self.products.count
        self.pageControl.currentPage = 0
        self.collectionView.reloadData()
    }
    
    func scrollViewWillBeginDragging(_ scrollView: UIScrollView) {
        indexOfCellBeforeDragging = indexOfMajorCell()
    }
    
    private var indexOfCellBeforeDragging = 0
    func scrollViewWillEndDragging(_ scrollView: UIScrollView,
                                   withVelocity velocity: CGPoint,
                                   targetContentOffset: UnsafeMutablePointer<CGPoint>) {
        
        targetContentOffset.pointee = scrollView.contentOffset
        var indexOfMajorCell = self.indexOfMajorCell()
        
        let swipeVelocityThreshold: CGFloat = 0.5
        let hasEnoughVelocityForNextCell =
            indexOfCellBeforeDragging + 1 < self.products.count &&
                velocity.x > swipeVelocityThreshold
        let hasEnoughVelocityForPreviousCell =
            indexOfCellBeforeDragging - 1 >= 0 &&
                velocity.x < -swipeVelocityThreshold
        let majorCellIsTheCellBeforeDragging: Bool = indexOfMajorCell == indexOfCellBeforeDragging
        let didUseSwipeToSkipCell =
            majorCellIsTheCellBeforeDragging &&
                (hasEnoughVelocityForNextCell || hasEnoughVelocityForPreviousCell)
        
        if didUseSwipeToSkipCell {
            
            let snapToIndex = indexOfCellBeforeDragging + (hasEnoughVelocityForNextCell ? 1 : -1)
            let toValue = cellWidth * CGFloat(snapToIndex)
            
            UIView.animate(withDuration: 0.3,
                           delay: 0,
                           usingSpringWithDamping: 1,
                           initialSpringVelocity: velocity.x,
                           options: .allowUserInteraction,
                           animations: {
                scrollView.contentOffset = CGPoint(x: toValue, y: 0)
                scrollView.layoutIfNeeded()
            }, completion: nil)
        } else {
            Crashlytics.sharedInstance().setObjectValue(indexOfMajorCell, forKey: "indexOfMajorCell")
            if indexOfMajorCell >= self.products.count {
                indexOfMajorCell = self.products.count - 1
            }
            if indexOfMajorCell < 0 {
                indexOfMajorCell = 0
            }
            let indexPath = IndexPath(item: indexOfMajorCell, section: 0)
            collectionView.scrollToItem(at: indexPath, at: .centeredHorizontally, animated: true)
        }
    }
    
    func scaleCells() {
        for cell in collectionView.visibleCells {
            cell.layer.transform = CATransform3DScale(CATransform3DIdentity, 0.9, 0.9, 1)
        }
        let indexOfMajorCell = self.indexOfMajorCell()
        self.pageControl.currentPage = indexOfMajorCell
        let cell = collectionView.cellForItem(at: IndexPath(item: indexOfMajorCell, section: 0))
        cell?.layer.transform = CATransform3DScale(CATransform3DIdentity, 1, 1, 1)
    }
    
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        scaleCells()
    }
    
    private func indexOfMajorCell() -> Int {
        let proportionalOffset = collectionView.contentOffset.x / cellWidth
        return Int(round(proportionalOffset))
    }
    
    func didPurchaseProduct(_ product: SKProduct, transactionId: String) {
        
        PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!, withTitle: "Making purchase")
        
        let tokenValue = Utilities().tokenValueForProduct(item: product)
        
        let params: [String: Any] = [
            "value": Int(tokenValue)!,
            "title": "IAP \(tokenValue) Tokens",
            "entry_variable": transactionId
        ]
        
        APIClient.creditAccount(parameters: params) { (jsonResponse, _) in
            if jsonResponse["success"] as? Bool == true {
                PBLoadingOverlay.shared.hideOverlay()
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    let message = jsonResponse["message"] as? String ?? ""
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    } else {
                        self.showAlertForAPIError(jsonResponse["message"] as? String ?? "")
                    }
                }
            }
        }
    }
}

// MARK: - UICollectionViewDelegate & UICollectionViewDelegateFlowLayout -

extension TAWTokensViewController: UICollectionViewDelegate, UICollectionViewDataSource, UICollectionViewDelegateFlowLayout {
    
    func numberOfSections(in collectionView: UICollectionView) -> Int {
        return 1
    }
    
    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return self.products.count
    }
    
    func collectionView(_ collectionView: UICollectionView,
                        willDisplay cell: UICollectionViewCell,
                        forItemAt indexPath: IndexPath) {
        let indexOfMajorCell = self.indexOfMajorCell()
        let isMajor: Bool = indexOfMajorCell == indexPath.item
        if isMajor {
            cell.layer.transform = CATransform3DScale(CATransform3DIdentity, 1.0, 1.0, 1)
        } else {
            cell.layer.transform = CATransform3DScale(CATransform3DIdentity, 0.9, 0.9, 1)
        }
    }
    
    func collectionView(_ collectionView: UICollectionView,
                        cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        let cell: TAWTokensCollectionViewCell =
            collectionView.dequeueReusableCell(withReuseIdentifier: "tokensCell",
                                               for: indexPath) as! TAWTokensCollectionViewCell
        cell.setProduct(self.products[indexPath.item])
        cell.delegate = self
        return cell
    }
    
    func collectionView(_ collectionView: UICollectionView,
                        layout collectionViewLayout: UICollectionViewLayout,
                        sizeForItemAt indexPath: IndexPath) -> CGSize {
        return CGSize(width: self.cellWidth, height: self.cellHeight)
    }
}

// MARK: - TokensCellDelegate -

extension TAWTokensViewController: TokensCellDelegate {
    
    func buy(product: SKProduct) {
        IAPClient.shared.purchaseProduct(product)
    }
}
