//
//  TAWVideoDetailViewController.swift
//  TAW
//
//  Created by Andrew Steven on 21/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import UICircularProgressRing
import AVKit
import Kingfisher

let videoPurchasedNotification = "VideoPurchasedNotification"

class TAWVideoDetailViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var videoImageView: UIImageView!
    @IBOutlet weak var lockImageView: UIImageView!
    @IBOutlet weak var premiumImageView: UIImageView!
    @IBOutlet weak var premiumImageViewOffsetX: NSLayoutConstraint!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var titleLabelOffsetY: NSLayoutConstraint!
    @IBOutlet weak var dateLabel: UILabel!
    @IBOutlet weak var playButton: UIButton!
    @IBOutlet weak var buyButton: PBButton!
    @IBOutlet weak var subscribeButton: PBButton!
    @IBOutlet weak var backgroundEnabledLabel: UILabel!
    @IBOutlet weak var backgroundEnabledLabelOffsetX: NSLayoutConstraint!
    
    @IBOutlet weak var unsubscribedView: UIView!
    @IBOutlet weak var unsubscribedViewHeight: NSLayoutConstraint!
    @IBOutlet weak var dynamicContentView: UIView!
    @IBOutlet weak var dynamicContentViewHeight: NSLayoutConstraint!
    
    @IBOutlet weak var tokensButton: UIButton!
    @IBOutlet weak var tokensActivityIndicator: UIActivityIndicatorView!
    
    @IBOutlet weak var contentViewPaddingBottom: NSLayoutConstraint!
    
    @IBOutlet weak var signUpButton: PBButton!
    @IBOutlet weak var loginButton: PBButton!
    
    // MARK: - Variables -
    
    var tokens: Int = 0
    var video: Video?
    var playerView: PBMiniPlayerView?
    var notificationCenter = NotificationCenter.default
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        APIClient.recordAnalytic(.viewedVideoArticle,
                                 variable: self.video!.postId,
                                 secondaryVariable: "1")

        self.tokensButton.titleLabel?.numberOfLines = 0
        self.configureNotifications()
        
        // Prompt user to rate the App once they have listened 2, 5, or 10 podcasts/videos
        PBAppStoreReviewer.checkKeyActivityAndAskForReview()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        self.animations = self.isMovingToParent ? [.slide(.left, .slightly), .fadeIn] : [.fadeIn]
        super.viewWillAppear(animated)

        DispatchQueue.main.async {
            self.setupView()
        }

        self.getWallet()
    }
    
    override func viewDidAppear(_ animated: Bool) {
        
        self.view.layoutIfNeeded()
        if PBAudioPlayer.shared.podcast != nil {
            if self.playerView == nil {
                let window = UIApplication.shared.keyWindow
                let bottomPadding: CGFloat = window?.safeAreaInsets.bottom ?? 0
                
                let height: CGFloat =
                    self.navigationController?.tabBarController?.tabBar.isHidden ?? true ? 60 + bottomPadding : 60
                self.playerView = PBMiniPlayerView(frame: CGRect(x: 0,
                                                                 y: self.view.frame.size.height - height,
                                                                 width: self.view.frame.size.width,
                                                                 height: height))
                self.playerView!.delegate = self
                self.view.addSubview(playerView!)
            }
            self.playerView?.updateProgressView()
        }
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        if self.playerView != nil {
            self.playerView?.stopTimer()
            self.playerView?.removeFromSuperview()
            self.playerView = nil
        }
        
        APIClient.recordAnalytic(.stoppedViewingVideoArticle,
                                 variable: self.video!.postId,
                                 secondaryVariable: "1")
    }
    
    private func hideTabBar() {
        self.tabBarController?.tabBar.isHidden = true
    }
    
    private func setVideoImage() {
        if let image = self.video?.largeImage {
            let imageProcessor = ResizingImageProcessor(referenceSize: self.videoImageView.frame.size, mode: .aspectFill)
            self.videoImageView.kf.setImage(with: URL(string: image),
                                            placeholder: UIImage(named: "placeholder_videos"),
                                            options: [.processor(imageProcessor),  .transition(.fade(0.2))])
        }
    }
    
    private func setupView() {
        guard let video = self.video, let user = LocalStorage.shared.user else {
            return
        }
        
        self.hideTabBar()
        self.setVideoImage()
        
        let guest = user.guest
        
        if user.isSubscribedToVideo ||
            !video.premium ||
            video.purchased {

            // premium state
            if user.isSubscribedToVideo {
                self.premiumImageView.image = UIImage(named: "subscribed_icon")
                self.premiumImageView.isHidden = false

            }
            else if !video.premium {
                self.premiumImageView.isHidden = true
            }
            else { // purchased
                self.premiumImageView.image = UIImage(named: "purchased_icon")
                self.premiumImageView.isHidden = false
            }
            self.lockImageView.isHidden = true
            self.premiumImageViewOffsetX.constant = 27

            // unsubscribed view
            self.unsubscribedViewHeight.constant = 0
            self.signUpButton.isHidden = true
            self.loginButton.isHidden = true
            self.buyButton.isHidden = true
            self.tokensButton.isHidden = guest
            self.playButton.isHidden = false
        }
        else { // Unsubscribed
        
            self.tokensButton.isHidden = guest
            
            // premium state
            self.premiumImageView.image = UIImage(named: "premium_icon")
            self.premiumImageView.isHidden = false
            self.lockImageView.isHidden = false
            self.premiumImageViewOffsetX.constant = 52
            
            // unsubscribed view
            self.unsubscribedViewHeight.constant = 100
            self.signUpButton.isHidden = !guest
            self.loginButton.isHidden = !guest
            self.buyButton.isHidden = guest
            self.playButton.isHidden = true
        }
        
        if guest {
            self.unsubscribedViewHeight.constant = 100
            self.signUpButton.isHidden = false
            self.loginButton.isHidden = false
            self.buyButton.isHidden = true
            self.tokensButton.isHidden = true
            self.playButton.isHidden = true
        }
        
        self.titleLabelOffsetY.constant = self.premiumImageView.isHidden ? 30 : 60
        self.buyButton.setTitle(video.formattedCost, for: .normal)
        
        DispatchQueue.main.async {
            self.titleLabel.text = video.title.htmlDecoded
        }
        
        if self.lockImageView.isHidden {
            if self.premiumImageView.isHidden {
                self.backgroundEnabledLabelOffsetX.constant = 27
            } else {
                self.backgroundEnabledLabelOffsetX.constant = self.premiumImageView.frame.maxX + 8
            }
        } else {
            if self.premiumImageView.isHidden {
                self.backgroundEnabledLabelOffsetX.constant = self.lockImageView.frame.maxX + 8
            } else {
                self.backgroundEnabledLabelOffsetX.constant = self.premiumImageView.frame.maxX + 8
            }
        }
        self.backgroundEnabledLabel.isHidden = video.type == .youtube
        
        self.dateLabel.text = video.formattedDate
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 91 : 31
        self.view.layoutIfNeeded()
        
        let contents = Utilities().parseContent(video.content)
        self.displayContent(contents)
        self.updatePlayButtonState()
    }
    
    private func displayContent(_ contents: [[String: Any]]) {
        
        var yAxis: CGFloat = 0
        let width: CGFloat = UIScreen.main.bounds.width - (2*29)
        
        for component in contents {
            
            if component["type"] as? String ?? "" == "string" {
                let text = component["value"] as? String ?? ""
                let height: CGFloat = text.height(withConstrainedWidth: width,
                                                  font: TAW.Fonts.arialRegular16!,
                                                  lineSpacing: 1,
                                                  lineHeightMultiple: 1.2)
                let label = UILabel(frame: CGRect(x: 29, y: yAxis, width: width, height: height))
                label.text = text
                label.numberOfLines = 0
                label.font = TAW.Fonts.arialRegular16
                label.textColor = UIColor.black
                label.setLineSpacing(lineSpacing: 1, lineHeightMultiple: 1.2)
                self.dynamicContentView.addSubview(label)
                yAxis += height + 5
            } else {
                if let imageURL = component["value"] as? URL {
                    if let imageSource = CGImageSourceCreateWithURL(imageURL as CFURL, nil) {
                        if let imageProperties = CGImageSourceCopyPropertiesAtIndex(imageSource, 0, nil) as Dictionary? {
                            guard let pixelWidth = imageProperties[kCGImagePropertyPixelWidth] as? CGFloat,
                                let pixelHeight = imageProperties[kCGImagePropertyPixelHeight] as? CGFloat else {
                                    continue
                            }
                            
                            let newHeight: CGFloat = Utilities().calculateHeightBasedOn(width: pixelWidth,
                                                                                        height: pixelHeight)
                            
                            let imageView = UIImageView(frame: CGRect(x: 0,
                                                                      y: yAxis,
                                                                      width: UIScreen.main.bounds.width,
                                                                      height: newHeight))
                            
                            let imageProcessor = ResizingImageProcessor(referenceSize: imageView.frame.size, mode: .aspectFill)
                            imageView.kf.setImage(with: imageURL,
                                                  placeholder: UIImage(named: "placeholder_podcasts"),
                                                  options: [.processor(imageProcessor),  .transition(.fade(0.2))])
                            
                            imageView.contentMode = .scaleAspectFill
                            self.dynamicContentView.addSubview(imageView)
                            yAxis += imageView.frame.size.height + 5
                        }
                    }
                }
            }
        }
        
        self.dynamicContentViewHeight.constant = yAxis
        self.view.layoutIfNeeded()
    }
    
    @objc private func displayPurchasedState() {
        
        self.video?.purchased = true
        self.signUpButton.isHidden = true
        self.loginButton.isHidden = true
        self.playButton.isHidden = false
        self.buyButton.isHidden = true
        self.lockImageView.isHidden = true
        
        self.premiumImageView.image = UIImage(named: "purchased_icon")
        self.premiumImageViewOffsetX.constant = 27
        self.unsubscribedViewHeight.constant = 0
        self.view.layoutIfNeeded()
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self,
                                            selector: #selector(podcastStateChanged),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(podcastStateChanged),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged),
                                            object: nil)
        self.notificationCenter.addObserver(self,
                                            selector: #selector(recordAnalyticForStoppedVideo(notification:)),
                                            name: NSNotification.Name(rawValue: TAWStoppedVideoPlayerNotification),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(displayPurchasedState),
                                            name: NSNotification.Name(rawValue: videoPurchasedNotification),
                                            object: nil)
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    @objc public func recordAnalyticForStoppedVideo(notification: NSNotification) {
        if let elapsedTime: Double = notification.userInfo?["elapsedTime"] as? Double {
            APIClient.recordAnalytic(.stoppedVideo,
                                     variable: self.video!.postId,
                                     secondaryVariable: "\(elapsedTime)")
        }
    }
    
    @objc func podcastStateChanged() {
        self.updatePlayButtonState()
    }
    
    func updatePlayButtonState() {
       if PBAudioPlayer.shared.podcast == nil {
            if self.playerView != nil {
                self.playerView?.stopTimer()
                self.playerView?.removeFromSuperview()
                self.playerView = nil
            }
        }
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 91 : 31
        self.view.layoutIfNeeded()
    }
    
    // MARK: - API functions -
    
    private func getWallet() {
        self.tokensButton.setTitle("Your Balance\n ", for: .normal)
        self.tokensActivityIndicator.startAnimating()

        if Connectivity.isConnectedToInternet {
            APIClient.getWallet { (jsonResponse, _) in
                if jsonResponse["success"] as? Bool == true {
                    self.updateTokensBalance(json: jsonResponse)
                } else {
                    self.tokensActivityIndicator.stopAnimating()
                    self.tokensButton.setTitle("Balance\nunavailable", for: .normal)
                    self.showAlertForAPIError(jsonResponse["message"] as? String ?? "")
                }
            }
        } else {
            self.tokensActivityIndicator.stopAnimating()
            self.tokensButton.setTitle("Balance\nunavailable", for: .normal)
        }
    }
    
    private func updateTokensBalance(json: NSDictionary) {
        if let wallet = json["wallet"] as? [String: Any] {
            if let tokens = wallet["balance"] as? Int {
                self.tokens = tokens
                self.tokensActivityIndicator.stopAnimating()
                self.tokensButton.setTitle("Your Balance\n\(tokens) Tokens", for: .normal)
            }
        } else {
            print("Error retrieving wallet")
            self.tokensActivityIndicator.stopAnimating()
            self.tokensButton.setTitle("Balance\nunavailable", for: .normal)
        }
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    @IBAction func play(_ sender: Any) {
        
        // Check if user has access to premium video or user has purchased this video
        if LocalStorage.shared.user!.isSubscribedToVideo || !self.video!.premium || self.video!.purchased {
            
            if PBAudioPlayer.shared.isPlaying {
                PBAudioPlayer.shared.pause()
            }
            
            // If video is hosted video
            if self.video!.type == .hosted {
                
                if let videoURL = self.video?.videoUrl {
                    if let theUrl = URL(string: videoURL) {
                    
                        let player = AVPlayer(url: theUrl)
                        player.allowsExternalPlayback = true
                        player.usesExternalPlaybackWhileExternalScreenIsActive = true
                        let viewController = AVPlayerViewController()
                        viewController.player = player
                        
                        self.modalPresentationStyle = .overFullScreen
                        self.present(viewController, animated: true) {
                            viewController.player?.play()
                            APIClient.recordAnalytic(.playedVideo,
                                                     variable: self.video!.postId,
                                                     secondaryVariable: "1")
                        }
                    }
                }
            } else {
                if let youtubeVideoVC = self.instantiateVC(identifier: "youtubeVideoVC") as? TAWYoutubeVideoViewController {
                    youtubeVideoVC.video = self.video!
                    self.modalPresentationStyle = .overFullScreen
                    self.present(youtubeVideoVC, animated: true, completion: nil)
                }
            }
        } else {
            // Show unlock to hear more
            if let tawVC = instantiateVC(identifier: "unlockVC") as? TAWUnlockViewController {
                tawVC.video = self.video
                let nav = TAWNavigationController(rootViewController: tawVC)
                nav.isNavigationBarHidden = true
                nav.modalPresentationStyle = UIModalPresentationStyle.overCurrentContext
                self.navigationController?.present(nav, animated: false, completion: nil)
            }
        }
    }
    
    @IBAction func buyEpisode(_ sender: Any) {
        guard let video = self.video else {
            return
        }
        
        if self.tokens < video.cost {
            self.showLowBalanceModal()
        } else {
            self.showPurchaseModal()
        }
    }
    
    private func showLowBalanceModal() {
        if let tawVC = instantiateVC(identifier: "alertVC") as? TAWAlertViewController {
            tawVC.set(title: "NOT ENOUGH TOKENS",
                      description: "Your token balance is too low to make this purchase. Buy tokens?",
                      type: .balance)
            tawVC.delegate = self
            self.navigationController?.present(tawVC, animated: false, completion: nil)
        }
    }
    
    private func showPurchaseModal() {
        guard let video = self.video else {
            return
        }
        
        if let tawVC = instantiateVC(identifier: "alertVC") as? TAWAlertViewController {
            tawVC.set(title: "ARE YOU SURE?",
                      description: "Are you sure you want to purchase \(video.title)",
                type: .purchase)
            tawVC.delegate = self
            self.navigationController?.present(tawVC, animated: false, completion: nil)
        }
    }
    
    private func makePurchase() {
        guard let video = self.video else {
            return
        }
        
        PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!,
                                            withTitle: "Purchasing video")
        
        let params = [
            "product_id": video.postId
        ]
        
        APIClient.purchaseItem(parameters: params) { (jsonResponse, _) in
            
            if jsonResponse["success"] as? Bool == true {
                self.updateTokensBalance(json: jsonResponse)
                self.displayPurchasedState()
                
                PBLoadingOverlay.shared.hideOverlay()
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    self.showAlertWithTitle("Error",
                                            message: jsonResponse["message"] as? String ?? "")
                }
            }
        }
    }
    
    @IBAction func howToSubscribe(_ sender: Any) {
        isSubscriptionPopupClose = false
        self.showHowToSubscribe(alertDelegate: self)
    }
    
    @IBAction func login(_ sender: Any) {
        if let viewController = self.instantiateVC(identifier: "loginVC") as? TAWLoginViewController {
            let nav = TAWNavigationController(rootViewController: viewController)
            nav.navigationBar.isHidden = true
            self.navigationController?.present(nav, animated: true, completion: nil)
        }
    }
    
    @IBAction func createAccount() {
        self.showCreateAccount()
    }
    
    @IBAction func showWallet() {
        if Connectivity.isConnectedToInternet {
            self.showUserWallet()
        } else {
            let alert = UIAlertController(title: "Error", message: "Your wallet is not available whilst offline. Please try again when your device is back online", preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "Ok", style: .cancel, handler: nil))
            self.navigationController?.present(alert, animated: true, completion: nil)
        }
    }
}

// MARK: - PBMiniPlayerViewDelegate -

extension TAWVideoDetailViewController: PBMiniPlayerViewDelegate {
    func openPlayer() {
        self.showPodcastPlayer()
    }
}

extension TAWVideoDetailViewController: TAWAlertViewControllerDelegate {
    func didSelectOption(_ option: TAWAlertViewOption, type: TAWAlertViewType) {
        switch type {
        case .purchase:
            if option == .accept {
                self.makePurchase()
            }
        case .balance:
            if option == .accept {
                self.showPurchaseTokens()
            }
        }
    }
}

extension AVPlayerViewController {
    open override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        let notifC = NotificationCenter.default
        // Post a notification to record an analytic for when user stopped video
        if let elapsed = self.player?.currentTime().seconds {
            let dataDictionary: [String: Double] = ["elapsedTime": elapsed]
            notifC.post(name: Notification.Name(rawValue: TAWStoppedVideoPlayerNotification),
                                            object: nil, userInfo: dataDictionary)
        }
        else {
            notifC.post(name: Notification.Name(rawValue: TAWStoppedYoutubePlayerNotification),
                        object: nil, userInfo: nil)
        }
    }
}
extension TAWVideoDetailViewController: TAWAlertViewDelegate{
    func freeSubsExpired()
    {
       // showHowToSubscribe()
        self.showSubscriptionView(isFreeSubExpired: true)
    }

    func acceptedNotifications() {

    }

    func unlinkAccount() {

    }

    func accountCreated() {

    }

    func didTapLogout() {

    }

}

