//
//  TAWVideosViewController.swift
//  TAW
//
//  Created by Andrew Steven on 19/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Spruce
import SkeletonView

class TAWVideosViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var headerView: UIView!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var headerViewHeight: NSLayoutConstraint!
    @IBOutlet weak var tableView: UITableView!
    @IBOutlet weak var contentViewPaddingBottom: NSLayoutConstraint!
    @IBOutlet weak var offlineView: UIView!
    
    // MARK: - Variables -
    
    var playerView: PBMiniPlayerView?
    var notificationCenter = NotificationCenter.default
    
    var videos: [Video] = []
    private let pullToRefresh = UIRefreshControl()
    
    // Paging
    var fetchedAllVideos = false
    var currentPage: Int = 1
    let pageSize: Int = 20
    var isLoading: Bool = false
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        self.tableView.refreshControl = pullToRefresh
        self.tableView.tableFooterView = UIView()
        
        pullToRefresh.addTarget(self, action: #selector(forceRefresh), for: .valueChanged)
        pullToRefresh.tintColor = TAW.Colors.pullToRefreshColor
        pullToRefresh.attributedTitle = NSAttributedString(string: "LOADING",
                                                           attributes: [
                                                            NSAttributedString.Key.foregroundColor: TAW.Colors.pullToRefreshColor,
                                                            NSAttributedString.Key.font: TAW.Fonts.dinAlternate10!
            ])
        
        let swipeGesture = UISwipeGestureRecognizer(target: self, action: #selector(forceRefresh))
        swipeGesture.direction = .down
        swipeGesture.delegate = self
        self.offlineView.addGestureRecognizer(swipeGesture)
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 60 : 0
        self.view.layoutIfNeeded()
        
        self.configureNotifications()
        
        // Check how many times the user has opened the app and prompt user to rate the App
        PBAppStoreReviewer.checkAppOpenedAndAskForReview()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        super.viewWillAppear(animated)

        self.tableView.isSkeletonable = true
        self.tabBarController?.tabBar.isHidden = false

        if LocalStorage.shared.loggedIn {
            self.checkUserSubscription()
            self.refreshData(forceRefresh: false)
        }
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        
        if PBAudioPlayer.shared.podcast != nil {
            if self.playerView == nil {
                let window = UIApplication.shared.keyWindow
                let bottomPadding: CGFloat = window?.safeAreaInsets.bottom ?? 0
                
                let height: CGFloat = self.navigationController?.tabBarController?.tabBar.isHidden ?? true ? 60 + bottomPadding : 60
                self.playerView = PBMiniPlayerView(frame: CGRect(x: 0, y: self.view.frame.size.height - height, width: self.view.frame.size.width, height: height))
                self.playerView!.delegate = self
                self.view.addSubview(playerView!)
            }
            self.playerView?.updateProgressView()
        }
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        if self.playerView != nil {
            self.playerView?.stopTimer()
            self.playerView?.removeFromSuperview()
            self.playerView = nil
        }
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(removeData), name: NSNotification.Name(rawValue: TAWUserHasLoggedOutNotification), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(removeData), name: NSNotification.Name(rawValue: TAWUserHasLoggedOutNotification), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(removeData), name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: nil)
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    @objc func removeData() {
        self.videos = []
        self.currentPage = 1
        self.fetchedAllVideos = false
        self.tableView.reloadData()
    }
    
    @objc func podcastStateChanged() {
        if PBAudioPlayer.shared.podcast == nil {
            if self.playerView != nil {
                self.playerView?.stopTimer()
                self.playerView?.removeFromSuperview()
                self.playerView = nil
            }
        }
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 60 : 0
        self.view.layoutIfNeeded()
    }
    
    // MARK: - Data fetching -
    
    @objc func forceRefresh() {
        self.checkUserSubscription()
        self.refreshData(forceRefresh: true)
    }
    
    @objc func refreshData(forceRefresh: Bool) {
        
        if !Connectivity.isConnectedToInternet {
            
            // Show "No Internet Connection" banner for 3 seconds
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!, withTitle: "No internet connection", style: .noInternet)
            self.pullToRefresh.endRefreshing()
            self.offlineView.isHidden = false
        }
        else {
            self.offlineView.isHidden = true
            
            // Check whether we have already fetched videos in the last 15 mins
            if self.videos.count == 0 {
                // Fetch latest videos with skeleton animation
                self.getVideos(animated: true)
            } else if self.videosNeedFetching() || forceRefresh {
                // Fetch latest videos whilst keeping the existing videos
                self.getVideos(animated: false)
            } else {
                // Animate existing videos
                self.tableView.spruce.prepare(with: self.tableViewAnimations)
                self.animateTableView(self.tableView)
                self.podcastStateChanged()
                self.pullToRefresh.endRefreshing()
            }
        }
    }
    
    func videosNeedFetching() -> Bool {
        
        let lastFetched = LocalStorage.shared.videosLastFetched
        if lastFetched == nil {
            LocalStorage.shared.videosLastFetched = Date()
            return true
        }
        else {
            let calendar = Calendar.current
            let now = Date()
            let components:DateComponents = (calendar as NSCalendar).components([NSCalendar.Unit.minute], from: lastFetched!, to: now, options: NSCalendar.Options())
            
            if components.minute! >= 15 {
                return true
            }
            else {
                return false
            }
        }
    }

    func isLoadingCell(for indexPath: IndexPath) -> Bool {
        return indexPath.row >= self.videos.count
    }
    
    // MARK: - API client -
    
    private func getVideos(animated: Bool) {
        
        if !self.isLoading {
            self.isLoading = true
            if animated {
                self.tableView.showAnimatedGradientSkeleton()
            }
            
            // Fetch the latest videos from API
            APIClient.getVideos(page: 1, number: self.pageSize) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    if let feed = jsonResponse["feed"] as? [[String: Any]] {

                        // Reset videos
                        self.currentPage = 1
                        self.videos = []
                        self.fetchedAllVideos = false

                        for json in feed {
                            if let video = Video(JSON: json) {
                                self.videos.append(video)
                            }
                        }

                        if self.videos.count != self.pageSize {
//                            print("Reached bottom of videos feed")
                            self.fetchedAllVideos = true
                        }

                        LocalStorage.shared.videosLastFetched = Date()
                        
                        DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.7, execute: {
                            self.tableView.reloadData()
                            if animated {
                                self.tableView.hideSkeleton()
                            }
                            self.pullToRefresh.endRefreshing()
                            self.isLoading = false
                        })
                    }
                }
                else {
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.pullToRefresh.endRefreshing()
                        self.isLoading = false
                        if animated {
                            self.tableView.hideSkeleton()
                        }
                        self.invalidateSession()
                    }
                    else {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                        self.pullToRefresh.endRefreshing()
                        self.isLoading = false
                        if animated {
                            self.tableView.hideSkeleton()
                        }
                    }
                }
            }
        }
    }
        
    private func getNextBatch() {
        
        if !self.fetchedAllVideos && !isLoading {

            self.isLoading = true
            self.currentPage += 1

            APIClient.getVideos(page: self.currentPage, number: self.pageSize) { (jsonResponse, error) in

                if jsonResponse["success"] as? Bool == true {

                    var newVideos: [Video] = []

                    if let feed = jsonResponse["feed"] as? [[String: Any]] {

                        for json in feed {
                            if let video = Video(JSON: json) {
                                newVideos.append(video)
                            }
                        }

                        if newVideos.count != self.pageSize {
//                            print("Reached bottom of videos feed")
                            self.fetchedAllVideos = true
                        }

                        self.tableView.performBatchUpdates({
                            let currentCount = self.videos.count
                            self.videos += newVideos

                            var indexPaths: [IndexPath] = []

                            for (index, _) in newVideos.enumerated() {
                                let indexPath = IndexPath(row: index + currentCount, section: 0)
                                indexPaths.append(indexPath)
                            }
                            self.tableView.insertRows(at: indexPaths, with: .automatic)
                        }, completion: { (finished) in
                            if finished {
                                self.isLoading = false
                            }
                        })
                    }
                }
                else {
                    self.isLoading = false
                }
            }
        }
    }
    
    private func checkUserSubscription() {
        
        if Connectivity.isConnectedToInternet {
            
            APIClient.checkSubscription { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    if let userJson = jsonResponse["user"] as? [String: Any] {
                        if let user = User(JSON: userJson) {
                            let needsUpdate = user.subscriptionLevel != LocalStorage.shared.user?.subscriptionLevel
                            user.authToken = LocalStorage.shared.user?.authToken
                            LocalStorage.shared.user = user
                            
                            if needsUpdate {
                                NotificationCenter.default.post(name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: self)
                                self.forceRefresh()
                            }
                        }
                    }
                }
                else {
                    let user = LocalStorage.shared.user
                    let needsUpdate = user?.subscriptionLevel != SubscriptionType.none
                    user?.subscriptions = []
                    LocalStorage.shared.user = user

                    if needsUpdate {
                        NotificationCenter.default.post(name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: self)
                        self.forceRefresh()
                    }

                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    }
                }
            }
            
        }
    }
    
    // MARK: - UIScrollView -
    
    var maxHeaderHeight: CGFloat = 100
    var minHeaderHeight: CGFloat = 70
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let offset = scrollView.contentOffset.y + scrollView.contentInset.top
        
        if offset > 0 && offset < maxHeaderHeight {
            
            if offset <= maxHeaderHeight/4 {
                let alpha: CGFloat = 1 - (offset / (maxHeaderHeight/4))
                for subview in headerView.subviews {
                    subview.alpha = alpha
                    titleLabel.alpha = 0
                }
            }
            else {
                let alpha: CGFloat = 1 - (offset / maxHeaderHeight)
                for subview in headerView.subviews {
                    subview.alpha = 0
                    titleLabel.alpha = 1 - alpha
                }
            }
            
            let percentage: CGFloat = 1 - (offset/maxHeaderHeight)
            headerViewHeight.constant = ((maxHeaderHeight - minHeaderHeight) * percentage) + minHeaderHeight
        }
        else if offset <= 0 {
            headerViewHeight.constant = maxHeaderHeight
            
            for subview in headerView.subviews {
                subview.alpha = 1
                titleLabel.alpha = 0
            }
        }
        else if offset >= maxHeaderHeight {
            headerViewHeight.constant = minHeaderHeight
            
            for subview in headerView.subviews {
                subview.alpha = 0
                titleLabel.alpha = 1
            }
        }
    }
}

// MARK: - UITableViewDelegate & UITableViewDataSource -

extension TAWVideosViewController: UITableViewDelegate {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return videos.count + 1
    }
    
    func tableView(_ tableView: UITableView, willDisplay cell: UITableViewCell, forRowAt indexPath: IndexPath) {

        if !self.isLoading && indexPath.row >= self.videos.count && Connectivity.isConnectedToInternet && self.videos.count != 0 {
            self.getNextBatch()
        }
        
        cell.spruce.prepare(with: self.tableViewAnimations)
        var animation = SpringAnimation(duration: 0.7)
        animation.animationOptions = [UIView.AnimationOptions.allowUserInteraction]
        
        cell.spruce.animate(self.tableViewAnimations, animationType: animation, sortFunction: self.tableViewSortFunction!)
        cell.layoutSubviews()
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {

        if indexPath.row >= self.videos.count {
            let cell = tableView.dequeueReusableCell(withIdentifier: "videoLoadingCell", for: indexPath) as! TAWLoadingTableViewCell
            if self.fetchedAllVideos {
                cell.titleLabel.text = "All caught up 👍"
                cell.activityIndicatorView.stopAnimating()
                cell.activityIndicatorView.isHidden = true
            } else {
                cell.titleLabel.text = "Loading more videos"
                cell.activityIndicatorView.startAnimating()
                cell.activityIndicatorView.isHidden = false
            }
            return cell
        } else {
            let cell: TAWVideoTableViewCell = tableView.dequeueReusableCell(withIdentifier: "videosCell", for: indexPath) as! TAWVideoTableViewCell
            if indexPath.row < self.videos.count {
                cell.setVideo(self.videos[indexPath.row], imageQuality: .medium)
            }

            return cell
        }
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        if self.videos.count <= indexPath.row && self.videos.count != 0 {
            return 100
        } else {
            return 312
        }
    }
    
    func tableView(_ tableView: UITableView, estimatedHeightForRowAt indexPath: IndexPath) -> CGFloat {
        if self.videos.count <= indexPath.row && self.videos.count != 0 {
            return 100
        } else {
            return 312
        }
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        if self.videos.count <= indexPath.row {
            return
        }
        
        let video = self.videos[indexPath.row]
        self.performSegue(withIdentifier: "showVideoDetail", sender: video)
        DispatchQueue.main.async {}
    }
    
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "showVideoDetail" {
            if let vc = segue.destination as? TAWVideoDetailViewController {
                vc.video = sender as? Video
            }
        }
    }
}

extension TAWVideosViewController: SkeletonTableViewDataSource {
    
    func numSections(in collectionSkeletonView: UITableView) -> Int {
        return 1
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return 2
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, cellIdentifierForRowAt indexPath: IndexPath) -> ReusableCellIdentifier {
        let identifier = indexPath.row == 0 ? "videosCell" : "videosCell"
        return identifier
    }
}

extension TAWVideosViewController: PBMiniPlayerViewDelegate {
    
    func openPlayer() {
        self.showPodcastPlayer()
    }
}
