//
//  TAWWritingDetailViewController.swift
//  TAW
//
//  Created by Andrew Steven on 21/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import CoreText
import Kingfisher
import ImageIO

class TAWWritingDetailViewController: PBAnimatedViewController, UIScrollViewDelegate {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var writingImageView: UIImageView!
    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var dateLabelView: UIView!
    @IBOutlet weak var dateLabel: UILabel!
    @IBOutlet weak var titleTextView: PBPaddedTextView!
    @IBOutlet weak var headerView: PBView!
    @IBOutlet weak var scrollView: UIScrollView!
    @IBOutlet weak var scrollViewContentView: UIView!
    @IBOutlet weak var gradientImageView: UIImageView!
    @IBOutlet weak var dynamicContentView: UIView!
    @IBOutlet weak var dynamicContentViewHeight: NSLayoutConstraint!
    @IBOutlet weak var contentViewPaddingBottom: NSLayoutConstraint!
    @IBOutlet weak var expandingImageHeight: NSLayoutConstraint!
    @IBOutlet weak var authorImageView: UIImageView!
    @IBOutlet weak var authorLabel: UILabel!
    
    // MARK: - Variables -
    
    var writing: Writing?
    var playerView: PBMiniPlayerView?
    var notificationCenter = NotificationCenter.default
    var contentOffset: CGFloat = 0
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        self.headerView.fillColor = UIColor.clear
        let backImage = UIImage(named: "back_arrow_icon")?.withRenderingMode(.alwaysTemplate)
        self.backButton.setImage(backImage, for: .normal)
        self.backButton.tintColor = .white
        
        self.contentOffset = 2 * (self.view.frame.size.height/10)
        self.scrollView.contentInset = UIEdgeInsets(top: contentOffset, left: 0, bottom: 0, right: 0)
        
        APIClient.recordAnalytic(.viewedWritingArticle, variable: writing!.postId, secondaryVariable: "1")
        self.configureNotifications()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        self.animations = [.slide(.left, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        
        DispatchQueue.main.async {
            self.setupView()
            self.setupArticle()
        }
        self.tabBarController?.tabBar.isHidden = true
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        self.view.layoutIfNeeded()
        if PBAudioPlayer.shared.podcast != nil {
            if self.playerView == nil {
                let window = UIApplication.shared.keyWindow
                let bottomPadding: CGFloat = window?.safeAreaInsets.bottom ?? 0
                
                let isTabBarHidden = self.navigationController?.tabBarController?.tabBar.isHidden
                let height: CGFloat = isTabBarHidden ?? true ? 60 + bottomPadding : 60
                self.playerView = PBMiniPlayerView(frame: CGRect(x: 0,
                                                                 y: self.view.frame.size.height - height,
                                                                 width: self.view.frame.size.width,
                                                                 height: height))
                self.playerView!.delegate = self
                self.view.addSubview(playerView!)
            }
            self.playerView?.updateProgressView()
        }
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        APIClient.recordAnalytic(.stoppedViewingWritingArticle, variable: writing!.postId, secondaryVariable: "1")
        
        if self.playerView != nil {
            self.playerView?.stopTimer()
            self.playerView?.removeFromSuperview()
            self.playerView = nil
        }
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self,
                                            selector: #selector(podcastStateChanged),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(podcastStateChanged),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged),
                                            object: nil)
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    @objc func podcastStateChanged() {
        self.updatePlayButtonState()
    }
    
    func updatePlayButtonState() {
        if PBAudioPlayer.shared.podcast == nil {
            if self.playerView != nil {
                self.playerView?.stopTimer()
                self.playerView?.removeFromSuperview()
                self.playerView = nil
            }
        }
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 91 : 31
        self.view.layoutIfNeeded()
    }
    
    private func setupView() {
        
        self.expandingImageHeight.constant = UIScreen.main.bounds.height * 0.7
        self.view.layoutIfNeeded()
        
        if let image = self.writing?.largeImage {
            self.writingImageView.kf.setImage(with: URL(string: image),
                                              placeholder: UIImage(named: "placeholder_writing"),
                                              options: [.transition(.fade(0.2))])
        }
        
        self.titleTextView.font = TAW.Fonts.dinPro24
        self.titleTextView.textColor = UIColor.white
        self.titleTextView.backgroundColor = UIColor.clear
        
        DispatchQueue.main.async {
            self.titleTextView.text = self.writing?.title.htmlDecoded
        }
        
        self.authorLabel.text = self.writing?.author ?? ""
        
        self.dateLabel.text = self.writing?.formattedDate
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 91 : 31
        self.view.layoutIfNeeded()
    }
    
    private func setupArticle() {
        let contents = Utilities().parseArticle(self.writing!.content)
        self.displayArticle(contents)
    }
    
    private func displayArticle(_ contents: [String]) {
        
        var yAxis: CGFloat = 0
        let width: CGFloat = UIScreen.main.bounds.width - (2*20)
        
        for component in contents {
            if let url = URL(string: component) {
                if let imageSource = CGImageSourceCreateWithURL(url as CFURL, nil) {
                    if let imageProperties =
                        CGImageSourceCopyPropertiesAtIndex(imageSource, 0, nil) as Dictionary? {
                        guard let pixelWidth = imageProperties[kCGImagePropertyPixelWidth] as? CGFloat,
                            let pixelHeight = imageProperties[kCGImagePropertyPixelHeight] as? CGFloat else {
                                continue
                        }
                        
                        let newHeight: CGFloat = Utilities().calculateHeightBasedOn(width: pixelWidth,
                                                                                    height: pixelHeight)
                        
                        let imageView = UIImageView(frame: CGRect(x: 0,
                                                                  y: yAxis,
                                                                  width: UIScreen.main.bounds.width,
                                                                  height: newHeight))
                        imageView.kf.setImage(with: url,
                                              placeholder: UIImage(named: "placeholder_podcasts"),
                                              options: [.transition(.fade(0.2))])
                        
                        imageView.contentMode = .scaleAspectFill
                        self.dynamicContentView.addSubview(imageView)
                        yAxis += imageView.frame.size.height + 5
                    }
                }
            }
            else {
                let height: CGFloat = component.height(withConstrainedWidth: width,
                                                  font: TAW.Fonts.arialRegular16!,
                                                  lineSpacing: 1,
                                                  lineHeightMultiple: 1.2)
                
                let label = UILabel(frame: CGRect(x: 20, y: yAxis, width: width, height: height))
                label.text = component
                label.numberOfLines = 0
                label.font = TAW.Fonts.arialRegular16
                label.textColor = UIColor.black
                label.setLineSpacing(lineSpacing: 1, lineHeightMultiple: 1.2)
                self.dynamicContentView.addSubview(label)
                yAxis += height + 5
            }
        }
        
        // If no content for article, we need to show white background enought to cover image with white drop shadow gap
        if yAxis == 27 {
            yAxis = 100
        }
        self.dynamicContentViewHeight.constant = yAxis
        self.view.layoutIfNeeded()
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    // MARK: - UIScrollView -
    
    let maxHeight = UIScreen.main.bounds.height * 0.7
    let minHeight: CGFloat = 114
    
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let offset = scrollView.contentOffset.y + scrollView.contentInset.top
        let actualOffset = self.contentOffset + self.gradientImageView.frame.size.height - 114
        
        if offset > 0 && offset < actualOffset {
            
            let difference = maxHeight - minHeight
            let percentage: CGFloat = 1 - (offset / difference)
            self.expandingImageHeight.constant = minHeight + (percentage * difference)
            
            if offset <= (3*(actualOffset/4)) {
                headerView.fillColor = UIColor.clear
                let backImage = UIImage(named: "back_arrow_icon")?.withRenderingMode(.alwaysTemplate)
                self.backButton.setImage(backImage, for: .normal)
                self.backButton.tintColor = .white
            } else {
                let alpha: CGFloat = 1 - (offset / actualOffset)
                headerView.fillColor = UIColor.white.withAlphaComponent(1 - alpha)
                let backImage = UIImage(named: "back_arrow_icon")?.withRenderingMode(.alwaysTemplate)
                self.backButton.setImage(backImage, for: .normal)
                self.backButton.tintColor = UIColor.black.withAlphaComponent(1 - alpha)
            }
        } else if offset <= 0 {

            self.expandingImageHeight.constant = maxHeight
            headerView.fillColor = UIColor.clear
            let backImage = UIImage(named: "back_arrow_icon")?.withRenderingMode(.alwaysTemplate)
            self.backButton.setImage(backImage, for: .normal)
            self.backButton.tintColor = .white
        } else if offset >= actualOffset {

            self.expandingImageHeight.constant = minHeight
            headerView.fillColor = UIColor.white
            let backImage = UIImage(named: "back_arrow_icon")?.withRenderingMode(.alwaysTemplate)
            self.backButton.setImage(backImage, for: .normal)
            self.backButton.tintColor = .black
        }
        
        self.scrollViewContentView.bringSubviewToFront(self.dateLabelView)
        self.scrollViewContentView.bringSubviewToFront(self.titleTextView)
        headerView.layoutSubviews()
    }
}

// MARK: - PBMiniPlayerViewDelegate -

extension TAWWritingDetailViewController: PBMiniPlayerViewDelegate {
    func openPlayer() {
        self.showPodcastPlayer()
    }
}
