//
//  TAWWritingsViewController.swift
//  TAW
//
//  Created by Andrew Steven on 19/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Spruce
import SkeletonView

class TAWWritingsViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var headerContainerView: PBView!
    @IBOutlet weak var headerView: UIView!
    @IBOutlet weak var titleLabel: PBPaddedLabel!
    @IBOutlet weak var headerViewHeight: NSLayoutConstraint!
    @IBOutlet weak var tableView: UITableView!
    @IBOutlet weak var contentViewPaddingBottom: NSLayoutConstraint!
    @IBOutlet weak var offlineView: UIView!
    
    // MARK: - Variables -
    
    var playerView: PBMiniPlayerView?
    var notificationCenter = NotificationCenter.default
    
    var writings: [Writing] = []
    private let pullToRefresh = UIRefreshControl()
    
    // Paging
    var fetchedAllWritings = false
    var currentPage: Int = 1
    let pageSize: Int = 20
    var isLoading: Bool = false
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()

        self.tableView.refreshControl = pullToRefresh
        self.tableView.tableFooterView = UIView()
        
        pullToRefresh.addTarget(self, action: #selector(forceRefresh), for: .valueChanged)
        pullToRefresh.tintColor = TAW.Colors.pullToRefreshColor
        pullToRefresh.attributedTitle = NSAttributedString(string: "LOADING",
                                                           attributes: [NSAttributedString.Key.foregroundColor: TAW.Colors.pullToRefreshColor,
                                                                        NSAttributedString.Key.font: TAW.Fonts.dinAlternate10!])
        
        let swipeGesture = UISwipeGestureRecognizer(target: self, action: #selector(forceRefresh))
        swipeGesture.direction = .down
        swipeGesture.delegate = self
        self.offlineView.addGestureRecognizer(swipeGesture)
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 60 : 0
        self.view.layoutIfNeeded()
        
        self.configureNotifications()
        
        // Check how many times the user has opened the app and prompt user to rate the App
        PBAppStoreReviewer.checkAppOpenedAndAskForReview()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        super.viewWillAppear(animated)
        self.tableView.isSkeletonable = true
        refreshData(forceRefresh: false)
        
        self.tabBarController?.tabBar.isHidden = false
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        
        if PBAudioPlayer.shared.podcast != nil {
            if self.playerView == nil {
                let window = UIApplication.shared.keyWindow
                let bottomPadding: CGFloat = window?.safeAreaInsets.bottom ?? 0
                
                let height: CGFloat = self.navigationController?.tabBarController?.tabBar.isHidden ?? true ? 60 + bottomPadding : 60
                self.playerView = PBMiniPlayerView(frame: CGRect(x: 0, y: self.view.frame.size.height - height, width: self.view.frame.size.width, height: height))
                self.playerView!.delegate = self
                self.view.addSubview(playerView!)
            }
            self.playerView?.updateProgressView()
        }
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        if self.playerView != nil {
            self.playerView?.stopTimer()
            self.playerView?.removeFromSuperview()
            self.playerView = nil
        }
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(removeData), name: NSNotification.Name(rawValue: TAWUserHasLoggedOutNotification), object: nil)
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    @objc func removeData() {
        self.writings = []
        self.currentPage = 1
        self.fetchedAllWritings = false
        self.tableView.reloadData()
    }
    
    @objc func podcastStateChanged() {
        if PBAudioPlayer.shared.podcast == nil {
            if self.playerView != nil {
                self.playerView?.stopTimer()
                self.playerView?.removeFromSuperview()
                self.playerView = nil
            }
        }
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 60 : 0
        self.view.layoutIfNeeded()
    }
    
    // MARK: - Data fetching -
    
    @objc func forceRefresh() {
        self.refreshData(forceRefresh: true)
    }
    
    @objc func refreshData(forceRefresh: Bool) {
        
        if !Connectivity.isConnectedToInternet {
            
            // Show "No Internet Connection" banner for 3 seconds
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!, withTitle: "No internet connection", style: .noInternet)
            self.pullToRefresh.endRefreshing()
            self.offlineView.isHidden = false
        } else {
            self.offlineView.isHidden = true
            
            // Check whether we have already fetched writings in the last 15 mins
            if self.writings.count == 0 {
                // Fetch latest writings
                self.getWritings(animated: true)
            } else if self.writingsNeedFetching() || forceRefresh {
                // Fetch latest writings
                self.getWritings(animated: false)
            } else {
                // Animate existing writings
                self.tableView.spruce.prepare(with: self.tableViewAnimations)
                self.animateTableView(self.tableView)
                self.podcastStateChanged()
                self.pullToRefresh.endRefreshing()
            }
        }
    }
    
    func writingsNeedFetching() -> Bool {
        
        let lastFetched = LocalStorage.shared.writingsLastFetched
        if lastFetched == nil {
            LocalStorage.shared.writingsLastFetched = Date()
            return true
        } else {
            let calendar = Calendar.current
            let now = Date()
            let components:DateComponents = (calendar as NSCalendar).components([NSCalendar.Unit.minute],
                                                                                from: lastFetched!,
                                                                                to: now,
                                                                                options: NSCalendar.Options())
            
            if components.minute! >= 15 {
                return true
            } else {
                return false
            }
        }
    }

    func isLoadingCell(for indexPath: IndexPath) -> Bool {
        return indexPath.row >= self.writings.count
    }
    
    // MARK: - API functions -
    
    private func getWritings(animated: Bool) {
        
        if !self.isLoading {
            self.isLoading = true
            if animated {
                self.tableView.showAnimatedGradientSkeleton()
            }
            
            // Fetch the latest writings from API
            APIClient.getWritings(page: 1, number: self.pageSize) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    if let feed = jsonResponse["feed"] as? [[String: Any]] {

                        // Reset writings
                        self.currentPage = 1
                        self.writings = []
                        self.fetchedAllWritings = false

                        for json in feed {
                            if let writing = Writing(JSON: json) {
                                self.writings.append(writing)
                            }
                        }
                        
                        if self.writings.count != self.pageSize {
//                            print("Reached bottom of writings feed")
                            self.fetchedAllWritings = true
                        }
                        
                        LocalStorage.shared.writingsLastFetched = Date()
                        
                        DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.7, execute: {
                            self.tableView.reloadData()
                            if animated {
                                self.tableView.hideSkeleton()
                            }
                            self.pullToRefresh.endRefreshing()
                            self.isLoading = false
                        })
                    }
                } else {
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.pullToRefresh.endRefreshing()
                        self.isLoading = false
                        if animated {
                            self.tableView.hideSkeleton()
                        }
                        self.invalidateSession()
                    } else {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                        self.pullToRefresh.endRefreshing()
                        self.isLoading = false
                        if animated {
                            self.tableView.hideSkeleton()
                        }
                    }
                }
            }
        }
    }
    
    private func getNextBatch() {
        
        if !self.fetchedAllWritings && !isLoading {
            
            self.isLoading = true
            self.currentPage += 1
            
            APIClient.getWritings(page: self.currentPage, number: self.pageSize) { (jsonResponse, error) in
                
                if jsonResponse["success"] as? Bool == true {
                    
                    var newWritings: [Writing] = []
                    
                    if let feed = jsonResponse["feed"] as? [[String: Any]] {
                        
                        for json in feed {
                            if let writing = Writing(JSON: json) {
                                newWritings.append(writing)
                            }
                        }
                        
                        if newWritings.count != self.pageSize {
//                            print("Reached bottom of writings feed")
                            self.fetchedAllWritings = true
                        }
                        
                        self.tableView.performBatchUpdates({
                            
                            let currentCount = self.writings.count
                            self.writings += newWritings
                            
                            var indexPaths: [IndexPath] = []
                            
                            for (index, _) in newWritings.enumerated() {
                                let indexPath = IndexPath(row: index + currentCount, section: 0)
                                indexPaths.append(indexPath)
                            }
                            
                            self.tableView.insertRows(at: indexPaths, with: .automatic)
                            
                        }, completion: { (_) in
                            self.isLoading = false
                        })
                    }
                } else {
                    self.isLoading = false
                }
            }
        }
    }
    
    // MARK: - Life cycle -
    
    var maxHeaderHeight: CGFloat = 100
    var minHeaderHeight: CGFloat = 70
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let offset = scrollView.contentOffset.y + scrollView.contentInset.top
        
        if offset > 0 && offset < maxHeaderHeight {
            
            if offset <= maxHeaderHeight/4 {
                let alpha: CGFloat = 1 - (offset / (maxHeaderHeight/4))
                for subview in headerView.subviews {
                    subview.alpha = alpha
                    titleLabel.alpha = 0
                }
            } else {
                let alpha: CGFloat = 1 - (offset / maxHeaderHeight)
                for subview in headerView.subviews {
                    subview.alpha = 0
                    titleLabel.alpha = 1 - alpha
                }
            }
            
            let percentage: CGFloat = 1 - (offset/maxHeaderHeight)
            headerViewHeight.constant = ((maxHeaderHeight - minHeaderHeight) * percentage) + minHeaderHeight
        } else if offset <= 0 {
            headerViewHeight.constant = maxHeaderHeight
            
            for subview in headerView.subviews {
                subview.alpha = 1
                titleLabel.alpha = 0
            }
        } else if offset >= maxHeaderHeight {
            headerViewHeight.constant = minHeaderHeight
            
            for subview in headerView.subviews {
                subview.alpha = 0
                titleLabel.alpha = 1
            }
        }
    }
}

// MARK: - UITableViewDelegate & UITableViewDataSource -

extension TAWWritingsViewController: UITableViewDelegate {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return writings.count + 1
    }
    
    func tableView(_ tableView: UITableView, willDisplay cell: UITableViewCell, forRowAt indexPath: IndexPath) {
        
        if !self.isLoading && indexPath.row >= self.writings.count && Connectivity.isConnectedToInternet && self.writings.count != 0 {
            self.getNextBatch()
        }
        
        cell.spruce.prepare(with: self.tableViewAnimations)
        var animation = SpringAnimation(duration: 0.7)
        animation.animationOptions = [UIView.AnimationOptions.allowUserInteraction]
        
        cell.spruce.animate(self.tableViewAnimations, animationType: animation, sortFunction: self.tableViewSortFunction!)
        cell.layoutSubviews()
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {

        if indexPath.row >= self.writings.count {
            let cell = tableView.dequeueReusableCell(withIdentifier: "writingLoadingCell", for: indexPath) as! TAWLoadingTableViewCell
            if self.fetchedAllWritings {
                cell.titleLabel.text = "All caught up 👍"
                cell.activityIndicatorView.stopAnimating()
                cell.activityIndicatorView.isHidden = true
            } else {
                cell.titleLabel.text = "Loading more articles"
                cell.activityIndicatorView.startAnimating()
                cell.activityIndicatorView.isHidden = false
            }
            return cell
        } else {
            let identifier = indexPath.row == 0 ? "featuredWritingCell" : "writingCell"
            let imageQuality: PodcastImageQuality = indexPath.row == 0 ? .large : .medium
            let cell: TAWFeedTableViewCell = tableView.dequeueReusableCell(withIdentifier: identifier, for: indexPath) as! TAWFeedTableViewCell
            if indexPath.row < self.writings.count {
                cell.setWriting(self.writings[indexPath.row], imageQuality: imageQuality)
            }

            return cell
        }
    }
    
    func tableView(_ tableView: UITableView, estimatedHeightForRowAt indexPath: IndexPath) -> CGFloat {
        if self.writings.count <= indexPath.row && self.writings.count != 0 {
            return 100
        } else {
            return indexPath.row == 0 ? 300 : 114
        }
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        if self.writings.count <= indexPath.row && self.writings.count != 0 {
            return 100
        } else {
            return indexPath.row == 0 ? 300 : 114
        }
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        if self.writings.count <= indexPath.row {
            return
        }
        
        let writing = self.writings[indexPath.row]
        self.performSegue(withIdentifier: "showWritingDetail", sender: writing)
        DispatchQueue.main.async {}
    }
    
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "showWritingDetail" {
            if let vc = segue.destination as? TAWWritingDetailViewController {
                vc.writing = sender as? Writing
            }
        }
    }
}

extension TAWWritingsViewController: SkeletonTableViewDataSource {
    
    func numSections(in collectionSkeletonView: UITableView) -> Int {
        return 1
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return UIDevice.isSmallDevice ? 1 : 3
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView,
                                cellIdentifierForRowAt indexPath: IndexPath) -> ReusableCellIdentifier {
        let identifier = indexPath.row == 0 ? "featuredWritingCell" : "writingCell"
        return identifier
    }
}

// MARK: - PBMiniPlayerViewDelegate -

extension TAWWritingsViewController: PBMiniPlayerViewDelegate {
    func openPlayer() {
        self.showPodcastPlayer()
    }
}
