//
//  APIClient.swift
//  TAW
//
//  Created by Andrew Steven on 07/01/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import Alamofire

class APIClient {
    
    private static func performRequest(route: String, method: HTTPMethod, parameters: Parameters, completionHandler: @escaping (NSDictionary, Error?) -> Void) {
        URLCache.shared.removeAllCachedResponses()
        
        var authToken = ""
        if LocalStorage.shared.user?.authToken != nil {
            authToken = LocalStorage.shared.user!.authToken!
        }
        
        let uuid: String = LocalStorage.shared.uuid!
        
        var headers: HTTPHeaders = [
            "api-token": TAW.ProductionServer.apiToken,
          //  "user-api-token": authToken,
            "device": uuid,
            "device-name": UIDevice.modelName,
            "app-version": Utilities().currentAppVersion()
        ]
        if authToken != ""
        {
            headers["user-api-token"] = authToken
        }

        let url = "\(TAW.ProductionServer.baseURL)\(route)"
        
        print("Api URL: \(url)")
        print("Api Headers: \(headers)")
        print("API Parameters: \(parameters)")

        Alamofire.request(url,
                          method: method,
                          parameters: parameters,
                          encoding: URLEncoding.httpBody,
                          headers: headers).responseJSON { (response:DataResponse<Any>) in
                            
            print("API Response: \(response)")
            switch response.result {
            case .success(let value):
                completionHandler(value as? NSDictionary ?? [:], nil)
            case .failure(let error):
                let theErrorMessage = ["success": false, "message": error.localizedDescription, "error": 1] as [String: Any]
                completionHandler(theErrorMessage as NSDictionary, error)
            }
        }
    }
    
    // MARK: - User Services -
    
    static func checkEmailIsAvailable(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/email/check"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func checkUsernameIsAvailable(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/username/check"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func registerUser(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/register"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func registerGuest(completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/register/guest"
        performRequest(route: route, method: .post, parameters: [:], completionHandler: completion)
    }
    
    static func convertToFullUser(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/register/guest/convert"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func activateUser(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/activate"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func loginUser(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/login"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func logoutUser(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/logout"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func recoverPassword(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/recover"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func resetPassword(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/reset"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func changePassword(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/password"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func updateUserDetails(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/update"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func updateUserEmail(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/update/email"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func updateUserAvatar(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/avatar/upload"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func checkEmail(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/email/check"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func resendActivationEmail(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/activation/resend"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func associateDeviceToken(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/notifications/associate"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    // MARK: - A Member Subscriptions -
    
    static func checkSubscription(completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/subscription/check"
        performRequest(route: route, method: .post, parameters: [:], completionHandler: completion)
    }
    
    static func getFreeSubscription(completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "free-subscriptions"
        performRequest(route: route, method: .post, parameters: [:], completionHandler: completion)
    }
    
    static func activateSubscription(parameters: Parameters,completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "user/activate-subscription"//"user/activate-paid-subscription"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func activateFreeSubscription(parameters: Parameters,completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "user/activate-free-subscription"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func unlinkAccount(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = ""
        performRequest(route: route, method: .post, parameters: [:], completionHandler: completion)
    }
    
    // MARK: - Device Management -
    
    static func getDevices(completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/devices"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    static func unlinkDevice(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/devices/remove"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    // MARK: - Wallet -
    
    static func getWallet(completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/wallet"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    static func purchaseItem(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/wallet/purchase"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func creditAccount(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/wallet/credit"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    // MARK: - Preferences -
    
    static func getPreferences(completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/preferences"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    static func updateNotificationPreference(type: String,
                                             parameters: Parameters,
                                             completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "users/preferences/\(type)"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    // MARK: - Podcasts -
    
    static func getPodcasts(page: Int, number: Int, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "feed/podcasts/\(page)/limit/\(number)"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    static func getPodcastById(postId: String, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "feed/podcast/\(postId)"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    // MARK: - Videos -
    
    static func getVideos(page: Int, number: Int, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "feed/videos/\(page)/limit/\(number)"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    static func getVideoById(postId: String, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "feed/video/\(postId)"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    // MARK: - Writings -
    
    static func getWritings(page: Int, number: Int, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "feed/writing/\(page)/limit/\(number)"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    static func getArticleById(postId: String, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "feed/article/\(postId)"
        performRequest(route: route, method: .get, parameters: [:], completionHandler: completion)
    }
    
    // MARK: - Analytics -
    
    private static func recordAnalytic(parameters: Parameters, completion: @escaping (NSDictionary, Error?) -> Void) {
        let route = "analytics/record"
        performRequest(route: route, method: .post, parameters: parameters, completionHandler: completion)
    }
    
    static func recordAnalytic(_ type: AppAnalytic, variable: String, secondaryVariable: String) {
        
        let params = [
            "type": type.rawValue,
            "variable": variable,
            "secondary_variable": secondaryVariable
        ]
        
        self.recordAnalytic(parameters: params) { (jsonResponse, _) in
            print("Recorded Analytic for \(type.rawValue): \(jsonResponse)")
        }
    }
}
