//
//  SZXSwitch.swift
//  Sizzix
//
//  Created by Andrew Steven on 04/04/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit

@IBDesignable
final class PBSwitch: UIControl {
        
    // MARK: Public properties
    
    @IBInspectable public var isOn: Bool = true
    
    public var animationDuration: Double = 0.5

    @IBInspectable
    var borderWidth: CGFloat {
        get {
            return layer.borderWidth
        }
        set {
            layer.borderWidth = newValue
        }
    }
    
    @IBInspectable
    var borderColor: UIColor? {
        get {
            if let color = layer.borderColor {
                return UIColor(cgColor: color)
            }
            return nil
        }
        set {
            if let color = newValue {
                layer.borderColor = color.cgColor
            } else {
                layer.borderColor = nil
            }
        }
    }
    
    @IBInspectable  public var padding: CGFloat = 1 {
        didSet {
            self.layoutSubviews()
        }
    }
    
    @IBInspectable  public var onTintColor: UIColor = UIColor(red: 144/255, green: 202/255, blue: 119/255, alpha: 1) {
        didSet {
            self.setupUI()
        }
    }
    
    @IBInspectable public var offTintColor: UIColor = UIColor.black {
        didSet {
            self.setupUI()
        }
    }
    
    @IBInspectable public var cornerRadius: CGFloat {
        
        get {
            return self.privateCornerRadius
        }
        set {
            if newValue > 0.5 || newValue < 0.0 {
                privateCornerRadius = 0.5
            } else {
                privateCornerRadius = newValue
            }
        }
        
    }
    
    private var privateCornerRadius: CGFloat = 0.5 {
        didSet {
            self.layoutSubviews()
        }
    }
    
    // thumb properties
    @IBInspectable public var thumbTintColor: UIColor =
        UIColor(red: 255/255, green: 128/255, blue: 139/255, alpha: 1.0) {
        didSet {
            self.thumbView.backgroundColor = self.thumbTintColor
        }
    }
    
    @IBInspectable public var thumbCornerRadius: CGFloat {
        get {
            return self.privateThumbCornerRadius
        }
        set {
            if newValue > 0.5 || newValue < 0.0 {
                privateThumbCornerRadius = 0.5
            } else {
                privateThumbCornerRadius = newValue
            }
        }
        
    }
    
    private var privateThumbCornerRadius: CGFloat = 0.5 {
        didSet {
            self.layoutSubviews()
        }
    }
    
    @IBInspectable public var thumbSize: CGSize = CGSize.zero {
        didSet {
            self.layoutSubviews()
        }
    }
    
    @IBInspectable public var thumbImage: UIImage? = nil {
        didSet {
            guard let image = thumbImage else {
                return
            }
            thumbView.thumbImageView.image = image
        }
    }
    
    public var onImage: UIImage? {
        didSet {
            self.onImageView.image = onImage
            self.layoutSubviews()
        }
    }
    
    public var offImage: UIImage? {
        didSet {
            self.offImageView.image = offImage
            self.layoutSubviews()
        }
    }

    @IBInspectable public var thumbShadowColor: UIColor = UIColor.black {
        didSet {
            self.thumbView.layer.shadowColor = self.thumbShadowColor.cgColor
        }
    }
    
    @IBInspectable public var thumbShadowOffset: CGSize = CGSize(width: 0.75, height: 2) {
        didSet {
            self.thumbView.layer.shadowOffset = self.thumbShadowOffset
        }
    }
    
    @IBInspectable public var thumbShaddowRadius: CGFloat = 1.5 {
        didSet {
            self.thumbView.layer.shadowRadius = self.thumbShaddowRadius
        }
    }
    
    @IBInspectable public var thumbShaddowOppacity: Float = 0.4 {
        didSet {
            self.thumbView.layer.shadowOpacity = self.thumbShaddowOppacity
        }
    }
    
    // labels
    
    public var labelOff: UILabel = UILabel()
    public var labelOn: UILabel = UILabel()
    
    public var areLabelsShown: Bool = true {
        didSet {
            self.setupUI()
        }
    }
    
    // MARK: Private properties
    fileprivate var thumbView = PBThumbView(frame: CGRect.zero)
    fileprivate var onImageView = UIImageView(frame: CGRect.zero)
    fileprivate var offImageView = UIImageView(frame: CGRect.zero)
    
    fileprivate var onPoint = CGPoint.zero
    fileprivate var offPoint = CGPoint.zero
    fileprivate var isAnimating = false
    
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        
        self.setupUI()
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        
        self.setupUI()
    }
}

// MARK: Private methods
extension PBSwitch {
    
    fileprivate func setupUI() {
        
        // clear self before configuration
        self.clear()
        
        self.clipsToBounds = false
        
        // configure thumb view
        self.thumbView.backgroundColor = self.thumbTintColor
        self.thumbView.isUserInteractionEnabled = false
        
        self.thumbView.layer.shadowColor = self.thumbShadowColor.cgColor
        self.thumbView.layer.shadowRadius = self.thumbShaddowRadius
        self.thumbView.layer.shadowOpacity = self.thumbShaddowOppacity
        self.thumbView.layer.shadowOffset = self.thumbShadowOffset
        
        self.backgroundColor = self.isOn ? self.onTintColor : self.offTintColor
        
        self.addSubview(self.thumbView)
        self.addSubview(self.onImageView)
        self.addSubview(self.offImageView)
        
        self.setupLabels()
    }
    
    private func clear() {
        for view in self.subviews {
            view.removeFromSuperview()
        }
    }
    
    override func beginTracking(_ touch: UITouch, with event: UIEvent?) -> Bool {
        super.beginTracking(touch, with: event)
        
        self.animate()
        return true
    }
    
    func setOn(on: Bool, animated: Bool) {
        
        switch animated {
        case true:
            self.animate(on: on)
        case false:
            self.isOn = on
            self.setupViewsOnAction()
            self.completeAction()
        }
    }
    
    fileprivate func animate(on: Bool? = nil) {
        
        self.isOn = on ?? !self.isOn
        
        self.isAnimating = true
        
        let thumbRed = UIColor(red: 237/255, green: 28/255, blue: 36/255, alpha: 1.0)
        let thumbGreen = UIColor(red: 65/255, green: 117/255, blue: 5/255, alpha: 1.0)
        
        UIView.animate(withDuration: self.animationDuration,
                       delay: 0, usingSpringWithDamping: 0.7,
                       initialSpringVelocity: 0.5,
                       options: [UIView.AnimationOptions.curveEaseOut,
                                 UIView.AnimationOptions.beginFromCurrentState,
                                 UIView.AnimationOptions.allowUserInteraction],
                       animations: {
                        
            self.setupViewsOnAction()
            
            self.labelOn.font = (self.isOn) ? TAW.Fonts.textFieldTextFont : TAW.Fonts.textFieldTextFont
            self.labelOff.font = (self.isOn) ? TAW.Fonts.textFieldTextFont : TAW.Fonts.textFieldTextFont
            self.labelOn.textColor = (self.isOn) ? UIColor.white : UIColor(red: 218/255, green: 218/255, blue: 218/255, alpha: 1.0)
            self.labelOff.textColor = (self.isOn) ? UIColor(red: 218/255, green: 218/255, blue: 218/255, alpha: 1.0) : UIColor.white
            self.thumbTintColor = (self.isOn) ? thumbGreen : thumbRed
            
        }, completion: { _ in
            self.completeAction()
        })
    }
    
    private func setupViewsOnAction() {
        self.thumbView.frame.origin.x = self.isOn ? self.onPoint.x : self.offPoint.x
        self.backgroundColor = self.isOn ? self.onTintColor : self.offTintColor
        self.setOnOffImageFrame()
    }
    
    private func completeAction() {
        self.isAnimating = false
        self.sendActions(for: UIControl.Event.valueChanged)
    }
}

// MARK: - Public methods -
extension PBSwitch {
    
    public override func layoutSubviews() {
        super.layoutSubviews()
        
        if !self.isAnimating {
            self.layer.cornerRadius = self.bounds.size.height * self.cornerRadius
            self.backgroundColor = self.isOn ? self.onTintColor : self.offTintColor
            
            // thumb managment
            // get thumb size, if none set, use one from bounds
            let thumbSize : CGSize =
                self.thumbSize != CGSize.zero ? self.thumbSize : CGSize(width: self.bounds.size.height - 2,
                                                                                    height: self.bounds.height - 2)
            let yPostition = (self.bounds.size.height - thumbSize.height) / 2
            
            self.onPoint = CGPoint(x: self.padding, y: yPostition)
            self.offPoint = CGPoint(x: self.bounds.size.width - thumbSize.width - self.padding, y: yPostition)
            
            self.thumbView.frame = CGRect(origin: self.isOn ? self.onPoint : self.offPoint, size: thumbSize)
            self.thumbView.layer.cornerRadius = thumbSize.height * self.thumbCornerRadius
            
            if self.areLabelsShown {
                self.labelOn.center = CGPoint(x: self.onPoint.x + self.thumbView.frame.size.width / 2,
                                              y: self.thumbView.center.y)
                
                self.labelOff.center = CGPoint(x: self.offPoint.x + self.thumbView.frame.size.width / 2,
                                               y: self.thumbView.center.y)
            }
            
            // on/off images
            //set to preserve aspect ratio of image in thumbView
            
            guard onImage != nil && offImage != nil else {
                return
            }
            
            let frameSize = thumbSize.width > thumbSize.height ? thumbSize.height * 0.7 : thumbSize.width * 0.7
            let onOffImageSize = CGSize(width: frameSize, height: frameSize)
            
            self.onImageView.frame.size = onOffImageSize
            self.offImageView.frame.size = onOffImageSize
            
            self.onImageView.center = CGPoint(x: self.onPoint.x + self.thumbView.frame.size.width / 2,
                                              y: self.thumbView.center.y)
            
            self.offImageView.center = CGPoint(x: self.offPoint.x + self.thumbView.frame.size.width / 2,
                                               y: self.thumbView.center.y)
            
            self.onImageView.alpha = self.isOn ? 1.0 : 0.0
            self.offImageView.alpha = self.isOn ? 0.0 : 1.0
        }
    }
}

// MARK: - Labels frame -
extension PBSwitch {
    
    fileprivate func setupLabels() {
        guard self.areLabelsShown else {
            self.labelOff.alpha = 0
            self.labelOn.alpha = 0
            return
        }
        
        let onTextColor = UIColor(red: 218/255, green: 218/255, blue: 218/255, alpha: 1.0)
        let offTextColor = UIColor.white
        
        self.labelOff.alpha = 1
        self.labelOn.alpha = 1
        
        let labelWidth = (self.bounds.width / 2) - (self.padding * 2)
        
        self.labelOn.frame = CGRect(x: self.padding,
                                    y: self.padding,
                                    width: labelWidth,
                                    height: self.frame.height - (self.padding * 2))
        
        self.labelOff.frame = CGRect(x: self.frame.width - labelWidth - self.padding,
                                     y: self.padding,
                                     width: labelWidth,
                                     height: self.frame.height - (self.padding * 2))
        
        self.labelOn.font = self.isOn ? TAW.Fonts.textFieldTextFont : TAW.Fonts.textFieldTextFont
        self.labelOff.font = self.isOn ? TAW.Fonts.textFieldTextFont : TAW.Fonts.textFieldTextFont
        self.labelOn.textColor = self.isOn ? offTextColor : onTextColor
        self.labelOff.textColor = self.isOn ? onTextColor : offTextColor
        
        self.labelOff.text = "OFF"
        self.labelOn.text = "ON"
        self.labelOff.textAlignment = .center
        self.labelOn.textAlignment = .center
        
        self.insertSubview(self.labelOff, aboveSubview: self.thumbView)
        self.insertSubview(self.labelOn, aboveSubview: self.thumbView)
    }
}

// MARK: - Animating on/off images -
extension PBSwitch {
    
    fileprivate func setOnOffImageFrame() {
        guard onImage != nil && offImage != nil else {
            return
        }
        
        self.onImageView.center.x = self.isOn ? self.onPoint.x + self.thumbView.frame.size.width / 2 : self.frame.width
        self.offImageView.center.x = !self.isOn ? self.offPoint.x + self.thumbView.frame.size.width / 2 : 0
        self.onImageView.alpha = self.isOn ? 1.0 : 0.0
        self.offImageView.alpha = self.isOn ? 0.0 : 1.0
    }
}
