package co.pixelbeard.theanfieldwrap.podcastDetails;

import android.content.Context;
import android.util.Log;

import com.tonyodev.fetch2.Download;

import co.pixelbeard.theanfieldwrap.data.Podcast;
import co.pixelbeard.theanfieldwrap.data.RealmPodcast;
import co.pixelbeard.theanfieldwrap.data.responses.GenericResponse;
import co.pixelbeard.theanfieldwrap.data.responses.GetWalletResponse;
import co.pixelbeard.theanfieldwrap.data.responses.PodcastByIdResponse;
import co.pixelbeard.theanfieldwrap.data.source.DataRepository;
import co.pixelbeard.theanfieldwrap.networking.CallbackWrapper;
import co.pixelbeard.theanfieldwrap.utils.PrefKey;
import co.pixelbeard.theanfieldwrap.utils.PrefUtils;
import io.reactivex.Observer;
import io.reactivex.disposables.Disposable;

public class PodcastDetailsPresenter implements PodcastDetailsContract.Presenter {


    private PodcastDetailsContract.View mView;
    private DataRepository mRepo;

    public PodcastDetailsPresenter(PodcastDetailsContract.View view, DataRepository repo) {
        if (view != null && repo != null) {
            this.mView = view;
            this.mRepo = repo;

            mView.setPresenter(this);
        }
    }


    @Override
    public void downloadPodcast(Podcast podcast, Context context) {
        mRepo.downloadPodcast(podcast, context);
    }

    @Override
    public void savePodcast(Podcast podcast, Download download) {
        if (download.getUrl().equals(podcast.getPodcastFullUrl()) && download.getProgress() == 100) {
            mRepo.savePodcast(podcast, download.getFile());
        }
    }

    @Override
    public void recordAnalytic(String type, String variable, String secondVar) {
        mRepo.recordAnalytic(type, variable, secondVar).subscribe(new Observer<GenericResponse>() {
            @Override
            public void onSubscribe(Disposable d) {
            }

            @Override
            public void onNext(GenericResponse genericResponse) {
                if (genericResponse != null) {
                    Log.d("ANALYTIC: " + type, genericResponse.getMessage());
                }
            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onComplete() {

            }
        });
    }

    @Override
    public RealmPodcast getPodcastByIdLocal(long id) {
        return mRepo.getPodcastByIdLocal(id);
    }

    @Override
    public void getPodcastByIdRemote(long id) {
        if (mView != null && mView.checkConnection()) {
            mView.showLoader("Loading");
            mRepo.getPodcastByIdRemote(id).subscribe(new CallbackWrapper<PodcastByIdResponse>(mView) {
                @Override
                protected void onSuccess(PodcastByIdResponse podcastByIdResponse) {
                    mView.setPodcast(podcastByIdResponse.getPodcast());
                }

                @Override
                protected void onSuccessFalse(PodcastByIdResponse podcastByIdResponse) {
                    mView.goBack();
                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void purchaseItem(String id) {
        if (mView != null && mView.checkConnection()) {
            mView.showLoader("Purchasing");
            mRepo.purchaseItem(id).subscribe(new CallbackWrapper<GetWalletResponse>(mView) {
                @Override
                protected void onSuccess(GetWalletResponse getWalletResponse) {
                    PrefUtils.getInstance().putIntPref(PrefKey.TOKEN_BALANCE, getWalletResponse.getWallet().getBalance());
                    mView.setPodcastPurchased(true);
                    mView.updateWalletBalance(getWalletResponse.getWallet().getBalance());
                }

                @Override
                protected void onSuccessFalse(GetWalletResponse getWalletResponse) {
                    if (getWalletResponse.getMessage().contains("balance was too low")) {
                        mView.showBalanceTooLowDialog();
                    } else {
                        mView.showErrorDialog(getWalletResponse.getMessage());
                    }
                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void checkWalletBalance() {
        if (mView != null && mView.checkConnection()) {
            mRepo.getWallet().subscribe(new CallbackWrapper<GetWalletResponse>(mView) {
                @Override
                protected void onSuccess(GetWalletResponse getWalletResponse) {
                    mView.updateWalletBalance(getWalletResponse.getWallet().getBalance());
                    PrefUtils.getInstance().putIntPref(PrefKey.TOKEN_BALANCE, getWalletResponse.getWallet().getBalance());
                }

                @Override
                protected void onSuccessFalse(GetWalletResponse getWalletResponse) {
                    mView.updateWalletBalance(PrefUtils.getInstance().getIntPref(PrefKey.TOKEN_BALANCE));
                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void subscribe() {

    }

    @Override
    public void unsubscribe() {
        mRepo.onDestroyRepo();
    }

    @Override
    public void clearUserData(Context context) {
        //TODO test this
//        mRepo.clearAllData(context);
        mRepo.logout().subscribe(new Observer<GenericResponse>() {
            @Override
            public void onSubscribe(Disposable d) {

            }

            @Override
            public void onNext(GenericResponse genericResponse) {
                Log.d("LOGOUT", genericResponse.getMessage());
            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onComplete() {

            }
        });

    }
}
