package co.pixelbeard.theanfieldwrap.videos;

import android.content.Context;
import android.os.Handler;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;

import co.pixelbeard.theanfieldwrap.data.Video;
import co.pixelbeard.theanfieldwrap.data.responses.GenericResponse;
import co.pixelbeard.theanfieldwrap.data.responses.LoginResponse;
import co.pixelbeard.theanfieldwrap.data.responses.VideosResponse;
import co.pixelbeard.theanfieldwrap.data.source.DataRepository;
import co.pixelbeard.theanfieldwrap.networking.CallbackWrapper;
import co.pixelbeard.theanfieldwrap.utils.PrefUtils;
import io.reactivex.Observer;
import io.reactivex.disposables.Disposable;

/**
 * Created by Michael Stoddart
 * 31/12/2018.
 */
public class VideosPresenter implements VideosContract.Presenter {

    private static final String TAG = VideosPresenter.class.getSimpleName();

    private VideosContract.View mView;
    private DataRepository mRepo;

    public VideosPresenter(VideosContract.View view, DataRepository repo) {
        if (view != null && repo != null) {
            this.mView = view;
            this.mRepo = repo;

            mView.setPresenter(this);
        }
    }

    @Override
    public void getVideos(int pageNum) {

        if (mView != null && mView.checkConnection()) {
            if (pageNum == 1) {
                mView.showLoader("");
            }
//            else {
//                mView.setSwipeRefreshRefreshing(true);
//            }

            mRepo.getVideos(pageNum).subscribe(new CallbackWrapper<VideosResponse>(mView) {
                @Override
                protected void onSuccess(VideosResponse videosResponse) {
                    List<Video> videos = videosResponse.getVideos();
                    new Handler().postDelayed(() -> {
                        if (mView != null) {

                            mView.setVideosAdapter(videos);
                            mView.showUi();
                        }
                    }, 50);
                }

                @Override
                protected void onSuccessFalse(VideosResponse videosResponse) {
                    mView.showSingleButtonDialog("Error", videosResponse.getMessage());

                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void checkUserSubs() {
        if (mView != null && mView.checkConnection()) {
            mRepo.checkUserSubscription().subscribe(new CallbackWrapper<LoginResponse>(mView) {
                @Override
                protected void onSuccess(LoginResponse loginResponse) {
                    PrefUtils.getInstance().storeSubs(loginResponse.getUser().getSubscriptions());
                }

                @Override
                protected void onSuccessFalse(LoginResponse loginResponse) {
                    if (loginResponse.getMessage().equals("User is not currently subscribed")) {
                        PrefUtils.getInstance().storeSubs(new ArrayList<>());
                    }
                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void refreshVideos() {
        if (mView != null) {
            if (mView.checkConnection()) {
                checkUserSubs();
                getVideos(1);
            } else {
                try {
                    mView.showNoConnectionHeader();
                    mView.setSwipeRefreshRefreshing(false);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    @Override
    public void recordAnalytic(String type, String variable, String secondVar) {
        mRepo.recordAnalytic(type, variable, secondVar).subscribe(new Observer<GenericResponse>() {
            @Override
            public void onSubscribe(Disposable d) {
            }

            @Override
            public void onNext(GenericResponse genericResponse) {
                if (genericResponse != null) {
                    Log.d("ANALYTIC: " + type, genericResponse.getMessage());
                }
            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onComplete() {

            }
        });
    }

    @Override
    public void subscribe() {
        getVideos(1);
    }

    @Override
    public void unsubscribe() {
        mRepo.onDestroyRepo();
        mView = null;
    }

    @Override
    public void clearUserData(Context context) {
        //TODO test this
//        mRepo.clearAllData(context);
        mRepo.logout().subscribe(new Observer<GenericResponse>() {
            @Override
            public void onSubscribe(Disposable d) {

            }

            @Override
            public void onNext(GenericResponse genericResponse) {
                Log.d("LOGOUT", genericResponse.getMessage());
            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onComplete() {

            }
        });

    }
}
