<?php
/**
 * @table integration
 * @id phpbb
 * @title phpBB
 * @visible_link http://www.phpbb.com/
 * @description phpBB is a high powered, fully scalable, and highly customizable
 * Open Source bulletin board package. phpBB has a user-friendly interface,
 * simple and straightforward administration panel, and helpful FAQ. Based on
 * the powerful PHP server language and your choice of MySQL, MS-SQL,
 * PostgreSQL or Access/ODBC database servers, phpBB is the ideal free
 * community solution for all web sites.
 * @different_groups 1
 * @single_login 1
 * @type Bulletin Boards
 * @am_protect_api 6.0
 */
class Am_Protect_Phpbb extends Am_Protect_Databased
{
    const PLUGIN_STATUS = self::STATUS_PRODUCTION; //dev status
    const PLUGIN_COMM = self::COMM_COMMERCIAL; //paid
    const PLUGIN_REVISION = '6.2.12';

    protected $groupMode = self::GROUP_MULTI;
    protected $guessTablePattern = "users";
    protected $guessFieldsPattern = [
        'user_id',
        'user_type',
        'group_id',
        'username',
        'username_clean'
    ];
    const PHPBB = 'phpbb';
    const USER_TYPE_NORMAL = 0;
    const USER_TYPE_INACTIVE = 1;
    protected $_bbconfig = [];

    protected $versions = [330 =>'3.3.0 and higher',316 => '3.1.6',300 => '3.0'];

    public function afterAddConfigItems(Am_Form_Setup_ProtectDatabased $form)
    {
        $form->addText("protect.{$this->getId()}.cookie_name")
            ->setLabel("phpBB cookie name\n" .
            "Get this value from phpBB admin -> Server Configuration -> Cookie Settings");
        $form->addText("protect.{$this->getId()}.cookie_domain")
            ->setLabel("phpBB cookie domain\n" .
            "Get this value from phpBB admin -> Server Configuration -> Cookie Settings");
        $form->addText("protect.{$this->getId()}.cookie_path")
            ->setLabel("phpBB cookie path\n" .
            "Get this value from phpBB admin -> Server Configuration -> Cookie Settings");
        $form->addSelect("protect.{$this->getId()}.version", '', ['options' => $this->versions])
            ->setLabel('phpBB version');

    }

    public function getPasswordFormat()
    {
        return ($this->getConfig('version') >= 316 ? SavedPassTable::PASSWORD_PASSWORD_HASH : self::PHPBB);
    }

    public function parseExternalConfig($path)
    {
        $config_path = $path . "/config.php";
        if (!is_file($config_path) || !is_readable($config_path))
            throw new Am_Exception_InputError("Specified path is not a valid phpBB installation!");

        include $config_path;

        if (!$dbms || !$dbhost)
            throw new Am_Exception_InputError("Specified path is not a valid phpBB installation!");
        if (!preg_match('/mysql/', $dbms))
            throw new Am_Exception_InputError("phpBB is not configured to use MySQL. Can't be integrated");
        $settings = [
            'host' => $dbhost,
            'db' => $dbname,
            'user' => $dbuser,
            'pass' => $dbpasswd,
            'prefix' => $table_prefix,
            'other_db' => '1'
        ];

        $class = get_class($this);
        $obj = new $class(Am_Di::getInstance(), $settings);
        $db = $obj->getDb();

        foreach (['cookie_name', 'cookie_domain', 'cookie_path'] as $k) {
            $settings[$k] = $obj->getBBConfig($k);
        }
        return $settings;
    }

    function getLastVisit(Am_Record $record)
    {
        return $record->user_lastvisit ? $record->user_lastvisit : $this->getDi()->time;
    }

    function createSessionTable()
    {
        $table = new Am_Protect_SessionTable($this, $this->getDb(), '?_sessions', 'session_id');
        $table->setTableConfig([
            Am_Protect_SessionTable::FIELD_SID => 'session_id',
            Am_Protect_SessionTable::FIELD_UID => 'session_user_id',
            Am_Protect_SessionTable::FIELD_CREATED => 'session_start',
            Am_Protect_SessionTable::FIELD_CHANGED => 'session_time',
            Am_Protect_SessionTable::FIELD_IP => 'session_ip',
            Am_Protect_SessionTable::FIELD_UA => 'session_browser',
            Am_Protect_SessionTable::COOKIE_PARAMS => [
                'domain' => $this->getConfig('cookie_domain'),
                'path' => $this->getConfig('cookie_path'),
                'exact_domain' => true
            ],
            Am_Protect_SessionTable::FIELDS_ADDITIONAL  => [
                'session_last_visit'  =>  [$this, "getLastVisit"]
            ],
            Am_Protect_SessionTable::SESSION_COOKIE => $this->getConfig('cookie_name') . "_sid",
            Am_Protect_SessionTable::COOKIES_ADDITIONAL => [
                $this->getConfig('cookie_name') . "_u" => function($r) {return $r->user_id;}
            ]
        ]);
        return $table;
    }

    function getAvailableUserGroupsSql()
    {
        return "SELECT
                group_id as id,
                group_name as title,
                (group_id in (4,5)) as is_admin
                FROM ?_groups";
    }

    function cryptPassword($pass, &$salt = null, User $user = null)
    {
        $ph = new PasswordHashPhpBB(8, true);
        // validation request
        if ($salt && $ph->CheckPassword($pass, $salt))
            return $salt;
        // new password
        return $ph->HashPassword($pass);
    }

    function getBBConfig($var)
    {
        if (empty($this->_bbconfig))
        {
            foreach ($this->getDb()->select("select config_name, config_value from ?_config") as $r)
            {
                $this->_bbconfig[$r['config_name']] = $r['config_value'];
            }
        }
        return @$this->_bbconfig[$var];
    }

    function getUserEmailHash($user, $table)
    {
        return crc32(strtolower($user->email) . strlen($user->email));
    }

    function getUserNameClean($user, $table)
    {
        return trim(mb_strtolower($user->login));
    }

    function createTable()
    {
        $table = new Am_Protect_Phpbb_Table($this, $this->getDb(), '?_users', 'user_id');
        $mapping = [
            [Am_Protect_Table::FIELD_REMOTE_ADDR, 'user_ip'],
            [Am_Protect_Table::FIELD_ADDED_STAMP, 'user_regdate'],
            [Am_Protect_Table::FIELD_LOGIN, 'username'],
            [[$this, "getUserNameClean"], 'username_clean'],
            [Am_Protect_Table::FIELD_EMAIL, 'user_email'],
            [Am_Protect_Table::FIELD_PASS, 'user_password'],
            [Am_Protect_Table::FIELD_ADDED_SQL, 'user_passchg'],
            [':' . $this->getBBConfig('default_lang'), 'user_lang'],
            [':' . $this->getBBConfig('default_dateformat'), 'user_dateformat'],
            [':' . $this->getBBConfig('board_timezone'), 'user_timezone'],
            [':' . $this->getBBConfig('default_style'), 'user_style'],
            [':' . self::USER_TYPE_NORMAL, 'user_type']
        ];
        if ($this->getConfig('version') < 330) {
            $mapping[] = [[$this, 'getUserEmailHash'], 'user_email_hash'];
        }
        $table->setFieldsMapping($mapping);
        $table->setGroupsTableConfig([
            Am_Protect_Table::GROUP_TABLE => '?_user_group',
            Am_Protect_Table::GROUP_GID => 'group_id',
            Am_Protect_Table::GROUP_UID => 'user_id',
            Am_Protect_Table::GROUP_ADDITIONAL_FIELDS => ['user_pending' => 0]
        ]);
        return $table;
    }


    function canAutoCreate()
    {
        return true;
    }
}

class Am_Protect_Phpbb_Table extends Am_Protect_Table
{
    function insertFromAmember(User $user, SavedPass $pass, $groups)
    {
        $record = parent::insertFromAmember($user, $pass, $groups);
        $this->getPlugin()->getDb()->query("UPDATE ?_config SET config_value=? WHERE config_name=?", $record->username, 'newest_username');
        $this->getPlugin()->getDb()->query("UPDATE ?_config SET config_value=? WHERE config_name=?", $record->user_id++, 'newest_user_id');
        $this->getPlugin()->getDb()->query("UPDATE ?_config SET config_value=config_value+1 WHERE config_name=?", 'num_users');
        $this->getPlugin()->getDb()->query("UPDATE ?_config SET config_value=? WHERE config_name=?", $record->user_colour, 'newest_user_colour');
        return $record;
    }

    function updateFromAmember(Am_Record $record, User $user, $groups)
    {
        parent::updateFromAmember($record, $user, $groups);
        $record->username_clean = $this->getPlugin()->getUserNameClean($user, $this);
        $fields = ['username_clean'];
        if ($this->getPlugin()->getConfig('version')<330) {
            $record->user_email_hash = $this->getPlugin()->getUserEmailHash($user, $this);
            $fields[] = 'user_email_hash';
        }
        $record->updateSelectedFields($fields);
    }

    function setGroups(Am_Record $record, $groups)
    {
        parent::setGroups($record, $groups);
        $group_id = array_shift($groups);
        $record->user_type = $group_id ? Am_Protect_Phpbb::USER_TYPE_NORMAL : Am_Protect_Phpbb::USER_TYPE_INACTIVE;
        $record->group_id = $group_id;
        $record->user_colour = $this->getPlugin()->getDb()->selectCell("SELECT group_colour FROM ?_groups WHERE group_id = ?", $group_id);
        $record->user_permissions = '';
        $record->user_rank = $this->getAdapter()->selectCell("SELECT rank_id FROM ?_ranks WHERE rank_min<=? AND rank_special=0 ORDER BY rank_min DESC LIMIT 1", $record->user_posts) ?: 0;
        $record->updateSelectedFields(['group_id', 'user_colour', 'user_type', 'user_permissions', 'user_rank']);
        $newest_user_id = $this->getAdapter()->selectCell("SELECT config_value FROM ?_config WHERE config_name='newest_user_id';");
        if ($newest_user_id && $newest_user_id == $record->pk()) {
            $this->getAdapter()->query("UPDATE ?_config SET config_value = ? WHERE config_name='newest_user_colour';", $record->user_colour);
        }
    }

    public function findByAmember(User $user)
    {
        $ret = $this->findFirstBy(['username_clean' => strtolower($user->login)]);
        return $ret;
    }

    function findAmember(Am_Record $record)
    {
        $objs = $this->getDi()->userTable->selectObjects("SELECT * from ?_user WHERE LOWER(login) = ? LIMIT 1", strtolower($record->username_clean));
        if ($objs)
            return current($objs);
    }

    function removeRecord(Am_Record $record)
    {
        parent::removeRecord($record);
        $this->getPlugin()->getDb()->query("UPDATE ?_config SET config_value=config_value-1 WHERE config_name= ?", 'num_users');
    }
}

spl_autoload_register(function($classname) {
if ($classname == 'PasswordHashPhpBB') :
    class PasswordHashPhpBB extends PasswordHash
    {
        function gensalt_private($input)
        {
            $output = '$H$';
            $output .= $this->itoa64[min($this->iteration_count_log2 +
                    ((PHP_VERSION >= '5') ? 5 : 3), 30)];
            $output .= $this->encode64($input, 6);
            return $output;
        }
    }
endif;
});