<?php

/**
 * @title A/B Signup
 * @setup_url ab-signup/admin-campaign
 *
 * @desc
 * This module allows to test signup forms and frontend themes
 * in A/B mode (form layout/form fields/payment plans)
 * and measure conversion rate for each form/theme
 * it allow to include more than two forms in same campaign
 *
 * @todo track clicks to measure conversion rate
 */
class Bootstrap_AbSignup extends Am_Module
{
    const COOKIE_NAME_FORM = 'am-form';
    const COOKIE_NAME_THEME = 'am-theme';
    const COOKIE_NAME_FORM_CUSTOM = 'am-form-';
    const CAMPAIGN_SAVEDFORM = 'saved_form';
    const CAMPAIGN_SAVEDFORM_CUSTOM = 'saved_form_custom';
    const CAMPAIGN_THEME = 'theme';

    public function onInitFinished()
    {
        $this->getDi()->front->registerPlugin(new Am_Mvc_Controller_Plugin_AbSignup($this), 1000);
    }

    function encodeTheme($theme)
    {
        return "{$theme}-{$this->getDi()->security->siteHash($theme)}";
    }

    function decodeTheme($data)
    {
        list($theme, $hash) = explode("-", $data);
        return ($hash == $this->getDi()->security->siteHash($theme)) ? $theme : null;
    }

    public function onAdminMenu(Am_Event $e)
    {
        $e->getMenu()
            ->addPage([
                'id' => 'ab-signup',
                'controller' => 'admin-campaign',
                'module' => 'ab-signup',
                'label' => ___("A/B Signup")
            ]);
    }

    function onLoadReports()
    {
        class_exists('Am_Report_Absignup', true);
    }

    function onLoadSignupForm(Am_Event $e)
    {
        $c = $this->getActiveCampaign(
            ($code = $e->getRequest()->getParam('c')) ? self::CAMPAIGN_SAVEDFORM_CUSTOM : self::CAMPAIGN_SAVEDFORM,
            $code
        );
        if (!$c) {
            return;
        } //there is not active SAVEDFORM campaign

        $cookie_name = $c->campaign_type == self::CAMPAIGN_SAVEDFORM_CUSTOM ?
            self::COOKIE_NAME_FORM_CUSTOM . $c->code :
            self::COOKIE_NAME_FORM;

        /* @var $request Am_Mvc_Request */
        $request = $e->getRequest();
        $form_id = null;

        $roster = $c->getRoster();

        if ($form = $this->getDi()->security->reveal($request->getCookie($cookie_name))) {
            if (in_array($form, $roster))
                $form_id = $form;
        } else {
            $form_id = $roster[array_rand($roster)];
        }

        try {
            $form = $this->getDi()->savedFormTable->load($form_id);
            $form->_campaign_id = $c->pk();
            $e->setReturn($form);
            Am_Cookie::set($cookie_name,
                    $this->getDi()->app->obfuscate($form_id),
                    amstrtotime($c->expire_date . ' 23:59:59'));
        } catch (Exception $e) {
            Am_Cookie::set($cookie_name, null, $this->getDi()->time - 24 * 3600);
        }
    }

    function onInvoiceSignup(Am_Event $e)
    {
        $invoice = $e->getInvoice();
        $savedForm = $e->getSavedForm();
        if (!empty($savedForm->_campaign_id)) {
            $invoice->campaign_id = $savedForm->_campaign_id;
        }
    }

    function onInvoiceBeforeInsert(Am_Event $e)
    {
        if (defined('AM_ADMIN') && AM_ADMIN) return;
        $c = $this->getActiveCampaign(self::CAMPAIGN_THEME);
        if (!$c) return;
        /* @var $invoice Invoice */
        $invoice = $e->getInvoice();
        if (isset($_COOKIE[self::COOKIE_NAME_THEME]) && ($theme = $this->decodeTheme($_COOKIE[self::COOKIE_NAME_THEME])))
        {
            $invoice->campaign_id = $c->pk();
            $invoice->theme = $theme;
        }
    }

    function getActiveCampaign($campaignType, $code = null)
    {
        switch ($campaignType)
        {
            case self::CAMPAIGN_SAVEDFORM_CUSTOM:
                return $this->getDi()->absignupCampaignTable->findFirstBy([
                    'campaign_type' => $campaignType,
                    'is_active' => 1,
                    ['begin_date', '<=', sqlDate('now')],
                    ['expire_date', '>=', sqlDate('now')],
                    'code' => $code,
                ]);
            case self::CAMPAIGN_SAVEDFORM:
                $type = [AbsignupCampaign::T_BOTH];
                $type[] = $this->getDi()->auth->getUserId() ? AbsignupCampaign::T_MEMBER : AbsignupCampaign::T_SIGNUP;

                return $this->getDi()->absignupCampaignTable->findFirstBy([
                    'campaign_type' => $campaignType,
                    'is_active' => 1,
                    ['begin_date', '<=', sqlDate('now')],
                    ['expire_date', '>=', sqlDate('now')],
                    'type' => $type,
                ]);

            case self::CAMPAIGN_THEME:
                return $this->getDi()->absignupCampaignTable->findFirstBy([
                    'campaign_type' => $campaignType,
                    'is_active' => 1,
                    ['begin_date', '<=', sqlDate('now')],
                    ['expire_date', '>=', sqlDate('now')],
                ]);
        }
    }
}

class Am_Mvc_Controller_Plugin_AbSignup extends Zend_Controller_Plugin_Abstract
{
    public function __construct(Bootstrap_AbSignup $plugin)
    {
        $this->plugin = $plugin;
    }

    public function preDispatch(Zend_Controller_Request_Abstract $request)
    {
        if (defined('AM_ADMIN') && AM_ADMIN) return;
        $c = $this->plugin->getActiveCampaign(Bootstrap_AbSignup::CAMPAIGN_THEME);
        if (!$c) return; //there is not active THEME campaign

        $themes = $c->getRoster();
        $theme = null;
        if (
            isset($_COOKIE[Bootstrap_AbSignup::COOKIE_NAME_THEME])
            && ($theme = $this->plugin->decodeTheme($_COOKIE[Bootstrap_AbSignup::COOKIE_NAME_THEME]))
            && !in_array($theme, $themes)
        ) {
            $theme = null;
        }
        if (!$theme) $theme = $themes[array_rand($themes)];
        $this->plugin->getDi()->config->set('theme', $theme);
        Am_Cookie::set(Bootstrap_AbSignup::COOKIE_NAME_THEME, $this->plugin->encodeTheme($theme), amstrtotime($c->expire_date . ' 23:59:59'));
    }
}