import ImageUpload from './imageupload';

class FeedEditor {

	/**
	 * Episode import data.
	 */
	importData;

	/**
	 * Manage Feed editor options.
	 * 
	 * @since 2.0
	 */
	constructor() {

		// Define variables.
		this.data = window.ppjsAdminOpt || {};
		this.action = jQuery('.pp-feed-edit, .pp-feed-sticky, .pp-remove-sticky, .pp-add-bulk, .pp-remove-bulk');
		this.editWrapper  = jQuery('.pp-toolkit-editor');
		this.importWrapper = jQuery('#pp-import-container');
		this.importedWrapper = jQuery('#pp-imported-container');
		this.abortAction = false;

		// Run methods.
		this.events();
	}

	// Event handling.
	events() {
		const _this = this;
		const toolkit = jQuery('#pp-options-module-toolkit');
		const importSelect = jQuery('#pp-feed-import-index');
		const importBtn = jQuery('.feed-url-import-btn');
		const editSelect = jQuery('#pp-feed-edit-index');
		const editBtn = jQuery('.feed-url-edit-btn');
		const tabContainer = jQuery('#pp-import-tabs-container');

		const resetImport = function() {
			const str = _this.data.messages;

			// Tab reset status.
			tabContainer.find('.toolkit-imported').removeClass('active');
			tabContainer.find('.toolkit-importer').addClass('active');
			
			// Reset import wrapper
			_this.importWrapper.find('.pp-toolkit-importer').html(str.import).removeAttr('data-podcast');
			_this.importWrapper.find('.pp-toolkit-import-options, .pp-import-podcast-btn, .pp-import-abort-btn').hide();

			// Reset imported wrapper
			_this.importedWrapper.find('.pp-toolkit-imported').html(str.imported).removeAttr('data-podcast');
			_this.importedWrapper.find('.pp-imported-podcast-btn').hide();

			// Hide all.
			tabContainer.hide();
			_this.importWrapper.hide();
			_this.importedWrapper.hide();
		}

		this.action.on('click', function() {
			const $this = jQuery(this);
			const wrapper = $this.parents('.pp-toolkit-content');
			const sVal = wrapper.find('.select-pp-feed-index').val();
			const inputVal = wrapper.find('.pp-toolkit-url').val();
			const feedBack = wrapper.find('.pp-toolkit-feedback');
			_this.ajaxRequest( $this, sVal, inputVal, feedBack );
		} );

		this.editWrapper.on(
			'click',
			'.pp-edit-save',
			function() {
				const episode = jQuery(this).closest('.pp-episode-wrapper');
				_this.saveEpisodeData(jQuery(episode));
			}
		);

		this.editWrapper.on(
			'click',
			'.pp-episode-title',
			function() {
				const form = jQuery(this).siblings('.pp-episode-form');
				if (form.hasClass('toggled-on')) {
					form.removeClass('toggled-on');
				} else {
					_this.editWrapper.find('.pp-episode-form.toggled-on').not('.pp-form-pending-save').removeClass('toggled-on');
					form.addClass('toggled-on');
				}
				
			}
		);

		this.editWrapper.on(
			'click',
			'.pp-edit-load-more',
			function() {
				const $this = jQuery(this);
				const elems = _this.editWrapper.find('.pp-episode-wrapper:hidden');
				const len   = elems.length;
				if (len) {
					elems.slice(0, 5).show();
					if (5 >= len) $this.hide();
				} else {
					$this.hide();
				}
			}
		);

		this.editWrapper.on(
			'click',
			'.dashicons-no',
			function() {
				_this.editWrapper.slideUp(
					300,
					function() {
						_this.editWrapper.empty();
					}
				);
			}
		);

		this.editWrapper.on(
			'change keyup',
			'input, select',
			function() {
				const form = jQuery(this).closest('.pp-episode-form');
				form.addClass('pp-form-pending-save');
				form.find('.pp-toolkit-feedback').removeClass('pp-error pp-success pp-running');
			}
		);

		importSelect.on( 'change', function() {
			const $this = jQuery(this);
			const wrapper = $this.parents('.pp-toolkit-content');
			const sVal = $this.val();
			const inputVal = '';
			const feedBack = wrapper.find('.pp-toolkit-feedback');
			resetImport();
			_this.ajaxRequest( $this, sVal, inputVal, feedBack );
		});

		importBtn.on( 'click', function() {
			const $this = jQuery(this);
			const wrapper = $this.parents('.pp-toolkit-content');
			const sVal = '';
			const inputVal = $this.prev('.label-feed-import').find('.pp-toolkit-url').val();
			const feedBack = wrapper.find('.pp-toolkit-feedback');
			resetImport();
			_this.ajaxRequest( $this, sVal, inputVal, feedBack );
		});

		editSelect.on( 'change', function() {
			const $this = jQuery(this);
			const wrapper = $this.parents('.pp-toolkit-content');
			const sVal = $this.val();
			const inputVal = '';
			const feedBack = wrapper.find('.pp-toolkit-feedback');
			_this.editWrapper.hide();
			_this.ajaxRequest( $this, sVal, inputVal, feedBack );
		});

		editBtn.on( 'click', function() {
			const $this = jQuery(this);
			const wrapper = $this.parents('.pp-toolkit-content');
			const sVal = '';
			const inputVal = $this.prev('.label-feed-edit').find('.pp-toolkit-url').val();
			const feedBack = wrapper.find('.pp-toolkit-feedback');
			_this.editWrapper.hide();
			_this.ajaxRequest( $this, sVal, inputVal, feedBack );
		});

		this.importWrapper.on('change', '.all-import-checkbox', function() {
			const $this = jQuery(this);
			const episodes = _this.importWrapper.find('.pp-all-episodes input[type="checkbox"]');
			episodes.prop('checked', $this.prop('checked'));
		});

		this.importedWrapper.on('change', '.all-import-checkbox', function() {
			const $this = jQuery(this);
			const episodes = _this.importedWrapper.find('.pp-all-episodes input[type="checkbox"]');
			episodes.prop('checked', $this.prop('checked'));
		});

		this.importWrapper.on('change', 'input[type="checkbox"]', function() {
			if (_this.importWrapper.find('.pp-all-episodes input[type="checkbox"]:checked').length) {
				_this.importWrapper.find('.pp-toolkit-import-options, .pp-import-podcast-btn').fadeIn();
			} else {
				_this.importWrapper.find('.pp-toolkit-import-options, .pp-import-podcast-btn, .pp-import-abort-btn').fadeOut();
			}
		});

		this.importedWrapper.on('change', 'input[type="checkbox"]', function() {
			if (_this.importedWrapper.find('.pp-all-episodes input[type="checkbox"]:checked').length) {
				_this.importedWrapper.find('.pp-imported-podcast-btn').fadeIn();
			} else {
				_this.importedWrapper.find('.pp-imported-podcast-btn').fadeOut();
			}
		});

		toolkit.on( 'change', '.select-pp-feed-method', function() {
			_this.changeFetchMethod(jQuery(this))
		});

		toolkit.on('change', '.podcast-player-ppa-posttype', function() {
			_this.postType(jQuery(this));
		});

		toolkit.on('change', '.podcast-player-ppa-taxonomy', function() {
			_this.taxonomy(jQuery(this));
		});

		toolkit.on('change', '.podcast-player-ppa-bulk-source', function() {
			const source = jQuery(this);
			const val = source.val();
			const wrapper = source.parents('.pp-toolkit-content');
			if ('cf' === val) {
				wrapper.find('.bulk-cf').show();
			} else {
				wrapper.find('.bulk-cf').hide();
			}
		});

		jQuery('#pp-admin--ppa-import-where').on('change', function() {
			const $this = jQuery(this);
			if ('noshow' !== $this.val()) {
				jQuery('.pp-import-player-options').show();
			} else {
				jQuery('.pp-import-player-options').hide();
			}
		});

		jQuery('.pp-import-podcast-btn').on('click', function() {
			_this.importEpisodes();
		});

		jQuery('.pp-import-abort-btn').on('click', function() {
			const feedBack = _this.importWrapper.siblings('.pp-toolkit-feedback');
			_this.response(_this.data.messages.aborting, 'pp-running', feedBack);
			_this.abortAction = true;
		});

		jQuery('.pp-imported-podcast-btn').on('click', function() {
			const $this = jQuery(this);
			_this.deleteImported($this);
		});

		jQuery('#pp-import-tabs-container').on('click', '.pp-import-tabs-item', function() {
			const $this = jQuery(this);
			if ($this.hasClass('active')) return;
			$this.siblings('.active').removeClass('active');
			$this.addClass('active');
			if ($this.hasClass('toolkit-importer')) {
				_this.importedWrapper.hide();
				_this.importWrapper.show();
			} else {
				_this.importWrapper.hide();
				_this.importedWrapper.show();
			}
		});

		new ImageUpload();
	}

	/**
	 * Feed editor Ajax.
	 * 
	 * @since 2.0
	 */
	ajaxRequest(aButton, sVal, inputVal, feedBack) {
		const ajaxConfig = this.getAjaxConfig( aButton, sVal, inputVal );
		this.response(this.data.messages.running, 'pp-running', feedBack);
		if (false === ajaxConfig) {
			this.response(this.data.messages.nourl, 'pp-error', feedBack);
			return;
		} else if ( 'no-source' === ajaxConfig ) {
			let msg = this.data.messages.nosource;
			if ('undefined' === typeof msg) {
				msg = this.data.messages.nourl;
			}
			this.response(this.data.messages.nosource, 'pp-error', feedBack);
			return;
		}

		// Let's get next set of episodes.
		jQuery.ajax( {
			url: this.data.ajaxurl,
			data: ajaxConfig,
			type: 'POST',
			timeout: 60000,
			success: response => {
				const details = JSON.parse( response );
				if (!jQuery.isEmptyObject(details)) {
					if ('undefined' !== typeof details.error) {
						this.response(details.error, 'pp-error', feedBack);
					} else if ('undefined' !== typeof details.data) {
						this.markup(details.data)
						this.response('', false, feedBack);
					} else if ('undefined' !== typeof details.idata) {
						this.imarkup(details.idata)
						this.response('', false, feedBack);
					} else if ('undefined' !== typeof details.message) {
						this.response(details.message, 'pp-success', feedBack);
					}
				}
			},
			error: (jqXHR, textStatus, errorThrown) => {
				this.response(errorThrown, 'pp-error', feedBack);
			}
		} );
	}

	/**
	 * Get args for Ajax request.
	 * 
	 * @since 2.0
	 */
	getAjaxConfig(aButton, sVal, inputVal) {
		if ( aButton.hasClass('pp-feed-sticky') ) {
			const wrapper = aButton.parents('.pp-toolkit-content');
			const tchecked = wrapper.find('.ppa-terms-checklist input[type="checkbox"]:checked');
			let terms = [];
			jQuery.each( tchecked, function() { terms.push(jQuery(this).val()) } );
			const fmethod     = wrapper.find('.select-pp-feed-method').val();
			const ptype       = wrapper.find('.podcast-player-ppa-posttype').val();
			const taxon       = wrapper.find('.podcast-player-ppa-taxonomy').val();
			const bgColor     = wrapper.find('.pp-bg-color').val();
			const accentColor = wrapper.find('.pp-accent-color').val();
			const hShare      = wrapper.find('.pp-social-sharing').is(':checked') ? 'yes' : '';
			const hDown       = wrapper.find('.pp-download').is(':checked') ? 'yes' : '';
			const audio       = wrapper.find('.ppa-audio-url').val();
			const atitle      = wrapper.find('.ppa-audio-title').val();
			const rfrom       = sVal ? 'indexKey' : '';
			const furl        = sVal ? sVal : inputVal;
			const ajaxAction  = 'pp_feed_editor_sticky';

			//Return if feed URL is not available.
			if ('feed' === fmethod && !furl) return false;

			//Return if audio URL is not available.
			if ('link' === fmethod && !audio) return false;

			return {
				action  : ajaxAction,
				security: this.data.security,
				fmethod: fmethod,
				ptype: ptype,
				taxon: taxon,
				terms: terms,
				audio: audio,
				atitle: atitle,
				feedUrl : furl,
				bg: bgColor,
				accent: accentColor,
				hs: hShare,
				hd: hDown,
				rFrom: rfrom,
			}
		}

		if ( aButton.hasClass('pp-remove-sticky') ) {
			const ajaxAction = 'pp_feed_remove_sticky';
			return {
				action  : ajaxAction,
				security: this.data.security,
			};
		}

		if ( aButton.hasClass('pp-add-bulk') ) {
			const wrapper = aButton.parents('.pp-toolkit-content');
			const bgColor     = wrapper.find('.pp-bg-color').val();
			const accentColor = wrapper.find('.pp-accent-color').val();
			const hShare      = wrapper.find('.pp-social-sharing').is(':checked') ? 'yes' : '';
			const hDown       = wrapper.find('.pp-download').is(':checked') ? 'yes' : '';
			const audio       = wrapper.find('.podcast-player-ppa-bulk-source').val();
			const cf          = wrapper.find('.ppa-bulk-cf').val();
			const location    = wrapper.find('.podcast-player-ppa-bulk-where').val();
			const style       = wrapper.find('.podcast-player-ppa-bulk-style').val();
			const isTitle     = wrapper.find('.pp-title').is(':checked') ? 'yes' : '';
			const isFeatured  = wrapper.find('.pp-featured').is(':checked') ? 'yes' : '';
			const ajaxAction = 'pp_add_bulk_players';

			//Return if single audio field is empty.
			if (!audio) return 'no-source';

			return {
				action  : ajaxAction,
				security: this.data.security,
				audio: audio,
				cf: cf,
				bg: bgColor,
				accent: accentColor,
				hs: hShare,
				hd: hDown,
				location: location,
				style: style,
				hT: isTitle,
				hF: isFeatured,
			};
		}

		if ( aButton.hasClass('pp-remove-bulk') ) {
			const ajaxAction = 'pp_remove_bulk_players';
			return {
				action  : ajaxAction,
				security: this.data.security,
			};
		}

		if ( aButton.is('.select-pp-feed-import-index, .feed-url-import-btn') ) {
			const rfrom      = sVal ? 'indexKey' : '';
			const furl       = sVal ? sVal : inputVal;
			const ajaxAction = 'pp_feed_import';
			return {
				action  : ajaxAction,
				security: this.data.security,
				feedUrl : furl,
				rFrom: rfrom,
			};
		}

		if ( aButton.is('.select-pp-feed-edit-index, .feed-url-edit-btn') ) {
			const rfrom      = sVal ? 'indexKey' : '';
			const furl       = sVal ? sVal : inputVal;
			const ajaxAction = 'pp_feed_editor_edit';
			return {
				action  : ajaxAction,
				security: this.data.security,
				feedUrl : furl,
				rFrom: rfrom,
			};
		}

		return false;
	}

	/**
	 * Create feed editor markup.
	 * 
	 * @since 3.5.0
	 * 
	 * @param object data
	 */
	markup(data) {
		const episodes = data.items;
		const podcast = data.fkey;
		const title = data.title ? data.title : '';
		const str = this.data.messages;
		if (!episodes || !podcast) return;
		this.editWrapper.empty().hide();
		const markup = jQuery('<div />' , { class: 'edit-episodes-container' });
		jQuery.each( episodes, function( index, val ){
			const catarr = jQuery.map(val.categories, function(v, i) { return [v]; });
			const catstring = catarr.join(', ');
			const featuredId = val.featured_id ? val.featured_id : '';
			let fMarkup = '';
			if (val.featured && featuredId) {
				fMarkup = `
				<button type="button" class="pp-upload-image-btn"><img class="pp-image" src="${val.featured}"></button>
				<button type="button" class="pp-remove-image-btn">${str.removeFeatured}</button>
				`;
			} else {
				fMarkup = `
				<button type="button" class="pp-upload-image-btn">
					<div class="pp-no-image">
						<span class="dashicon dashicons dashicons-format-image"></span>
						${str.uploadFeatured}
					</div>
				</button>
				`;
			}
			const episode = `
			<div class="pp-episode-wrapper" data-key="${index}">
				<div class="pp-episode-title">
					<span class="dashicons dashicons-edit"></span>
					<span>${val.title}</span>
				</div>
				<div class="pp-episode-form">
					<div class="pp-form-element pp-form-full">
						<label class="pp-form-label">${str.tlabel}</label>
						<input class="pp-edit-title" type="text" value="${val.title}" data-value="${val.title}">
					</div>
					<div class="pp-form-element pp-form-half">
						<label class="pp-form-label">${str.alabel}</label>
						<input class="pp-edit-author" type="text" value="${val.author}" data-value="${val.author}">
					</div>
					<div class="pp-form-element pp-form-half">
						<label class="pp-form-label">${str.slabel}</label>
						<input class="pp-edit-season" type="number" value="${val.season}" data-value="${val.season}">
					</div>
					<div class="pp-form-element pp-form-full">
						<label class="pp-form-label">${str.catlabel}</label>
						<input class="pp-edit-cats" type="text" value="${catstring}" data-value="${catstring}" placeholder="${str.catph}">
					</div>
					<div class="pp-form-element pp-form-half">
						<label class="pp-form-label">${str.feLabel}</label>
						${fMarkup}
						<input class="pp-edit-featured" type="hidden" value="${featuredId}" data-value="${featuredId}">
					</div>
					<div class="pp-save-changes pp-form-full">
						<button class="pp-edit-save button"><span class="pp-edit-save-btn-text">${str.update}</span></button>
					</div>
					<div class="pp-toolkit-feedback pp-form-full">
						<span class="dashicons dashicons-update"></span>
						<span class="dashicons dashicons-yes"></span>
						<span class="dashicons dashicons-no"></span>
						<span class="pp-feedback"></span>
					</div>
				</div>
			</div>
			`;
			markup.append(jQuery(episode));
		});
		if (! markup.is(':empty')) {
			const heading = jQuery('<h4 />', { class: 'pp-podcast-title' }).html(title);
			const dashclose = jQuery('<span />', { class: 'dashicons dashicons-no' });
			const headwrap = jQuery('<div />', { class: 'pp-podcast-head' }).html(heading);
			headwrap.append(dashclose);
			const loadMore = `
			<div class="pp-edit-lmore">
				<button class="pp-edit-load-more button">
					<span class="pp-edit-load-more-text">${str.loadmore}</span>
				</button>
			</div>
			`;
			this.editWrapper.attr('data-podcast', podcast).html(markup);
			const episodes = this.editWrapper.find('.pp-episode-wrapper');
			if (episodes.length > 5) {
				this.editWrapper.append(loadMore);
				episodes.slice(5).hide();
			}
			this.editWrapper.prepend(headwrap).addClass('active').fadeIn();
		}		
	}

	/**
	 * Create feed import markup.
	 * 
	 * @since 5.1.0
	 * 
	 * @param object idata
	 */
	imarkup(idata) {
		const data = idata.data;
		const episodes = data.items;
		const podcast = data.fkey;
		const title = data.title ? data.title : '';
		const str = this.data.messages;
		const tabContainer = jQuery('#pp-import-tabs-container');
		if (!episodes || !podcast) return;
		const markup = jQuery('<div />' , { class: 'episodes-container' });
		const sAll = `
		<div class="pp-episode-wrapper">
			<div class="pp-episode-title">
				<input id="pp-import-select-all-1" class="all-import-checkbox" type="checkbox" />
				<label for="pp-import-select-all-1" class="pp-import-episode-title">${str.all}</label>
			</div>
		</div>
		<div class="pp-all-episodes"></div>
		`;
		markup.append(jQuery(sAll));
		const imarkup = markup.clone();
		imarkup.find('#pp-import-select-all-1').attr('id', 'pp-import-select-all-2').next('label').attr('for', 'pp-import-select-all-2');
		const tobeImported = markup.find('.pp-all-episodes');
		const imported = imarkup.find('.pp-all-episodes');
		const importWrapper = this.importWrapper.find('.pp-toolkit-importer').attr('data-podcast', podcast);
		const importedWrapper = this.importedWrapper.find('.pp-toolkit-imported').attr('data-podcast', podcast);
		jQuery.each( episodes, function( index, val ){
			const eTitle = val.title;
			const id    = val.post_id ? val.post_id : false;
			const eLabel = `
					<div class="pp-episode-wrapper">
						<div class="pp-episode-title">
							<input id="${index}" value="${index}" data-id="${id}" type="checkbox" />
							<label for="${index}" class="pp-import-episode-title">${eTitle}</label>
						</div>
					</div>
				`;
			if (! id) {
				tobeImported.append(jQuery(eLabel));
			} else {
				imported.append(jQuery(eLabel));
			}
		});
		if (! tobeImported.is(':empty')) {
			importWrapper.html(markup).addClass('active');
		}
		if (! imported.is(':empty')) {
			importedWrapper.html(imarkup).addClass('active');
		}
		tabContainer.show();
		this.importDefaultFill(idata.index);
		this.importWrapper.fadeIn();
	}

	/**
	 * Fill up import form with default values.
	 * 
	 * @since 3.5.0
	 *
	 * @param Object feedBack
	 */
	importDefaultFill(fdata) {
		console.log(fdata);
		const oWrap = this.importWrapper.find('.pp-toolkit-import-options');
		if (!fdata) {
			oWrap.find('.pp-import-featured').prop('checked', false);
			oWrap.find('#pp-import-post-types').val('post');
			oWrap.find('#pp-import-post-status').val('publish');
			oWrap.find('#pp-admin--ppa-import-where').val('');
			oWrap.find('.pp-import-player-options').show();
			oWrap.find('#pp-admin--ppa-import-style').val('');
			oWrap.find('.pp-import-bg-color').val('').trigger('change');
			oWrap.find('.pp-import-accent-color').val('').trigger('change');
			oWrap.find('.pp-import-social-sharing').prop('checked', false);
			oWrap.find('.pp-import-download').prop('checked', false);
			oWrap.find('.pp-import-title').prop('checked', false);
			return;
		}
		oWrap.find('.pp-import-featured').prop('checked', fdata.is_get_img);
		oWrap.find('#pp-import-post-types').val(fdata.post_type);
		oWrap.find('#pp-import-post-status').val(fdata.post_status);
		oWrap.find('#pp-import-author').val(fdata.author);
		oWrap.find('#pp-admin--ppa-import-where').val(fdata.location);
		if ('noshow' === fdata.location) {
			oWrap.find('.pp-import-player-options').hide();
		} else {
			oWrap.find('.pp-import-player-options').show();
			oWrap.find('#pp-admin--ppa-import-style').val(fdata.style);
			oWrap.find('.pp-import-bg-color').val(fdata.bgcolor).trigger('change');
			oWrap.find('.pp-import-accent-color').val(fdata.accent).trigger('change');
			oWrap.find('.pp-import-social-sharing').prop('checked', fdata.hide_social);
			oWrap.find('.pp-import-download').prop('checked', fdata.hide_download);
			oWrap.find('.pp-import-title').prop('checked', fdata.hide_title);
		}
	}

	/**
	 * Display request feedback.
	 * 
	 * @since 3.3.0
	 * 
	 * @param string message
	 * @param string type
	 * @param Object feedBack
	 */
	response(message = '', type = false, feedBack) {
		if (! feedBack) {
			feedBack = jQuery('.pp-toolkit-feedback').first();
		}
		if (type) {
			feedBack.removeClass('pp-error pp-success pp-running').addClass(type);
			feedBack.find('.pp-feedback').text(message);
		} else {
			feedBack.removeClass('pp-error pp-success pp-running');
		}
	}

	/**
	 * Save current episode's edited data.
	 * 
	 * @since 3.5.0
	 * 
	 * @param object episode
	 */
	saveEpisodeData(episode) {
		const key = episode.attr('data-key');
		const podcast = this.editWrapper.attr('data-podcast');
		const wrapper = episode.parents('.pp-toolkit-content');
		if (!key || !podcast || !wrapper.length ) return;
		const feedBack = episode.find('.pp-toolkit-feedback');

		const title = episode.find('.pp-edit-title');
		const author = episode.find('.pp-edit-author');
		const season = episode.find('.pp-edit-season');
		const cats = episode.find('.pp-edit-cats');
		const fimg = episode.find('.pp-edit-featured');
		const vtitle = title.val() !== title.attr('data-value') ? title.val() : false;
		const vauthor = author.val() !== author.attr('data-value') ? author.val() : false;
		const vseason = season.val() !== season.attr('data-value') ? season.val() : false;
		const vcats = cats.val() !== cats.attr('data-value') ? cats.val() : false;
		const vfimg = fimg.val() !== fimg.attr('data-value') ? fimg.val() : false;

		if (false === vtitle && false === vauthor && false === vseason && false === vcats && false === vfimg) {
			this.response(this.data.messages.nochange, 'pp-error', feedBack);
			episode.find('.pp-episode-form').removeClass('pp-form-pending-save');
			return;
		} else {
			this.response(this.data.messages.running, 'pp-running', feedBack);
		}

		const data = {
			action  : 'pp_feed_editor_save',
			security: this.data.security,
			podcast : podcast,
			episode : key,
			title   : vtitle,
			author  : vauthor,
			season  : vseason,
			cats    : vcats,
			featured: vfimg,
		}

		// Let's get next set of episodes.
		jQuery.ajax( {
			url: this.data.ajaxurl,
			data: data,
			type: 'POST',
			timeout: 4000,
			success: response => {
				const details = JSON.parse( response );
				if (!jQuery.isEmptyObject(details)) {
					if ('undefined' !== typeof details.error) {
						this.response(details.error, 'pp-error', feedBack);
					} else if ('undefined' !== typeof details.message) {
						this.response(details.message, 'pp-success', feedBack);
					}
				}
			},
			error: (jqXHR, textStatus, errorThrown) => {
				this.response(errorThrown, 'pp-error', feedBack);
			},
			complete: () => {
				episode.find('.pp-episode-form').removeClass('pp-form-pending-save');
			},
		} );
	}

	/**
	 * Start importing episodes from feed to WordPress.
	 * 
	 * @since 5.1.0
	 */
	importEpisodes() {
		const podcast = this.importWrapper.find('.pp-toolkit-importer');
		this.importData = {};
		this.importData.podkey = podcast.attr('data-podcast');
		this.importData.episodes = podcast.find('.pp-all-episodes input[type="checkbox"]:checked');
		this.importData.post = {
			type  : jQuery('#pp-import-post-types').val(),
			status: jQuery('#pp-import-post-status').val(),
			author: jQuery('#pp-import-author').val(),
			featured: this.importWrapper.find('.pp-import-featured').is(':checked'),
		};
		this.importData.player = {
			location: jQuery('#pp-admin--ppa-import-where').val(),
			style: jQuery('#pp-admin--ppa-import-style').val(),
			bgcolor: this.importWrapper.find('.pp-import-bg-color').val(),
			acccolor: this.importWrapper.find('.pp-import-accent-color').val(),
			hideSocial: this.importWrapper.find('.pp-import-social-sharing').is(':checked'),
			hideDownload: this.importWrapper.find('.pp-import-download').is(':checked'),
			hideTitle: this.importWrapper.find('.pp-import-title').is(':checked'),
		}
		this.importRequest(true);
	}

	/**
	 * Send Ajax request for podcast episode import.
	 * 
	 * @since 5.1.0
	 *
	 * @param {bool} isPrimaryCall
	 */
	importRequest(isPrimaryCall) {
		isPrimaryCall = true === isPrimaryCall ? true : false;
		const __this = this;
		const lotSize = 2;
		const feedBack = this.importWrapper.siblings('.pp-toolkit-feedback');
		const eWrapper = this.importWrapper.find('.pp-all-episodes');
		const str = this.data.messages;
		const sAll = `
			<div class="episodes-container">
				<div class="pp-episode-wrapper">
					<div class="pp-episode-title">
						<input id="pp-import-select-all-2" class="all-import-checkbox" type="checkbox" />
						<label for="pp-import-select-all-2" class="pp-import-episode-title">${str.all}</label>
					</div>
				</div>
				<div class="pp-all-episodes"></div>
			</div>
		`;

		if (this.abortAction) {
			jQuery('#pp-import-progress-bar').attr('data-width', 0).removeAttr('data-total');
			jQuery('#pp-import-progress').hide();
			this.response(str.abort, 'pp-error', feedBack);
			this.importWrapper.find('.pp-toolkit-import-options, .pp-import-podcast-btn, .pp-import-abort-btn').hide();
			return;
		}

		let importData = jQuery.extend( {}, this.importData );
		const isImport = (importData.post && importData.post.featured) ? true : false ;

		if (!importData || !importData.episodes || !importData.episodes.length) return;
		const episodes = isImport ? importData.episodes.slice(0, lotSize) : importData.episodes;
		importData.episodes = [];
		jQuery.each(episodes, function() { importData.episodes.push(jQuery(this).val()); });

		const data = {
			action  : 'pp_feed_import_episodes',
			security: this.data.security,
			idata : importData,
			pcall: isPrimaryCall,
		}

		this.response(this.data.messages.running, 'pp-running', feedBack);
		if (isImport) {
			if (isPrimaryCall) {
				jQuery('.pp-import-abort-btn').show();
				jQuery('#pp-import-progress').show();
			}
			const pbar = jQuery('#pp-import-progress-bar');
			const bar = +(pbar.attr('data-width'));
			let total = +(jQuery('#pp-import-progress-bar').attr('data-total'));
			if (! total) {
				total = this.importData.episodes.length;
				pbar.attr('data-total', total);
			}
			const lot = Math.min(lotSize, importData.episodes.length);
			const newBar  = bar + (lot * 100 / total);
			pbar.width(bar + '%').attr('data-width', newBar).text(bar + '%');
		}

		// Let's get next set of episodes.
		jQuery.ajax( {
			url: this.data.ajaxurl,
			data: data,
			type: 'POST',
			timeout: 120000,
			success: response => {
				const details = JSON.parse( response );
				if (!jQuery.isEmptyObject(details)) {
					if ('undefined' !== typeof details.error) {
						this.response(details.error, 'pp-error', feedBack);
					} else if ('undefined' !== typeof details.message && 'undefined' !== typeof details.data) {
						const data = details.data;
						jQuery.each(episodes, function() {
							const $this = jQuery(this);
							const epWrap = $this.closest('.pp-episode-wrapper');
							const key = $this.val();
							const all = __this.importedWrapper.find('.pp-all-episodes');
							$this.attr('data-id', data[key].post_id).prop('checked', false);
							if (all.length) {
								all.prepend(epWrap);
							} else {
								__this.importedWrapper.find('.pp-toolkit-imported').html(sAll).find('.pp-all-episodes').prepend(epWrap);
							}
							if (eWrapper.is(':empty')) {
								eWrapper.closest('.pp-toolkit-importer').html(str.import);
							}
						});
						this.importData.episodes = this.importData.episodes.slice(lotSize);
						if (this.importData.episodes.length) {
							this.importRequest();
						} else {
							jQuery('#pp-import-progress-bar').attr('data-width', 0).removeAttr('data-total');
							jQuery('#pp-import-progress').hide();
							this.response(details.message, 'pp-success', feedBack);
							this.importWrapper.find('.pp-toolkit-import-options, .pp-import-podcast-btn, .pp-import-abort-btn').hide();
						}
					}
				}
			},
			error: (jqXHR, textStatus, errorThrown) => {
				this.response(errorThrown, 'pp-error', feedBack);
			}
		} );
	}

	/**
	 * Start importing episodes from feed to WordPress.
	 * 
	 * @since 5.1.0
	 */
	deleteImported() {
		const __this = this;
		const feedBack = this.importedWrapper.siblings('.pp-toolkit-feedback');
		const eWrapper = this.importedWrapper.find('.pp-all-episodes');
		const podcast = this.importedWrapper.find('.pp-toolkit-imported');
		const podkey = podcast.attr('data-podcast');
		const epiId = [];
		const epiKey = [];
		const episodes = podcast.find('.pp-all-episodes input[type="checkbox"]:checked');
		const str = this.data.messages;
		const sAll = `
		<div class="episodes-container">
			<div class="pp-episode-wrapper">
				<div class="pp-episode-title">
					<input id="pp-import-select-all-2" class="all-import-checkbox" type="checkbox" />
					<label for="pp-import-select-all-2" class="pp-import-episode-title">${str.all}</label>
				</div>
			</div>
			<div class="pp-all-episodes"></div>
		</div>
		`;
		jQuery.each(episodes, function() { epiKey.push(jQuery(this).val()); });
		jQuery.each(episodes, function() { epiId.push(jQuery(this).attr('data-id')); });

		const data = {
			action  : 'pp_feed_delete_imported',
			security: this.data.security,
			idata: { podkey, epiId, epiKey }
		}

		this.response(this.data.messages.running, 'pp-running', feedBack);

		// Let's get next set of episodes.
		jQuery.ajax( {
			url: this.data.ajaxurl,
			data: data,
			type: 'POST',
			timeout: 60000,
			success: response => {
				const details = JSON.parse( response );
				if (!jQuery.isEmptyObject(details)) {
					if ('undefined' !== typeof details.error) {
						this.response(details.error, 'pp-error', feedBack);
					} else if ('undefined' !== typeof details.message) {
						jQuery.each(episodes, function() {
							const $this = jQuery(this);
							const epWrap = $this.closest('.pp-episode-wrapper');
							const all = __this.importWrapper.find('.pp-all-episodes');
							$this.removeAttr('data-id').prop('checked', false);
							if (all.length) {
								all.prepend(epWrap);
							} else {
								__this.importWrapper.find('.pp-toolkit-importer').html(sAll).find('.pp-all-episodes').prepend(epWrap);
							}
							if (eWrapper.is(':empty')) {
								eWrapper.closest('.pp-toolkit-imported').html(str.imported).removeClass('active');
							}
						});
						this.response(details.message, 'pp-success', feedBack);
						this.importedWrapper.find('.pp-imported-podcast-btn').hide();
					}
				}
			},
			error: (jqXHR, textStatus, errorThrown) => {
				this.response(errorThrown, 'pp-error', feedBack);
			}
		} );
	}

	/**
	 * Change Fetch Method.
	 * 
	 * @since 4.3.0
	 * 
	 * @param obj method
	 */
	changeFetchMethod(method) {
		const wrapper = method.closest('.pp-toolkit-content');
		const fetchMethod = method.val();
		const commonHide = [
			'.ppa-terms',
		];
		const feedSpecific = [
			'.pp-fetch-feed',
		];
		const postSpecific = [
			'.pp-fetch-posts',
		];
		const linkSpecific = [
			'.pp-fetch-audio',
		];

		wrapper.find('select.podcast-player-ppa-taxonomy').val('');
		wrapper.find(commonHide.join(',')).hide();

		// Fetch Method Specific actions.
		if ('feed' === fetchMethod) {
			wrapper.find(feedSpecific.join(',')).show();
			wrapper.find(postSpecific.join(',')).hide();
			wrapper.find(linkSpecific.join(',')).hide();
		} else if ('post' === fetchMethod) {
			wrapper.find(feedSpecific.join(',')).hide();
			wrapper.find(postSpecific.join(',')).show();
			wrapper.find(linkSpecific.join(',')).hide();
		} else if ('link' === fetchMethod) {
			wrapper.find(feedSpecific.join(',')).hide();
			wrapper.find(postSpecific.join(',')).hide();
			wrapper.find(linkSpecific.join(',')).show();
		}
	}

	/**
	 * postType.
	 * 
	 * @since 4.3.0
	 * 
	 * @param {Obj} obj
	 */
	postType(obj) {
		const postType = obj.val();
		const wrapper  = obj.closest('.pp-toolkit-content');
		const taxonomy = wrapper.find('.podcast-player-ppa-taxonomy');

		// Hide all taxonomy options.
		taxonomy.find( 'option' ).hide();

		// Now display only selected options.
		taxonomy.find( '.always-visible, .' + postType ).show();
		
		// Empty taxonomy value.
		taxonomy.val('');
	}

	/**
	 * Taxonomy.
	 * 
	 * @since 4.3.0
	 * 
	 * @param {Obj} obj
	 */
	taxonomy(obj) {
		const value   = obj.val();
		const wrapper = obj.closest('.pp-toolkit-content');
		const terms   = wrapper.find('.ppa-terms');

		// Uncheck all terms options and Hide Terms checkbox dropdown.
		terms.find('.ppa-terms-checklist input:checkbox').removeAttr('checked');
		terms.hide();

		// Show terms only for currently selected taxonomy.
		if (value) {
			terms.find( '.ppa-terms-checklist li' ).hide();
			terms.find( '.ppa-terms-checklist .' + value ).show();
			terms.show();
		}
	}
}

export default FeedEditor;
