<?php
/**
 * Podcast player premium.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    Podcast_Player
 */

namespace PP_Pro\Inc;

use PP_Pro\Inc\General\General;
use PP_Pro\Inc\Feed\Feed;
use PP_Pro\Inc\Post\Post;
use PP_Pro\Inc\Audio\Audio;
use PP_Pro\Inc\Misc\Render_Modification;
use PP_Pro\Admin\Options;

/**
 * Podcast player premium.
 *
 * @package    Podcast_Player
 * @author     vedathemes <contact@vedathemes.com>
 */
class Register {

	/**
	 * Register hooked functions.
	 *
	 * @since 3.3.0
	 */
	public static function init() {
		/*
		 * ------------------------------------------------------------------------------
		 * General Section.
		 * ------------------------------------------------------------------------------
		 *
		 * Register hooks to extend general functionality of the free plugin.
		 */

		$general = General::get_instance();

		// Declare that pro version is available.
		self::make_pro_available( $general );

		// Extend player's display styles and relevant classes.
		self::extend_pp_display( $general );

		// Enqueue additional resources.
		self::enqueue_resources( $general );

		// Enqueue resources for elementor editor screen.
		self::elementor_support( $general );

		// Extend podcast player widget.
		self::extend_widget( $general );

		// Extend podcast player block.
		self::extend_block( $general );

		// Extend podcast player shortcode.
		self::extend_shortcode( $general );

		// Add support for additional episode fetch methods.
		self::add_fetch_methods( $general );

		// Display bottom sticky player on page load.
		self::show_sticky_player( $general );
		self::show_bulk_player( $general );

		/*
		 * ------------------------------------------------------------------------------
		 * Podcast Feed Section.
		 * ------------------------------------------------------------------------------
		 *
		 * Extend already available display_from_feed functionality of the free version.
		 */

		$feed = Feed::get_instance();

		// Extend already available features in free version.
		self::extend_feed_features( $feed );

		// Add new feed features.
		self::add_feed_features( $feed );

		// Register REST API routes for feed fetch method.
		self::register_feed_data_hooks( $feed );

		/*
		 * ------------------------------------------------------------------------------
		 * Post Type Section.
		 * ------------------------------------------------------------------------------
		 *
		 * Add support to display podcast player from any post or post types.
		 */

		$post = Post::get_instance();

		// Add support functionality for post types.
		self::add_post_type_support( $post );

		// Make episodes data available to scripts.
		self::post_types_script_data( $post );

		// Support podcast player Ajax functionality scripts.
		self::support_ajax_functionality( $post );

		// Register REST API routes for post types fetch method.
		self::register_post_routes( $post );

		/*
		 * ------------------------------------------------------------------------------
		 * Single Audio Section.
		 * ------------------------------------------------------------------------------
		 *
		 * Add support to display podcast player from a single audio file.
		 */

		$audio = Audio::get_instance();
		self::add_single_audio_support( $audio );

		/*
		 * ------------------------------------------------------------------------------
		 * Templates
		 * ------------------------------------------------------------------------------
		 *
		 * Override free version display functions.
		 */
		$render_mod = Render_Modification::get_instance();
		self::override_display_functions( $render_mod );

		// Register admin options.
		$options = Options::get_instance();
		$options->init();
	}

	/**
	 * Declare that pro version is available.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function make_pro_available( $inst ) {
		add_filter( 'podcast_player_is_premium', '__return_true' );
		add_filter( 'podcast_player_mediaelement_settings', array( $inst, 'mejs_settings' ), 12 );
	}

	/**
	 * Extend player's display styles and relevant classes.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function extend_pp_display( $inst ) {
		add_filter( 'podcast_player_display_styles', array( $inst, 'display_styles' ) );
		add_filter( 'podcast_player_wrapper_classes', array( $inst, 'wrapper_classes' ), 10, 2 );
		add_filter( 'podcast_player_fonts', array( $inst, 'google_fonts' ) );
	}

	/**
	 * Enqueue additional resources.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function enqueue_resources( $inst ) {

		// Front-end.
		add_action( 'wp_footer', array( $inst, 'enqueue_styles' ), 12 );
		add_action( 'wp_footer', array( $inst, 'enqueue_fonts_styles' ) );

		// Back-end.
		add_action( 'admin_enqueue_scripts', array( $inst, 'enqueue_styles' ), 12 );
		add_action( 'admin_footer', array( $inst, 'enqueue_fonts_styles' ) );
		add_action( 'enqueue_block_editor_assets', array( $inst, 'enqueue_editor_assets' ) );
	}

	/**
	 * Add Elementor preview screen support.
	 *
	 * @since 3.3.0
	 *
	 * @param object $instance PP front loader instance.
	 */
	public static function elementor_support( $instance ) {
		if (
			in_array(
				'elementor/elementor.php',
				apply_filters( 'active_plugins', get_option( 'active_plugins' ) ),
				true
			)
		) {
			add_action(
				'elementor/preview/enqueue_scripts',
				array( $instance, 'enqueue_editor_assets' )
			);
		}
	}

	/**
	 * Extend podcast player widget.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function extend_widget( $inst ) {
		add_filter( 'podcast_player_widget_options', array( $inst, 'widget_options' ), 10, 3 );
		add_filter( 'podcast_player_widget_update', array( $inst, 'widget_update' ), 10, 3 );
		add_filter( 'podcast_player_widget_display', array( $inst, 'widget_display' ), 10, 2 );
	}

	/**
	 * Extend podcast player block.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function extend_block( $inst ) {
		add_filter( 'podcast_player_block_display', array( $inst, 'block_display' ), 10, 2 );
	}

	/**
	 * Extend podcast player shortcode.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function extend_shortcode( $inst ) {
		add_filter( 'podcast_player_shcode_display', array( $inst, 'shcode_display' ), 10, 2 );
	}

	/**
	 * Add support for additional episode fetch methods.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function add_fetch_methods( $inst ) {
		add_filter( 'podcast_player_fetch_method_class', array( $inst, 'fetch_methods' ) );
	}

	/**
	 * Show sticky player at the bottom of the page.
	 *
	 * @since 4.0.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function show_sticky_player( $inst ) {
		add_filter( 'wp_footer', array( $inst, 'sticky_player' ), 0 );
	}

	/**
	 * Show bulk player at the bottom of the page.
	 *
	 * @since 4.3.0
	 *
	 *  @param object $inst PP Pro general instance.
	 */
	public static function show_bulk_player( $inst ) {
		add_filter( 'the_content', array( $inst, 'bulk_player' ), 0 );
		add_filter( 'the_content', array( $inst, 'import_player' ), 0 );
	}

	/**
	 * Extend already available features in free version.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro feed instance.
	 */
	public static function extend_feed_features( $inst ) {
		add_filter( 'podcast_player_feed_mods', array( $inst, 'add_mod_args' ), 10, 2 );
		add_filter( 'podcast_player_wrapper_classes', array( $inst, 'wrapclass' ), 10, 3 );
		add_filter( 'podcast_player_instance_data', array( $inst, 'script_data' ), 10, 2 );
		add_filter( 'podcast_player_ajax_args', array( $inst, 'get_ajax_args' ), 10, 2 );
	}

	/**
	 * Add new feed features.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro feed instance.
	 */
	public static function add_feed_features( $inst ) {
		add_filter( 'podcast_player_episode_filters', array( $inst, 'episode_filter' ), 10, 2 );
		add_filter( 'podcast_player_search_results', array( $inst, 'deepsearch' ), 10, 3 );
		add_action( 'wp_head', array( $inst, 'sharing_tags' ), 999999 );
	}

	/**
	 * Register data fetching hooks for feed fetch method.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro feed instance.
	 */
	public static function register_feed_data_hooks( $inst ) {

		// Fetch requried feed data in editor blocks JavaScripts.
		add_action( 'rest_api_init', array( $inst, 'register_routes' ) );

		// Fetch requried feed data in widget JavaScripts.
		add_action( 'wp_ajax_pp_feed_data_list', array( $inst, 'feed_data' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_data_list', array( $inst, 'feed_data' ) );
	}

	/**
	 * Add support functionality for post types.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro post instance.
	 */
	public static function add_post_type_support( $inst ) {
		add_filter( 'posts_where', array( $inst, 'modify_query_where' ), 10, 2 );
		add_filter( 'podcast_player_posts_elist', array( $inst, 'get_episodes_list' ) );
	}

	/**
	 * Make episodes data available to scripts.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro post instance.
	 */
	public static function post_types_script_data( $inst ) {
		add_filter( 'podcast_player_script_data', array( $inst, 'scripts_data' ) );
	}

	/**
	 * Support podcast player Ajax functionality scripts.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro post instance.
	 */
	public static function support_ajax_functionality( $inst ) {
		add_action( 'wp_ajax_pp_fetch_posts', array( $inst, 'fetch_episodes' ) );
		add_action( 'wp_ajax_nopriv_pp_fetch_posts', array( $inst, 'fetch_episodes' ) );
		add_action( 'wp_ajax_pp_search_posts', array( $inst, 'search_episodes' ) );
		add_action( 'wp_ajax_nopriv_pp_search_posts', array( $inst, 'search_episodes' ) );
		add_action( 'wp_ajax_pp_post_episodes_list', array( $inst, 'episodes_list' ) );
		add_action( 'wp_ajax_nopriv_pp_post_episodes_list', array( $inst, 'episodes_list' ) );
	}

	/**
	 * Support podcast player Ajax functionality scripts.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro post instance.
	 */
	public static function register_post_routes( $inst ) {
		add_action( 'rest_api_init', array( $inst, 'register_routes' ) );
	}

	/**
	 * Add support to display podcast player from a single audio file.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro audio instance.
	 */
	public static function add_single_audio_support( $inst ) {
		add_filter( 'podcast_player_script_data', array( $inst, 'scripts_data' ) );
		add_action( 'wp_ajax_pp_fetch_media_url', array( $inst, 'fetch_media_url' ) );
		add_action( 'wp_ajax_nopriv_pp_fetch_media_url', array( $inst, 'fetch_media_url' ) );
	}

	/**
	 * Override free version display functions.
	 *
	 * @since 3.3.0
	 *
	 *  @param object $inst PP Pro templates instance.
	 */
	public static function override_display_functions( $inst ) {

		// Override episodes list entry markup.
		add_filter( 'pp_render_tree_episode_list_wrapper', array( $inst, 'episodes_list_wrapper' ), 10, 2 );
		add_filter( 'pp_render_tree_pp_podcast_single', array( $inst, 'pp_podcast_single' ), 10, 2 );
		add_filter( 'podcast_player_dynamic_css', array( $inst, 'dynamic_css' ), 10, 2 );
		add_filter( 'podcast_player_wrapper_classes', array( $inst, 'wrapper_classes' ), 10, 2 );
	}
}
