<?php
/**
 * Podcast player display class.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    Podcast_Player
 */

namespace PP_Pro\Inc\Feed;

use Podcast_Player\Helper\Functions\Getters as Get_Fn;
use Podcast_Player\Helper\Functions\Validation as Validation_Fn;
use Podcast_Player\Helper\Functions\Utility as Utility_Fn;

/**
 * Display podcast player instance.
 *
 * @package    Podcast_Player
 * @author     vedathemes <contact@vedathemes.com>
 */
class Feed {

	/**
	 * Holds the instance of this class.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    object
	 */
	private static $instance = null;

	/**
	 * Holds shared podcast key.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    string
	 */
	private $ppplayer = '';

	/**
	 * Holds shared podcast key.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    string
	 */
	private $ppepisode = '';

	/**
	 * Holds sharedby text.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    string
	 */
	private $sharedby = '';

	/**
	 * Class cannot be instantiated directly.
	 *
	 * @since  1.0.0
	 */
	private function __construct() {
		/*
		 * ------------------------------------------------------------------------------
		 * Shared Podcast Episode.
		 * ------------------------------------------------------------------------------
		 *
		 * If a user/bot arrived on the webpage using podcast player episode share link,
		 * let's get shared podcast and episode's unique key.
		 */
		$this->ppplayer  = isset( $_GET['ppplayer'] ) ? sanitize_text_field( wp_unslash( $_GET['ppplayer'] ) ) : '';
		$this->ppepisode = isset( $_GET['ppepisode'] ) ? sanitize_text_field( wp_unslash( $_GET['ppepisode'] ) ) : '';
		$this->sharedby  = isset( $_GET['sharedby'] ) ? sanitize_text_field( wp_unslash( $_GET['sharedby'] ) ) : '';
	}

	/**
	 * Rearrange feed items based on url query string.
	 *
	 * @since 2.6.0
	 *
	 * @param array $mods Feed episode filter args.
	 * @param array $args Podcast display args.
	 */
	public function add_mod_args( $mods, $args ) {
		$feed_url = isset( $args['url'] ) ? $args['url'] : '';
		$feed_url = wp_strip_all_tags( $args['url'] );
		if ( ! Validation_Fn::is_valid_url( $feed_url ) ) {
			return;
		}

		// Item to be fixed to top.
		$feed_key = md5( $feed_url );
		if ( $feed_key === $this->ppplayer ) {
			$mods['fixed'] = $this->ppepisode;
		}

		return array_merge(
			$mods,
			array(
				'slist'    => array_filter( $args['slist'] ),
				'elist'    => array_filter( $args['elist'] ),
				'catlist'  => array_filter( $args['catlist'] ),
				'edisplay' => isset( $args['edisplay'] ) && 'hide' === $args['edisplay'] ? false : true,
				'episodes' => $args['episodes'],
			)
		);
	}

	/**
	 * Display Podcast wrapper classes.
	 *
	 * @since 2.6.0
	 *
	 * @param array $classes Podcast player wrapper classes.
	 * @param array $args    Podcast display instance settings.
	 */
	public function wrapclass( $classes, $args ) {
		$url = isset( $args['url'] ) ? $args['url'] : '';
		$key = Get_Fn::get_url_key( $url );
		if ( $key && $key === $this->ppplayer ) {
			$classes[] = 'pp-social-shared';
		}
		return $classes;
	}

	/**
	 * Modify podcast instance data available to front-end scripts.
	 *
	 * @since 3.3.0
	 *
	 * @param string $script_data Podcast data to be sent to front-end script.
	 * @param array  $args        Podcast display args.
	 */
	public function script_data( $script_data, $args ) {
		$load_args                        = $script_data['load_info']['args'];
		$script_data['load_info']['args'] = array_merge(
			$load_args,
			array(
				'episodes' => esc_html( $args['episodes'] ),
				'slist'    => array_map( 'absint', $args['slist'] ),
				'elist'    => array_map( 'esc_html', $args['elist'] ),
				'catlist'  => array_map( 'esc_html', $args['catlist'] ),
				'gridcol'  => absint( $args['grid-columns'] ),
			)
		);

		// Add data specific to custom audio message.
		$amsg = isset( $args['audio-msg'] ) ? $args['audio-msg'] : '';
		if ( $amsg && 'audio' === Get_Fn::get_media_type( $amsg ) ) {
			$script_data['rdata'] = array_merge(
				$script_data['rdata'],
				array(
					'audiomsg' => esc_url( $args['audio-msg'] ),
					'playfreq' => absint( $args['play-freq'] ),
					'msgstart' => esc_html( $args['msg-start'] ),
					'msgtime'  => array_map( 'absint', $args['msg-time'] ),
					'msgtext'  => esc_html( $args['msg-text'] ),
				)
			);
		}
		return $script_data;
	}

	/**
	 * Get and sanitize pro specific ajax args.
	 *
	 * @since 3.3.0
	 *
	 * @param array $args       Final snaitized args list.
	 * @param array $query_args Received args from ajax query.
	 */
	public function get_ajax_args( $args, $query_args ) {
		$elist   = isset( $query_args['elist'] ) ? array_map( 'sanitize_text_field', $query_args['elist'] ) : array();
		$slist   = isset( $query_args['slist'] ) ? array_map( 'absint', $query_args['slist'] ) : array();
		$catlist = isset( $query_args['catlist'] ) ? array_map( 'sanitize_text_field', $query_args['catlist'] ) : array();

		$args['filters'] = array(
			'episodes'     => sanitize_text_field( $query_args['episodes'] ),
			'slist'        => $slist,
			'elist'        => $elist,
			'catlist'      => $catlist,
			'grid-columns' => absint( $query_args['gridcol'] ),
		);
		return $args;
	}

	/**
	 * Podcast player episode filters.
	 *
	 * @since 3.0.0
	 *
	 * @param array $items Feed items to be filtered.
	 * @param array $mods  Podcast list modification args.
	 */
	public function episode_filter( $items, $mods ) {
		$seasons    = isset( $mods['slist'] ) ? $mods['slist'] : array();
		$episodes   = isset( $mods['episodes'] ) ? $mods['episodes'] : false;
		$elist      = isset( $mods['elist'] ) ? array_filter( $mods['elist'] ) : array();
		$categories = isset( $mods['catlist'] ) ? array_filter( $mods['catlist'] ) : array();
		$edisplay   = isset( $mods['edisplay'] ) ? $mods['edisplay'] : true;

		// Return if no filters exist.
		if ( ! $episodes && empty( $elist ) && empty( $seasons ) && empty( $categories ) ) {
			return $items;
		}

		// Apply seasons filters.
		$seasons = array_map( 'absint', $seasons );
		$seasons = array_filter( $seasons );
		if ( ! empty( $seasons ) ) {
			foreach ( $items as $key => $item ) {
				if ( isset( $item['season'] ) && ! in_array( absint( $item['season'] ), $seasons, true ) ) {
					unset( $items[ $key ] );
				}
			}
		}

		// Apply categories filters.
		if ( ! empty( $categories ) ) {
			foreach ( $items as $key => $item ) {
				if ( ! isset( $item['categories'] ) || ! $item['categories'] ) {
					unset( $items[ $key ] );
				} else {
					$common = array_intersect( array_keys( $item['categories'] ), $categories );
					if ( empty( $common ) ) {
						unset( $items[ $key ] );
					}
				}
			}
		}

		if ( $episodes ) {
			// Apply Episode filters.
			$episodes = explode( ',', $episodes );
			$episodes = array_map( 'trim', $episodes );
			$episodes = array_filter( $episodes );
			if ( ! empty( $episodes ) ) {
				foreach ( $items as $key => $item ) {
					if ( isset( $item['episode'] ) && ! in_array( $item['episode'], $episodes, true ) ) {
						unset( $items[ $key ] );
					}
				}
			}
		}

		// Apply elist filter.
		if ( ! empty( $elist ) ) {
			$new_items = array();
			if ( $edisplay ) {
				foreach ( $items as $key => $item ) {
					if ( in_array( $key, $elist, true ) ) {
						$new_items[ $key ] = $item;
					}
				}
			} else {
				foreach ( $items as $key => $item ) {
					if ( ! in_array( $key, $elist, true ) ) {
						$new_items[ $key ] = $item;
					}
				}
			}
			$items = $new_items;
		}

		return $items;
	}

	/**
	 * Perform approximate and deep search for podcast episodes.
	 *
	 * @since 3.1.0
	 *
	 * @param array $results     Search results array.
	 * @param str   $feed_items  Relavant feed items.
	 * @param str   $search_term Search Term.
	 *
	 * @return array
	 */
	public function deepsearch( $results, $feed_items, $search_term ) {
		// No need to go deeper if we have minimum required results.
		$min_req_results = apply_filters( 'podcast_player_minreq_search_results', 5 );

		$count = count( $results );
		// Look for similar terms in episode title. (Loose match).
		if ( $min_req_results && $min_req_results > $count ) {
			foreach ( $feed_items as $key => $item ) {
				$item_title = strtolower( $item['title'] );
				$title_arr  = explode( ' ', $item_title );
				$term_arr   = explode( ' ', $search_term );
				$tlen       = 0;
				$found      = array();
				$final      = 0;
				foreach ( $term_arr as $term ) {
					$len   = strlen( $term );
					$tlen += $len;
					$match = 0;
					foreach ( $title_arr as $str ) {
						similar_text( $str, $term, $per );
						if ( 80 <= $per ) {
							$match = $per;
							break;
						}
					}
					$found[] = array( $len, $match );
				}
				foreach ( $found as $m ) {
					$final += ( $m[0] / $tlen ) * $m[1];
				}
				if ( 70 <= $final ) {
					$results[ $key ] = $feed_items[ $key ];
					unset( $feed_items[ $key ] );
				}
			}
		}

		$count = count( $results );
		// Finally search the term in episode description.
		if ( $min_req_results && $min_req_results > $count ) {
			foreach ( $feed_items as $key => $item ) {
				$item_desc = strtolower( wp_strip_all_tags( $item['description'] ) );
				if ( false !== strpos( $item_desc, $search_term ) ) {
					$results[ $key ] = $feed_items[ $key ];
					unset( $feed_items[ $key ] );
				}
			}
		}

		return $results;
	}

	/**
	 * Create Open Meta and other meta Tags for social media.
	 *
	 * @since 2.6.0
	 */
	public function sharing_tags() {
		if ( $this->ppplayer && $this->ppepisode ) {
			$pp       = $this->ppplayer;
			$epi      = $this->ppepisode;
			$data_key = 'pp_feed_data_' . $pp;
			$html     = '';

			$data = get_option( $data_key );
			if ( $data ) {
				$items = $data['items'];
				if ( isset( $items[ $epi ] ) ) {
					global $wp;
					$episode = $items[ $epi ];
					$title   = $episode['title'];
					$audio   = $episode['src'];
					$image   = $episode['featured'] ? $episode['featured'] : $data['image'];
					$text    = wp_strip_all_tags( $episode['description'] );
					$text    = esc_html( wp_trim_words( $text, 30, ' &hellip; ' ) );
					$url     = add_query_arg(
						array(
							'ppplayer'  => $pp,
							'ppepisode' => $epi,
						),
						esc_url( home_url( add_query_arg( array(), $wp->request ) ) )
					);

					$html .= ' <!-- Open Graph Tags By Podcast Player -->';
					$html .= ' <meta property="og:site_name" content="' . get_bloginfo( 'name', 'display' ) . '"/>';
					$html .= ' <meta property="og:type" content="article"/>';
					$html .= ' <meta property="og:title" content="' . $title . '"/>';

					$imgdata = false;
					if ( isset( $episode['featured_id'] ) && $episode['featured_id'] ) {
						$imgdata = wp_get_attachment_image_src( $episode['featured_id'], 'large' );
						if ( $imgdata ) {
							list( $image, $width, $height ) = $imgdata;
							// Image open graph tag.
							$html .= $image ? ' <meta property="og:image" content="' . esc_url( $image ) . '"/>' : '';
							$html .= $image ? ' <meta property="og:image:width" content="' . absint( $width ) . '"/>' : '';
							$html .= $image ? ' <meta property="og:image:height" content="' . absint( $height ) . '"/>' : '';
						}
					} else {
						$html .= $image ? ' <meta property="og:image" content="' . esc_url( $image ) . '"/>' : '';
					}

					$html .= ' <meta property="og:url" content="' . $url . '"/>';
					$html .= ' <meta property="og:description" content="' . $text . '"/>';
					if ( 'audio' === $episode['mediatype'] ) {
						$html .= ' <meta property="og:audio" content="' . $audio . '"/>';
						$html .= ' <meta property="og:audio:type" content="audio/mp3"/>';
					}

					if ( $imgdata && absint( $width ) > absint( $height ) ) {
						$html .= $image ? ' <meta property="twitter:card" content="summary_large_image"/>' : '';
						$html .= ' <meta property="twitter:title" content="' . $title . '"/>';
						$html .= $text ? ' <meta property="twitter:description" content="' . $text . '"/>' : '';
						$html .= $image ? ' <meta property="twitter:image" content="' . esc_url( $image ) . '"/>' : '';
					}
				}
			}

			echo $html; // 	phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} elseif ( 'pplayer' === $this->sharedby ) {
			if ( ! is_singular() ) {
				return;
			}

			// Return if open graph tags can be added by other plugin.
			if ( defined( 'WPSEO_VERSION' ) || defined( 'WEBDADOS_FB_VERSION' ) ) {
				return;
			}

			$post_id = get_queried_object_id();

			$image  = '';
			$width  = '';
			$height = '';
			if ( has_post_thumbnail( $post_id ) ) {
				$att_id = get_post_thumbnail_id( $post_id );
				$image  = wp_get_attachment_image_src( $att_id, array( 1200, 630 ) );
				$width  = $image ? $image[1] : '';
				$height = $image ? $image[2] : '';
				$image  = $image ? $image[0] : '';
			}

			global $wp;
			$title   = get_the_title( $post_id );
			$excerpt = get_the_excerpt( $post_id );
			$excerpt = esc_html( wp_trim_words( $excerpt, 30, ' &hellip; ' ) );
			$url     = add_query_arg(
				array(
					'sharedby' => 'pplayer',
				),
				esc_url( home_url( add_query_arg( array(), $wp->request ) ) )
			);

			$html  = '';
			$html .= ' <!-- Open Graph Tags By Podcast Player -->';
			$html .= ' <meta property="og:type" content="article"/>';
			$html .= ' <meta property="og:title" content="' . $title . '"/>';
			$html .= ' <meta property="og:url" content="' . $url . '"/>';
			$html .= $excerpt ? ' <meta property="og:description" content="' . $excerpt . '"/>' : '';
			$html .= $image ? ' <meta property="og:image" content="' . esc_url( $image ) . '"/>' : '';
			$html .= $image ? ' <meta property="og:image:width" content="' . absint( $width ) . '"/>' : '';
			$html .= $image ? ' <meta property="og:image:height" content="' . absint( $height ) . '"/>' : '';
			$html .= $image ? ' <meta property="twitter:card" content="summary_large_image"/>' : '';
			$html .= ' <meta property="twitter:title" content="' . $title . '"/>';
			$html .= $excerpt ? ' <meta property="twitter:description" content="' . $excerpt . '"/>' : '';
			$html .= $image ? ' <meta property="twitter:image" content="' . esc_url( $image ) . '"/>' : '';
			echo $html; // 	phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	/**
	 * Create REST API endpoints to get all pages list.
	 *
	 * @since 1.8.0
	 */
	public function register_routes() {
		register_rest_route(
			'podcastplayer/v1',
			'/fElist',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'fepisodes_list' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'feedURL'    => array(
						'description' => esc_html__( 'Feed URL', 'podcast-player' ),
						'type'        => 'string',
					),
					'seasons'    => array(
						'description' => esc_html__( 'Seasons', 'podcast-player' ),
						'type'        => 'string',
					),
					'categories' => array(
						'description' => esc_html__( 'Categories', 'podcast-player' ),
						'type'        => 'string',
					),
				),
			)
		);
		register_rest_route(
			'podcastplayer/v1',
			'/fSlist',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'fseasons_list' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'feedURL' => array(
						'description' => esc_html__( 'Feed URL', 'podcast-player' ),
						'type'        => 'string',
					),
				),
			)
		);
		register_rest_route(
			'podcastplayer/v1',
			'/fcatlist',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'fcategories_list' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'feedURL' => array(
						'description' => esc_html__( 'Feed URL', 'podcast-player' ),
						'type'        => 'string',
					),
				),
			)
		);
	}

	/**
	 * Fetch podcast episodes list for WP REST route.
	 *
	 * @param WP_REST_Request $request Request data.
	 *
	 * @since 1.0.0
	 */
	public function fepisodes_list( $request ) {
		$seasons = isset( $request['seasons'] ) ? sanitize_text_field( $request['seasons'] ) : '';
		$cats    = isset( $request['categories'] ) ? sanitize_text_field( $request['categories'] ) : '';

		// Fetch required data from podcast feed.
		$url    = isset( $request['feedURL'] ) ? $request['feedURL'] : '';
		$mods   = array(
			'slist'   => array_filter( array_map( 'absint', explode( ',', $seasons ) ) ),
			'catlist' => array_filter( array_map( 'trim', explode( ',', $cats ) ) ),
			'sortby'  => 'sort_date_desc',
		);
		$fields = array( 'title' );
		$feed   = Get_Fn::get_feed_data( $url, $mods, $fields );
		if ( is_wp_error( $feed ) ) {
			return array('' => $url);
		}

		// Properly format feed items as 'item_key => item_title'.
		$items = array_map(
			function( $arr ) {
				return $arr['title'];
			},
			$feed['items']
		);

		$options = array( '' => esc_html__( 'Show All Episodes', 'podcast-player' ) );
		return array_merge( $options, $items );
	}

	/**
	 * Fetch podcast seasons list for WP REST route.
	 *
	 * @param WP_REST_Request $request Request data.
	 *
	 * @since 1.0.0
	 */
	public function fseasons_list( $request ) {
		$feed_url = isset( $request['feedURL'] ) ? esc_url_raw( $request['feedURL'] ) : '';
		$feed     = Get_Fn::get_feed_data( $feed_url );
		if ( is_wp_error( $feed ) || empty( $feed['seasons'] ) ) {
			return array();
		}

		$seasons = array();
		foreach ( $feed['seasons'] as $season ) {
			$season                   = esc_html( $season );
			$seasons[ '0' . $season ] = esc_html__( 'Season', 'podcast-player' ) . '-' . $season;
		}

		if ( ! empty( $seasons ) ) {
			$default = array( '' => esc_html__( 'Show All Seasons', 'podcast-player' ) );
			return array_merge( $default, $seasons );
		}

		return array();
	}

	/**
	 * Fetch podcast seasons list for WP REST route.
	 *
	 * @param WP_REST_Request $request Request data.
	 *
	 * @since 1.0.0
	 */
	public function fcategories_list( $request ) {
		$feed_url = isset( $request['feedURL'] ) ? esc_url_raw( $request['feedURL'] ) : '';
		$feed     = Get_Fn::get_feed_data( $feed_url );
		if ( is_wp_error( $feed ) || empty( $feed['categories'] ) ) {
			return array();
		}
		$default = array( '' => esc_html__( 'Show All Categories', 'podcast-player' ) );
		return array_merge( $default, $feed['categories'] );
	}

	/**
	 * Fetch podcast episodes list for admin Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function feed_data() {
		check_ajax_referer( 'podcast-player-admin-ajax-nonce', 'security' );

		// Get variable values from Ajax request.
		$feed_url = isset( $_POST['feedUrl'] ) ? esc_url_raw( wp_unslash( $_POST['feedUrl'] ) ) : '';
		$get_all  = isset( $_POST['getAll'] ) ? sanitize_text_field( wp_unslash( $_POST['getAll'] ) ) : 'true';
		$seasons  = isset( $_POST['seasons'] ) ? array_map( 'absint', wp_unslash( $_POST['seasons'] ) ) : array();
		$cats     = isset( $_POST['categories'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['categories'] ) ) : array();

		// Fetch required podcast data.
		$mods   = array(
			'slist'   => array_filter( $seasons ),
			'catlist' => array_filter( $cats ),
			'sortby'  => 'sort_date_desc',
		);
		$fields = array( 'title' );

		// Fetch required data.
		$feed = Get_Fn::get_feed_data( $feed_url, $mods, $fields );
		if ( is_wp_error( $feed ) ) {
			echo wp_json_encode( array( 'error' => esc_html( $feed->get_error_message() ) ) );
			wp_die();
		}
		$flist = Utility_Fn::multi_array_columns( array( 'items', 'seasons', 'categories' ), $feed );

		// Properly format feed items as 'item_key => item_title'.
		$flist['items'] = array_map(
			function( $arr ) {
				return $arr['title'];
			},
			$flist['items']
		);

		// Properly format seasons list'.
		$fseasons = array();
		array_walk(
			$flist['seasons'],
			function( $val, $key ) use ( &$fseasons ) {
				$fseasons[ '0' . $val ] = esc_html__( 'Season', 'podcast-player' ) . '-' . $val;
			}
		);
		$flist['seasons'] = $fseasons;

		// Remove empty array elements.
		$flist = array_filter( $flist );

		if ( 'true' === $get_all ) {
			echo wp_json_encode( $flist );
			wp_die();
		}
		echo wp_json_encode( array( 'items' => $flist['items'] ) );
		wp_die();
	}

	/**
	 * Returns the instance of this class.
	 *
	 * @since  1.0.0
	 *
	 * @return object Instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}


}
