<?php
/**
 * Podcast player premium.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    Podcast_Player
 */

namespace PP_Pro\Inc\General;

use Podcast_Player\Frontend\Inc\Display;
use Podcast_Player\Helper\Functions\Getters as Get_Fn;
use Podcast_Player\Helper\Functions\Validation as Validation_Fn;
use PP_Pro\Helper\Functions\Getters as PP_Get_Fn;
use PP_Pro\Helper\Functions\Gfonts as Gfonts;
use Podcast_Player\Frontend\Inc\Instance_Counter;

/**
 * Podcast player premium.
 *
 * @package    Podcast_Player
 * @author     vedathemes <contact@vedathemes.com>
 */
class General {

	/**
	 * Holds the instance of this class.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    object
	 */
	private static $instance = null;

	/**
	 * Holds list of google fonts to be enqueued.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    object
	 */
	private $gfonts = array();

	/**
	 * Holds list of all google fonts.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    object
	 */
	private $gfonts_list = array();

	/**
	 * Class cannot be instantiated directly.
	 *
	 * @since  1.0.0
	 */
	private function __construct() {}

	/**
	 * Media Element player settings.
	 *
	 * @param array $settings Array of mejs settings.
	 * @since 1.0.0
	 */
	public function mejs_settings( $settings ) {
		$settings['isPremium'] = true;
		$settings['stSup']     = Get_Fn::get_style_supported();
		$settings['isSticky']  = 'yes' === Get_Fn::get_plugin_option( 'sticky_on_scroll' ) ? true : false;
		return $settings;
	}

	/**
	 * Podcast player premium styles.
	 *
	 * @param array $styles Podcast player styles.
	 *
	 * @since  1.0.0
	 */
	public function display_styles( $styles = array() ) {

		return array_merge(
			$styles,
			array(
				array(
					'style_id' => 'lv1',
					'label'    => esc_html__( 'List - Normal', 'pp-pro' ),
					'support'  => array( 'excerpt', 'thumbnail', 'bgcolor', 'txtcolor' ),
				),
				array(
					'style_id' => 'lv2',
					'label'    => esc_html__( 'List - Small', 'pp-pro' ),
					'support'  => array( 'thumbnail', 'bgcolor', 'txtcolor' ),
				),
				array(
					'style_id' => 'lv3',
					'label'    => esc_html__( 'List - Minimal', 'pp-pro' ),
					'support'  => array( 'playbtn', 'bgcolor', 'txtcolor' ),
				),
				array(
					'style_id' => 'lv4',
					'label'    => esc_html__( 'List - Individual', 'pp-pro' ),
					'support'  => array( 'playbtn', 'bgcolor' ),
				),
				array(
					'style_id' => 'gv1',
					'label'    => esc_html__( 'Grid - Normal', 'pp-pro' ),
					'support'  => array( 'excerpt', 'thumbnail', 'grid', 'bgcolor', 'txtcolor' ),
				),
				array(
					'style_id' => 'gv2',
					'label'    => esc_html__( 'Grid - Cards', 'pp-pro' ),
					'support'  => array( 'thumbnail', 'grid', 'bgcolor' ),
				),
			)
		);
	}

	/**
	 * Podcast player wrapper classes.
	 *
	 * @param array $classes Podcast player wrapper classes.
	 * @param array $args    Podcast display instance settings.
	 *
	 * @since  1.0.0
	 */
	public function wrapper_classes( $classes, $args ) {
		$style = $args['display-style'];
		if ( in_array( $style, array( 'lv1', 'lv2', 'lv3', 'lv4', 'gv1', 'gv2' ), true ) ) {
			$classes[] = 'postview';
		}

		if ( Validation_Fn::is_style_support( $style, 'grid' ) && $args['grid-columns'] ) {
			$classes[] = 'columns-' . absint( $args['grid-columns'] );
		}

		$classes[] = $args['aspect-ratio'] ? $args['aspect-ratio'] : 'no-crop';
		$classes[] = $args['aspect-ratio'] ? $args['crop-method'] : '';

		return array_filter( $classes );
	}

	/**
	 * Get list of google fonts.
	 *
	 * @since 3.3.0
	 *
	 * @param array $fonts Fonts array.
	 */
	public function google_fonts( $fonts ) {
		$gfonts = Gfonts::get_list();
		return array_merge( $fonts, $gfonts );
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_styles() {

		// Return if podcast player is not available on the page.
		if ( ! $this->has_podcast_player() ) {
			return;
		}

		wp_enqueue_style(
			'pppremium',
			PP_PRO_URL . 'css/podcast-player-premium.css',
			array(),
			PP_PRO_VERSION,
			'all'
		);
		wp_style_add_data( 'pppremium', 'rtl', 'replace' );
	}

	/**
	 * Register the stylesheets for the block editor side of the site.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_editor_assets() {
		wp_enqueue_style(
			'pppremium',
			PP_PRO_URL . 'css/podcast-player-premium.css',
			array(),
			PP_PRO_VERSION,
			'all'
		);
		wp_style_add_data( 'pppremium', 'rtl', 'replace' );
	}

	/**
	 * Enqueue google fonts style in footer.
	 *
	 * @since  2.6.0
	 */
	public function enqueue_fonts_styles() {
		if ( ! empty( $this->gfonts ) ) {
			$fonts     = array_unique( $this->gfonts );
			$fonts_url = esc_url_raw(
				add_query_arg(
					array( 'family' => rawurlencode( implode( '|', $fonts ) ) ),
					'https://fonts.googleapis.com/css'
				)
			);
			wp_enqueue_style(
				'podcast-player-fonts',
				esc_url( $fonts_url ),
				array(),
				PP_PRO_VERSION
			);
		}
	}

	/**
	 * Podcast player premium widget options.
	 *
	 * @param array $options Podcast player widget options array.
	 * @param Obj   $widget Current widget instance.
	 * @param Array $settings Settings for current widget instance.
	 *
	 * @since  1.0.0
	 */
	public function widget_options( $options = array(), $widget, $settings ) {

		$fonts = PP_Get_Fn::get_fonts_list();

		$options['default']['items'] = $this->insert_array(
			$options['default']['items'],
			array(
				'pp_fetch_method' => array(
					'setting' => 'pp_fetch_method',
					'label'   => esc_html__( 'Podcast Episodes', 'podcast-player' ),
					'type'    => 'select',
					'choices' => array(
						'feed' => esc_html__( 'From FeedURL', 'podcast-player' ),
						'post' => esc_html__( 'From Posts', 'podcast-player' ),
						'link' => esc_html__( 'From Audio/Video Link', 'podcast-player' ),
					),
				),
			),
			'title'
		);

		$options['default']['items'] = $this->insert_array(
			$options['default']['items'],
			array(
				'pp_post_type' => array(
					'setting'       => 'pp_post_type',
					'label'         => esc_html__( 'Select Post Type', 'podcast-player' ),
					'type'          => 'select',
					'choices'       => PP_Get_Fn::get_post_types(),
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'post', $settings );
					},
				),
				'pp_taxonomy'  => array(
					'setting'       => 'pp_taxonomy',
					'type'          => 'taxonomy',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'post', $settings );
					},
				),
				'pp_terms'     => array(
					'setting'       => 'pp_terms',
					'type'          => 'terms',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'post', $settings ) || $widget->is_option_equal( 'pp_taxonomy', '', $settings );
					},
				),
				'pp_audiosrc'  => array(
					'setting'       => 'pp_audiosrc',
					'label'         => esc_html__( 'Enter Audio/Video Link (i.e, mp3, ogg, m4a etc.)', 'podcast-player' ),
					'type'          => 'text',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'link', $settings );
					},
				),
			)
		);

		$options = $this->insert_array(
			$options,
			array(
				'lshow' => array(
					'title'       => esc_html__( 'Show/Hide player items', 'podcast-player' ),
					'op_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_equal( 'pp_fetch_method', 'link', $settings );
					},
					'items'       => array(
						'pp_ahide_download' => array(
							'setting'       => 'pp_ahide_download',
							'label'         => esc_html__( 'Hide Episode Download Link', 'podcast-player' ),
							'type'          => 'checkbox',
							'hide_callback' => function() use ( $widget, $settings ) {
								return $widget->is_option_not_equal( 'pp_fetch_method', 'link', $settings );
							},
						),
						'pp_ahide_social'   => array(
							'setting'       => 'pp_ahide_social',
							'label'         => esc_html__( 'Hide Social Share Links', 'podcast-player' ),
							'type'          => 'checkbox',
							'hide_callback' => function() use ( $widget, $settings ) {
								return $widget->is_option_not_equal( 'pp_fetch_method', 'link', $settings );
							},
						),
					),
				),
			),
			'info'
		);

		$options['info']['items'] = $this->insert_array(
			$options['info']['items'],
			array(
				'pp_podtitle'   => array(
					'setting'       => 'pp_podtitle',
					'label'         => esc_html__( 'Podcast Title', 'podcast-player' ),
					'type'          => 'text',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_equal( 'pp_fetch_method', 'feed', $settings ) || $widget->is_option_equal( 'pp_fetch_method', 'link', $settings );
					},
				),
				'pp_audiotitle' => array(
					'setting'       => 'pp_audiotitle',
					'label'         => esc_html__( 'Episode Title', 'podcast-player' ),
					'type'          => 'text',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'link', $settings );
					},
				),
				'pp_audiolink'  => array(
					'setting'       => 'pp_audiolink',
					'label'         => esc_html__( 'Podcast episode link for social sharing (optional)', 'podcast-player' ),
					'type'          => 'text',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'link', $settings );
					},
				),
			),
			0
		);

		$options['style']['items'] = $this->insert_array(
			$options['style']['items'],
			array(
				'pp_aspect_ratio' => array(
					'setting'       => 'pp_aspect_ratio',
					'label'         => esc_html__( 'Thumbnail Cropping', 'podcast-player' ),
					'type'          => 'select',
					'choices'       => array(
						''       => esc_html__( 'No Cropping', 'podcast-player' ),
						'land1'  => esc_html__( 'Landscape (4:3)', 'podcast-player' ),
						'land2'  => esc_html__( 'Landscape (3:2)', 'podcast-player' ),
						'port1'  => esc_html__( 'Portrait (3:4)', 'podcast-player' ),
						'port2'  => esc_html__( 'Portrait (2:3)', 'podcast-player' ),
						'wdscrn' => esc_html__( 'Widescreen (16:9)', 'podcast-player' ),
						'squr'   => esc_html__( 'Square (1:1)', 'podcast-player' ),
					),
					'hide_callback' => function() use ( $widget, $settings ) {
						return ! Validation_Fn::is_style_support( $settings['pp_display_style'], 'thumbnail' );
					},
				),
				'pp_crop_method'  => array(
					'setting'       => 'pp_crop_method',
					'label'         => esc_html__( 'Thumbnail Cropping Position', 'podcast-player' ),
					'type'          => 'select',
					'choices'       => array(
						'topleftcrop'      => esc_html__( 'Top Left Cropping', 'podcast-player' ),
						'topcentercrop'    => esc_html__( 'Top Center Cropping', 'podcast-player' ),
						'centercrop'       => esc_html__( 'Center Cropping', 'podcast-player' ),
						'bottomleftcrop'   => esc_html__( 'Bottom Left Cropping', 'podcast-player' ),
						'bottomcentercrop' => esc_html__( 'Bottom Center Cropping', 'podcast-player' ),
					),
					'hide_callback' => function() use ( $widget, $settings ) {
						return ! ( Validation_Fn::is_style_support( $settings['pp_display_style'], 'thumbnail' ) && '' !== $settings['pp_aspect_ratio'] );
					},
				),
				'pp_grid_columns' => array(
					'setting'       => 'pp_grid_columns',
					'label'         => esc_html__( 'Maximum Grid Columns.', 'podcast-player' ),
					'type'          => 'number',
					'input_attrs'   => array(
						'step' => 1,
						'min'  => 1,
						'max'  => 6,
						'size' => 1,
					),
					'hide_callback' => function() use ( $widget, $settings ) {
						return ! Validation_Fn::is_style_support( $settings['pp_display_style'], 'grid' );
					},
				),
				'pp_fonts'        => array(
					'setting' => 'pp_fonts',
					'label'   => esc_html__( 'Select Google Font', 'podcast-player' ),
					'type'    => 'select',
					'choices' => $fonts,
				),
			)
		);

		$options['style']['items'] = $this->insert_array(
			$options['style']['items'],
			array(
				'pp_bgcolor'  => array(
					'setting'       => 'pp_bgcolor',
					'label'         => esc_html__( 'Player Background Color', 'podcast-player' ),
					'type'          => 'color',
					'hide_callback' => function() use ( $widget, $settings ) {
						return ! Validation_Fn::is_style_support( $settings['pp_display_style'], 'bgcolor' );
					},
				),
				'pp_txtcolor' => array(
					'setting'       => 'pp_txtcolor',
					'label'         => esc_html__( 'Text Color Scheme', 'podcast-player' ),
					'type'          => 'select',
					'choices'       => array(
						''      => esc_html__( 'Dark Text', 'podcast-player' ),
						'ltext' => esc_html__( 'Light Text', 'podcast-player' ),
					),
					'hide_callback' => function() use ( $widget, $settings ) {
						return ! Validation_Fn::is_style_support( $settings['pp_display_style'], 'txtcolor' ) || $widget->is_option_equal( 'pp_fetch_method', 'link', $settings );
					},
				),
			),
			0
		);

		$options['filter']['items'] = $this->insert_array(
			$options['filter']['items'],
			array(
				'pp_slist'    => array(
					'setting'       => 'pp_slist',
					'type'          => 'slist',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'feed', $settings );
					},
				),
				'pp_catlist'  => array(
					'setting'       => 'pp_catlist',
					'type'          => 'catlist',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'feed', $settings );
					},
				),
				'pp_elist'    => array(
					'setting'       => 'pp_elist',
					'type'          => 'elist',
					'hide_callback' => function() use ( $widget, $settings ) {
						return $widget->is_option_not_equal( 'pp_fetch_method', 'feed', $settings ) && $widget->is_option_not_equal( 'pp_fetch_method', 'post', $settings );
					},
				),
				'pp_edisplay' => array(
					'setting'       => 'pp_edisplay',
					'label'         => esc_html__( 'Show or Hide above selected episodes', 'podcast-player' ),
					'type'          => 'select',
					'choices'       => array(
						''     => esc_html__( 'Show above selected episodes', 'podcast-player' ),
						'hide' => esc_html__( 'Hide above selected episodes', 'podcast-player' ),
					),
					'hide_callback' => function() use ( $widget, $settings ) {
						$elist = isset( $settings['pp_elist'] ) ? array_filter( $settings['pp_elist'] ) : array();
						return empty( $elist ) || $widget->is_option_not_equal( 'pp_fetch_method', 'feed', $settings ) && $widget->is_option_not_equal( 'pp_fetch_method', 'post', $settings );
					},
				),
			)
		);

		$options['message'] = array(
			'title' => esc_html__( 'Dynamic Audio Message', 'podcast-player' ),
			'items' => array(
				'pp_audio_message'  => array(
					'setting' => 'pp_audio_message',
					'label'   => esc_html__( 'Enter URL of mp3 audio file to be played', 'podcast-player' ),
					'type'    => 'text',
				),
				'pp_play_frequency' => array(
					'setting' => 'pp_play_frequency',
					'label'   => esc_html__( 'Replay Frequency', 'podcast-player' ),
					'type'    => 'number',
					'desc'    => esc_html__( 'After how many episodes the audio should be replayed.', 'podcast-player' ),
				),
				'pp_start_when'     => array(
					'setting' => 'pp_start_when',
					'label'   => esc_html__( 'When to start playing the audio message', 'podcast-player' ),
					'type'    => 'select',
					'choices' => array(
						'start'  => esc_html__( 'Start of the Episode', 'podcast-player' ),
						'end'    => esc_html__( 'End of the Episode', 'podcast-player' ),
						'custom' => esc_html__( 'Custom Time', 'podcast-player' ),
					),
				),
				'pp_start_time'     => array(
					'setting'       => 'pp_start_time',
					'label'         => esc_html__( 'Start playing audio at (time in hh:mm:ss)', 'podcast-player' ),
					'type'          => 'mmss',
					'hide_callback' => function() use ( $widget, $settings ) {
						return 'custom' !== $settings['pp_start_when'];
					},
				),
				'pp_msg_text'       => array(
					'setting' => 'pp_msg_text',
					'label'   => esc_html__( 'Message to be displayed while playing audio.', 'podcast-player' ),
					'type'    => 'text',
				),
			),
		);

		return $options;
	}

	/**
	 * Podcast player premium widget options update.
	 *
	 * @param Array $instance Widget options to be saved.
	 * @param Array $new_instance Modified widget options.
	 * @param Obj   $widget Current widget instance.
	 *
	 * @since  1.0.0
	 */
	public function widget_update( $instance, $new_instance, $widget ) {
		$sanitize_int = array(
			'pp_grid_columns',
			'pp_play_frequency',
		);
		foreach ( $sanitize_int as $setting ) {
			$instance[ $setting ] = absint( $new_instance[ $setting ] );
		}

		$sanitize_text = array(
			'pp_aspect_ratio',
			'pp_crop_method',
			'pp_fetch_method',
			'pp_post_type',
			'pp_podtitle',
			'pp_audiotitle',
			'pp_start_when',
			'pp_msg_text',
			'pp_fonts',
			'pp_txtcolor',
		);
		foreach ( $sanitize_text as $setting ) {
			$instance[ $setting ] = sanitize_text_field( $new_instance[ $setting ] );
		}

		$sanitize_url = array(
			'pp_audiolink',
			'pp_audio_message',
		);
		foreach ( $sanitize_url as $url ) {
			$instance[ $url ] = esc_url_raw( $new_instance[ $url ] );
		}

		$sanitize_bool = array(
			'pp_ahide_download',
			'pp_ahide_social',
		);
		foreach ( $sanitize_bool as $setting ) {
			$instance[ $setting ] = ( 'yes' === $new_instance[ $setting ] ) ? 'yes' : '';
		}

		$sanitize_color = array(
			'pp_bgcolor',
		);

		foreach ( $sanitize_color as $color ) {
			$instance[ $color ] = sanitize_hex_color( $new_instance[ $color ] );
		}

		if ( 'post' !== $instance['pp_fetch_method'] ) {
			$instance['pp_taxonomy'] = '';
			$instance['pp_terms']    = array();
		} else {
			if ( $instance['pp_post_type'] && $new_instance['pp_taxonomy'] ) {
				$instance['pp_taxonomy'] = sanitize_text_field( $new_instance['pp_taxonomy'] );
			} else {
				$instance['pp_taxonomy'] = '';
			}

			if ( $instance['pp_taxonomy'] && $new_instance['pp_terms'] ) {
				$instance['pp_terms'] = array_map( 'sanitize_text_field', $new_instance['pp_terms'] );
			} else {
				$instance['pp_terms'] = array();
			}
		}

		$is_valid = Validation_Fn::is_valid_url( $new_instance['pp_audiosrc'] );
		if ( $is_valid ) {
			$instance['pp_audiosrc'] = esc_url_raw( $new_instance['pp_audiosrc'] );
		} else {
			$instance['pp_audiosrc'] = sanitize_text_field( $new_instance['pp_audiosrc'] );
		}

		$instance['pp_elist']   = array_map( 'sanitize_text_field', $new_instance['pp_elist'] );
		$instance['pp_slist']   = array_map( 'sanitize_text_field', $new_instance['pp_slist'] );
		$instance['pp_catlist'] = array_map( 'sanitize_text_field', $new_instance['pp_catlist'] );

		$instance['pp_edisplay'] = 'hide' === $new_instance['pp_edisplay'] ? 'hide' : '';

		$instance['pp_start_time'] = array_map( 'absint', $new_instance['pp_start_time'] );

		return $instance;
	}

	/**
	 * Podcast player premium widget options update.
	 *
	 * @param Array $display_args Podcast player display args.
	 * @param Array $instance     Settings for current widget instance.
	 *
	 * @since  1.0.0
	 */
	public function widget_display( $display_args, $instance ) {
		$ff = '';
		if ( isset( $instance['pp_fonts'] ) && $instance['pp_fonts'] ) {
			$allff = PP_Get_Fn::get_fonts_list();
			$ff    = str_replace( ' ', '', strtolower( $instance['pp_fonts'] ) );
			if ( isset( $allff[ $ff ] ) ) {
				$this->gfonts[] = $allff[ $ff ];
			} else {
				$ff = '';
			}
		}
		return array_merge(
			$display_args,
			array(
				'aspect-ratio'   => $instance['pp_aspect_ratio'],
				'crop-method'    => $instance['pp_crop_method'],
				'grid-columns'   => $instance['pp_grid_columns'],
				'fetch-method'   => $instance['pp_fetch_method'],
				'post-type'      => $instance['pp_post_type'],
				'taxonomy'       => $instance['pp_taxonomy'],
				'terms'          => $instance['pp_terms'],
				'podtitle'       => $instance['pp_podtitle'],
				'audiosrc'       => $instance['pp_audiosrc'],
				'audiotitle'     => $instance['pp_audiotitle'],
				'audiolink'      => $instance['pp_audiolink'],
				'ahide-download' => $instance['pp_ahide_download'],
				'ahide-social'   => $instance['pp_ahide_social'],
				'audio-msg'      => $instance['pp_audio_message'],
				'play-freq'      => $instance['pp_play_frequency'],
				'msg-start'      => $instance['pp_start_when'],
				'msg-time'       => $instance['pp_start_time'],
				'msg-text'       => $instance['pp_msg_text'],
				'bgcolor'        => $instance['pp_bgcolor'],
				'txtcolor'       => $instance['pp_txtcolor'],
				'font-family'    => $ff,
				'slist'          => $instance['pp_slist'],
				'elist'          => $instance['pp_elist'],
				'catlist'        => $instance['pp_catlist'],
				'edisplay'       => $instance['pp_edisplay'],
			)
		);
	}

	/**
	 * Podcast player premium shortcode options update.
	 *
	 * @param Array $display_args Podcast player display args.
	 * @param Array $atts         Attributes for current shortcode instance.
	 *
	 * @since  1.0.0
	 */
	public function shcode_display( $display_args, $atts ) {
		$ff    = '';
		$terms = array();
		$time  = array( 0, 0, 0 );
		if ( ! empty( $atts['terms'] ) ) {
			$terms = explode( ',', $atts['terms'] );
			$terms = array_map( 'trim', $terms );
		}

		if ( isset( $atts['msg_time'] ) && $atts['msg_time'] ) {
			$tx      = explode( ':', $atts['msg_time'] );
			$tx      = array_map( 'absint', $tx );
			$time[0] = isset( $tx[0] ) ? $tx[0] : 0;
			$time[1] = isset( $tx[1] ) ? $tx[1] : 0;
			$time[2] = isset( $tx[2] ) ? $tx[2] : 0;
		}

		if ( isset( $atts['font_family'] ) && $atts['font_family'] ) {
			$allff = PP_Get_Fn::get_fonts_list();
			$ff    = str_replace( ' ', '', strtolower( $atts['font_family'] ) );
			if ( isset( $allff[ $ff ] ) ) {
				$this->gfonts[] = $allff[ $ff ];
			} else {
				$ff = '';
			}
		}

		if ( isset( $atts['seasons'] ) && $atts['seasons'] ) {
			$slist = array_map( 'trim', explode( ',', $atts['seasons'] ) );
		} else {
			$slist = array();
		}

		if ( isset( $atts['categories'] ) && $atts['categories'] ) {
			$catlist = array_map( 'trim', explode( ',', $atts['categories'] ) );
		} else {
			$catlist = array();
		}

		if ( isset( $atts['elist'] ) && $atts['elist'] ) {
			$elist = array_map( 'trim', explode( ',', $atts['elist'] ) );
		} else {
			$elist = array( '' );
		}

		return array_merge(
			$display_args,
			array(
				'aspect-ratio' => $atts['aspect_ratio'],
				'crop-method'  => $atts['crop_method'],
				'grid-columns' => $atts['grid_columns'],
				'fetch-method' => $atts['fetch_method'],
				'post-type'    => $atts['post_type'],
				'taxonomy'     => $atts['taxonomy'],
				'terms'        => $terms,
				'podtitle'     => $atts['podtitle'],
				'audiosrc'     => $atts['mediasrc'],
				'audiotitle'   => $atts['episodetitle'],
				'audiolink'    => $atts['episodelink'],
				'audio-msg'    => $atts['audio_msg'],
				'play-freq'    => $atts['play_freq'],
				'msg-start'    => $atts['msg_start'],
				'msg-time'     => $time,
				'msg-text'     => $atts['msg_text'],
				'bgcolor'      => $atts['bgcolor'],
				'txtcolor'     => $atts['txtcolor'],
				'font-family'  => $ff,
				'episodes'     => $atts['episodes'],
				'elist'        => $elist,
				'slist'        => $slist,
				'catlist'      => $catlist,
			)
		);
	}

	/**
	 * Podcast player premium block options update.
	 *
	 * @param Array $display_args Podcast player display args.
	 * @param Array $atts         Attributes for current block instance.
	 *
	 * @since  1.0.0
	 */
	public function block_display( $display_args, $atts ) {
		$ff = '';
		if ( isset( $atts['fontFamily'] ) && $atts['fontFamily'] ) {
			$allff = PP_Get_Fn::get_fonts_list();
			$ff    = str_replace( ' ', '', strtolower( $atts['fontFamily'] ) );
			if ( isset( $allff[ $ff ] ) ) {
				$this->gfonts[] = $allff[ $ff ];
			} else {
				$ff = '';
			}
		}
		return array_merge(
			$display_args,
			array(
				'aspect-ratio'   => $atts['aspectRatio'],
				'crop-method'    => $atts['cropMethod'],
				'grid-columns'   => $atts['gridColumns'],
				'fetch-method'   => $atts['fetchMethod'],
				'post-type'      => $atts['postType'],
				'taxonomy'       => $atts['taxonomy'],
				'terms'          => $atts['terms'],
				'podtitle'       => $atts['podtitle'],
				'audiosrc'       => $atts['audioSrc'],
				'audiotitle'     => $atts['audioTitle'],
				'audiolink'      => $atts['audioLink'],
				'ahide-download' => true === $atts['ahideDownload'] ? 1 : 0,
				'ahide-social'   => true === $atts['ahideSocial'] ? 1 : 0,
				'audio-msg'      => $atts['audioMsg'],
				'play-freq'      => $atts['playFreq'],
				'msg-start'      => $atts['msgStart'],
				'msg-time'       => $atts['msgTime'],
				'msg-text'       => $atts['msgText'],
				'bgcolor'        => $atts['bgColor'],
				'txtcolor'       => $atts['txtColor'],
				'font-family'    => $ff,
				'elist'          => $atts['elist'],
				'slist'          => $atts['slist'],
				'catlist'        => $atts['catlist'],
				'edisplay'       => $atts['edisplay'],
			)
		);
	}

	/**
	 * Add additional fetching methods for podcast player.
	 *
	 * @param Array $methods Podcast player episodes fetch methods.
	 *
	 * @since  1.0.0
	 */
	public function fetch_methods( $methods = array() ) {
		return array_merge(
			$methods,
			array(
				'post' => 'PP_Pro\Inc\Post\Post',
				'link' => 'PP_Pro\Inc\Audio\Audio',
			)
		);
	}

	/**
	 * Show podcast player at the bottom of the player.
	 *
	 * @since  4.0.0
	 */
	public function sticky_player() {
		$podcast = get_option( 'pp_sticky_player_feed' );
		if ( ! $podcast ) {
			return;
		}

		if ( ! is_array( $podcast ) ) {
			$podcast = array(
				'url' => $podcast,
			);
		}

		$fmethod = isset( $podcast['fmethod'] ) ? esc_html( $podcast['fmethod'] ) : 'feed';
		$audio   = isset( $podcast['audio'] ) ? esc_url( $podcast['audio'] ) : '';
		$atitle  = isset( $podcast['atitle'] ) ? esc_html( $podcast['atitle'] ) : '';
		$ptype   = isset( $podcast['posttype'] ) ? esc_html( $podcast['posttype'] ) : 'post';
		$taxon   = isset( $podcast['taxonomies'] ) ? esc_html( $podcast['taxonomies'] ) : '';
		$terms   = isset( $podcast['terms'] ) ? array_map( 'sanitize_text_field', $podcast['terms'] ) : array();
		$url     = isset( $podcast['url'] ) ? $podcast['url'] : '';
		$bg      = isset( $podcast['bg'] ) ? sanitize_hex_color( $podcast['bg'] ) : '';
		$accent  = isset( $podcast['accent'] ) ? sanitize_hex_color( $podcast['accent'] ) : '';
		$share   = isset( $podcast['share'] ) && 'yes' === $podcast['share'] ? true : false;
		$dl      = isset( $podcast['dl'] ) && 'yes' === $podcast['dl'] ? true : false;

		$display_args = apply_filters(
			'podcast_player_sticky_display',
			array(
				'fetch-method'  => $fmethod,
				'post-type'     => $ptype,
				'taxonomy'      => $taxon,
				'terms'         => $terms,
				'audiosrc'      => $audio,
				'audiotitle'    => $atitle,
				'url'           => $url,
				'number'        => 2,
				'accent-color'  => $accent,
				'bgcolor'       => $bg,
				'display-style' => 'lv3',
				'hide-header'   => true,
				'hide-search'   => true,
				'hide-download' => $dl,
				'hide-social'   => $share,
				'from'          => 'sticky',
			)
		);

		$display = Display::get_instance();
		$pp      = $display->init( $display_args );
		printf( '<div class="pp-sticky-player" style="display:none !important;">%s</div>', $pp );
	}

	/**
	 * Show bulk podcast player on single posts.
	 *
	 * @since  4.3.0
	 *
	 * @param str $content Post Content.
	 */
	public function bulk_player( $content ) {
		$podcast = get_option( 'pp_bulk_player' );
		$id      = get_the_ID();
		if ( ! $podcast ) {
			return $content;
		}

		$audio      = isset( $podcast['audio'] ) ? esc_html( $podcast['audio'] ) : '';
		$cf         = isset( $podcast['cf'] ) ? esc_html( $podcast['cf'] ) : '';
		$loc        = isset( $podcast['loc'] ) ? esc_html( $podcast['loc'] ) : '';
		$style      = isset( $podcast['style'] ) ? esc_html( $podcast['style'] ) : '';
		$h_title    = isset( $podcast['htitle'] ) && 'yes' === $podcast['htitle'] ? true : false;
		$h_featured = isset( $podcast['hfeatured'] ) && 'yes' === $podcast['hfeatured'] ? true : false;
		$bg         = isset( $podcast['bg'] ) ? sanitize_hex_color( $podcast['bg'] ) : '';
		$accent     = isset( $podcast['accent'] ) ? sanitize_hex_color( $podcast['accent'] ) : '';
		$share      = isset( $podcast['share'] ) && 'yes' === $podcast['share'] ? true : false;
		$dl         = isset( $podcast['dl'] ) && 'yes' === $podcast['dl'] ? true : false;

		if ( ! $audio ) {
			return $content;
		}

		$url = '';
		if ( 'blubrry' === $audio ) {
			// Check if powerpress media is available.
			if ( function_exists( 'powerpress_get_enclosure_data' ) ) {
				$mdata = powerpress_get_enclosure_data( $id );
				if ( ! empty( $mdata['url'] ) ) {
					$mtype = ! empty( $mdata['type'] ) ? $mdata['type'] : '';
					if ( false !== strpos( $mtype, 'audio' ) ) {
						$url = esc_url( $mdata['url'] );
					}
				}
			}
			if ( ! $url ) {
				return $content;
			}
		}

		if ( 'ssp' === $audio ) {
			if ( defined( 'SSP_VERSION' ) ) {
				$url = get_post_meta( $id, 'audio_file', true );
			}
			if ( ! $url ) {
				return $content;
			}
		}

		if ( 'cf' === $audio ) {
			if ( $cf ) {
				$url = get_post_meta( $id, $cf, true );
			}
			if ( ! $url ) {
				return $content;
			}
		}

		if ( ! $url ) {
			return $content;
		}

		$title = '';
		if ( ! $h_title ) {
			$title = get_the_title();
		}

		$image = '';
		if ( ! $h_featured ) {
			$image = get_post_thumbnail_id();
		}

		$display_args = apply_filters(
			'podcast_player_bulk_display',
			array(
				'fetch-method'  => 'link',
				'audiosrc'      => $url,
				'audiotitle'    => $title,
				'accent-color'  => $accent,
				'bgcolor'       => $bg,
				'display-style' => $style,
				'hide-download' => $dl,
				'hide-social'   => $share,
				'image'         => $image,
				'from'          => 'bulk',
			)
		);

		$display = Display::get_instance();
		$pp      = $display->init( $display_args );

		if ( 'before' === $loc ) {
			return $pp . $content;
		}

		return $content . $pp;
	}

	/**
	 * Show single episode podcast player on imported posts.
	 *
	 * @since  5.1.0
	 *
	 * @param str $content Post Content.
	 */
	public function import_player( $content ) {
		$id = get_the_ID();
		if ( ! $id ) {
			return $content;
		}

		$import = get_post_meta( $id, 'pp_import_data', true );
		if ( ! $import ) {
			return $content;
		}

		$findex = get_option( 'pp_feed_index' );
		if ( $findex && is_array( $findex ) && isset( $findex[ $import['podkey'] ] ) ) {
			$podcast = $findex[ $import['podkey'] ];
			if ( isset( $podcast['import'] ) ) {
				$podcast = $podcast['import'];
				$audio   = isset( $import['src'] ) ? esc_url( $import['src'] ) : '';
				$loc     = isset( $podcast['location'] ) ? esc_html( $podcast['location'] ) : '';
				$style   = isset( $podcast['style'] ) ? esc_html( $podcast['style'] ) : '';
				$h_title = isset( $podcast['hide_title'] ) && $podcast['hide_title'] ? true : false;
				$bg      = isset( $podcast['bgcolor'] ) ? sanitize_hex_color( $podcast['bgcolor'] ) : '';
				$accent  = isset( $podcast['accent'] ) ? sanitize_hex_color( $podcast['accent'] ) : '';
				$share   = isset( $podcast['hide_social'] ) && $podcast['hide_social'] ? true : false;
				$dl      = isset( $podcast['hide_download'] ) && $podcast['hide_download'] ? true : false;

				if ( ! $audio || 'noshow' === $loc ) {
					return $content;
				}

				$title = '';
				if ( ! $h_title ) {
					$title = get_the_title();
				}

				$image = get_post_thumbnail_id();

				$display_args = apply_filters(
					'podcast_player_import_display',
					array(
						'fetch-method'  => 'link',
						'audiosrc'      => $audio,
						'audiotitle'    => $title,
						'accent-color'  => $accent,
						'bgcolor'       => $bg,
						'display-style' => $style,
						'hide-download' => $dl,
						'hide-social'   => $share,
						'image'         => $image,
						'from'          => 'import',
					)
				);

				$display = Display::get_instance();
				$pp      = $display->init( $display_args );

				if ( 'after' === $loc ) {
					return $content . $pp;
				}

				return $pp . $content;
			}
		}

		return $content;
	}

	/**
	 * Check if podcast player display class is instantiated.
	 *
	 * @since 3.3.0
	 */
	public function has_podcast_player() {
		// Always load scripts on customizer preview screen.
		if ( is_customize_preview() ) {
			return true;
		}

		$pp         = Instance_Counter::get_instance();
		$has_player = $pp->has_podcast_player();

		return apply_filters( 'podcast_player_has_podcast', $has_player );
	}

	/**
	 * Add an element to an array after certain associative key.
	 *
	 * @param array $array Main array.
	 * @param array $added Array items to be added.
	 * @param str   $key   Array key after which items to be added.
	 *
	 * @since  1.0.0
	 */
	public function insert_array( $array, $added, $key = false ) {

		if ( is_int( $key ) ) {
			$pos = $key;
		} else {
			$pos = $key ? array_search( $key, array_keys( $array ), true ) : false;
			$pos = ( false !== $pos ) ? $pos + 1 : $pos;
		}

		if ( false !== $pos ) {
			return array_merge(
				array_slice( $array, 0, $pos ),
				$added,
				array_slice( $array, $pos )
			);
		}

		return array_merge( $array, $added );
	}

	/**
	 * Returns the instance of this class.
	 *
	 * @since  1.0.0
	 *
	 * @return object Instance of this class.
	 */
	public static function get_instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}
}
