<?php

use League\OAuth2\Server\Grant\AbstractGrant;
use League\OAuth2\Server\ResponseTypes\ResponseTypeInterface;
use Psr\Http\Message\ServerRequestInterface;
use League\OAuth2\Server\Exception\OAuthServerException;
use League\OAuth2\Server\RequestEvent;


class Am_Oauth_Grant_AmemberDevice extends AbstractGrant implements Am_Oauth_Grant_Interface
{
    use Am_Oauth_Traits_ValidateClientId;
    
    public function respondToAccessTokenRequest(
        ServerRequestInterface $request,
        ResponseTypeInterface $responseType,
        \DateInterval $accessTokenTTL
    ) {
        // Validate request
        $client = $this->validateClient($request);
        
        $device = $this->validateDevice($request);
        
        $scopes = $this->validateScopes($this->getRequestParameter('scope', $request, $this->defaultScope));

        // Finalize the requested scopes
        $finalizedScopes = $this->scopeRepository->finalizeScopes($scopes, $this->getIdentifier(), $client);

        // Issue and persist access token
        $accessToken = $this->issueAccessToken($accessTokenTTL, $client, null, $finalizedScopes);

        // Send event to emitter
        $this->getEmitter()->emit(new RequestEvent(RequestEvent::ACCESS_TOKEN_ISSUED, $request));

        // Inject access token into response type
        $responseType->setAccessToken($accessToken);

        return $responseType;
    }

    protected function validateDevice(ServerRequestInterface $request)
    {

        $deviceId = $this->getRequestParameter('amember_device_id', $request);
        if (is_null($deviceId)) {
            throw OAuthServerException::invalidRequest('amember_device_id');
        }

        $device = Am_Di::getInstance()->oauthDeviceTable->findFirstBy(['device_code'=>$deviceId]);
        
        if ($device instanceof OauthDevice === false) {
            $this->getEmitter()->emit(new RequestEvent(RequestEvent::CLIENT_AUTHENTICATION_FAILED, $request));
            throw new OAuthServerException(___('Unregistered device'), 4, 'invalid_device', 401);
        }


        if($device->is_disabled)
        {
            $this->getEmitter()->emit(new RequestEvent(RequestEvent::CLIENT_AUTHENTICATION_FAILED, $request));
            throw new OAuthServerException(___('Device is disabled'), 4, 'invalid_device', 401);
        }
        return $device;
    }
    
    
    
    public function getIdentifier()
    {
        return 'amember_device';
    }
    
    public
        function getDescription()
    {
        return <<<CUT
Grant that should be used for public requests. Ability to exchange client_id and amember_device_id to auth token. 
CUT;
    }

}
