<?php

use League\OAuth2\Server\Entities\ClientEntityInterface;
use League\OAuth2\Server\Repositories\ClientRepositoryInterface;

class OauthClient extends \Am_Record implements ClientEntityInterface
{

    public
        function getIdentifier()
    {
        return $this->client_id;
    }

    public
        function getName()
    {
        return $this->name;
    }

    public
        function getRedirectUri()
    {
        return preg_split("/[\s,]+/", $this->redirect_uri);
    }

    function getAllowedScopes($grantType)
    {
        $scopes = !empty($this->scope) ? json_decode($this->scope, true) : [];
        return !empty($scopes[$grantType]) ? $scopes[$grantType] : [];
    }

    function isAllowedScope($scope, $grantType)
    {
        return in_array($scope->getIdentifier(), $this->getAllowedScopes($grantType));
    }

    function isAllowedGrantType($grantType)
    {
        return in_array($grantType, $this->getGrantType());
    }

    function getGrantType()
    {
        return !empty($this->grant_type) ? json_decode($this->grant_type, true) : [];
    }
    
    function update()
    {
        parent::update();
        
        if($this->is_disabled)
        {
            $this->revokeTokens();
        }
        
        return $this;
    }
    
    function delete()
    {
        $this->revokeTokens();
        
        parent::delete();
    }
    function revokeTokens()
    {
        // Client is disabled. Revoke all issued tokens; 
        foreach(['?_oauth_access_token', '?_oauth_auth_code', '?_oauth_refresh_token'] as $tbl)
        {
            $this->_db->query("update {$tbl} set is_revoked=1 where client_id=?", $this->pk());
        }
    }

}

class OauthClientTable extends \Am_Table implements ClientRepositoryInterface
{

    protected
        $_key = 'oauth_client_id';
    protected
        $_table = '?_oauth_client';
    protected
        $_recordClass = 'OauthClient';

    function getClientEntity($clientIdentifier, $grantType = null, $clientSecret = null, $mustValidateSecret = true)
    {
        $client = $this->findFirstBy(['client_id' => $clientIdentifier]);

        if (!$client)
            return;
        
        if($client->is_disabled)
            return;
        
        if (($mustValidateSecret || $client->require_secret) && ($client->client_secret != $clientSecret))
            return;

        if ($grantType && !$client->isAllowedGrantType($grantType))
        {
            return;
        }

        return $client;
    }
    
    function loadOptions()
    {
        $options = [];
        foreach($this->selectObjects("select * from ?_oauth_client where is_disabled=0") as $client)
        {
            $options[$client->pk()] = $client->name;
        }
        return $options;
    }
}
