<?php
use League\OAuth2\Server\Entities\ScopeEntityInterface;
use League\OAuth2\Server\Repositories\ScopeRepositoryInterface;
use League\OAuth2\Server\Entities\ClientEntityInterface;

class OauthScope implements ScopeEntityInterface
{
    public $id;
    public $description;
    public $grantTypes=[];

    function __construct($id, $description, $grantTypes)
    {
        $this->id = $id;
        $this->description = $description;
        $this->grantTypes = $grantTypes;
    }

    public function getIdentifier()
    {
        return $this->id;
    }

    function getDescription()
    {
        return $this->description;
    }

    function isAvailableForGrantType($grant_type)
    {
        return empty($this->grantTypes) || in_array($grant_type, $this->grantTypes);
    }

    public function jsonSerialize()
    {
        return [
            'id' => $this->id,
            'description' => $this->description,
            'grantTypes' => $this->grantTypes
        ];
    }

    /**
     * Check if another scope (with same id) has same grantTypes and then merge descriptions
     * @return bool true if success, false if other scope is incompatible
     */
    public function merge(OauthScope $scope)
    {
        if ($this->grantTypes != $scope->grantTypes)
            return false;
        if ($scope->description && ($scope->description != $this->description))
            $this->description .= ", " . $scope->description;
        return true;
    }

    static function create($id, $description, $grantTypes)
    {
        return new self($id, $description, $grantTypes);
    }

}

class OauthScopeTable implements ScopeRepositoryInterface
{
    protected $scopes=[];

    public
        function finalizeScopes(array $scopes, $grantType, ClientEntityInterface $clientEntity, $userIdentifier = null)
    {
// isAvailable for grantType
        foreach($scopes as $k=>$scope)
        {
            if(!$scope->isAvailableForGrantType($grantType))
                unset($scopes[$k]);

            if(!$clientEntity->isAllowedScope($scope,$grantType)){
                unset($scopes[$k]);
            }

        }
        return $scopes;
    }

    /**
     *
     * @param OauthScope[] $scopes
     * @throws Am_Exception_InternalError
     */
    function addScope($scopes)
    {

        if (!is_array($scopes)) $scopes = [$scopes];
        foreach ($scopes as $scope)
        {
            $scopeId = $scope->getIdentifier();
            if (isset($this->scopes[$scopeId]))
            {
                if (!$this->scopes[$scopeId]->merge($scope)) // merge with existing
                    throw new Am_Exception_InternalError("Duplicate scope ID: $scopeId and merge is impossible");
            } else {
                $this->scopes[$scopeId] = $scope;
            }
        }
        return $this;
    }


    public
        function getScopeEntityByIdentifier($identifier)
    {
        $ids = is_array($identifier)?$identifier : [$identifier];
        $ret = [];
        foreach($ids as $id)
        {
            if(isset($this->scopes[$id]))
            {
                $ret[] = $this->scopes[$id];
            }
        }

        return is_array($identifier) ? $ret : array_pop($ret);


    }

    function getScopeOptions($grant=null)
    {
        $options = [];
        ksort($this->scopes);
        foreach($this->scopes as $id=>$scope)
        {
            if(is_null($grant)||$scope->isAvailableForGrantType($grant))
                $options[$id] = $id .' - '. $scope->getDescription();
        }
        return $options;
    }
    /**
     *
     * @return OauthScope[]  List of all added scopes
     */
    function getAllScopes()
    {
        return $this->scopes;
    }

}
