package co.pixelbeard.theanfieldwrap.buyTokens;


import android.content.Context;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.viewpager2.widget.ViewPager2;

import com.android.billingclient.api.BillingClient;
import com.android.billingclient.api.BillingResult;
import com.android.billingclient.api.Purchase;
import com.android.billingclient.api.SkuDetails;
import com.android.billingclient.api.SkuDetailsParams;
import com.android.billingclient.api.SkuDetailsResponseListener;

import java.io.IOException;
import java.util.List;

import butterknife.BindView;
import butterknife.ButterKnife;
import co.pixelbeard.theanfieldwrap.R;
import co.pixelbeard.theanfieldwrap.billing.BillingConstants;
import co.pixelbeard.theanfieldwrap.billing.BillingManager;
import co.pixelbeard.theanfieldwrap.billing.Security;
import co.pixelbeard.theanfieldwrap.data.source.DataRepository;
import co.pixelbeard.theanfieldwrap.data.source.local.LocalRepository;
import co.pixelbeard.theanfieldwrap.data.source.remote.RemoteRepository;
import co.pixelbeard.theanfieldwrap.home.HomeFragmentListener;
import co.pixelbeard.theanfieldwrap.utils.BaseActivity;
import co.pixelbeard.theanfieldwrap.utils.BaseFragment;
import co.pixelbeard.theanfieldwrap.utils.ConnectionUtils;
import co.pixelbeard.theanfieldwrap.utils.CustomPagerTransformer;
import co.pixelbeard.theanfieldwrap.utils.FontUtils;
import co.pixelbeard.theanfieldwrap.utils.TawController;
import io.realm.Realm;

public class BuyTokensFragment extends BaseFragment implements SkuDetailsResponseListener, BillingManager.BillingUpdatesListener, BuyTokensContract.View {

    private static final String TAG = BuyTokensFragment.class.getSimpleName();

    private BillingManager mBillingManager;

    @BindView(R.id.ll_back)
    LinearLayout llBack;
    @BindView(R.id.txt_buy_tokens_title)
    TextView txtBuyTokensTitle;
    @BindView(R.id.rv_pager_indicator)
    RecyclerView rvPagerIndicator;
    @BindView(R.id.vp_in_app_purchases)
    ViewPager2 vpInAppPurchase;

    private IndicatorAdapter indicatorAdapter;
    private TokensAdapter tokensAdapter;

    private Context mContext;
    private HomeFragmentListener mListener;

    private BuyTokensContract.Presenter mPresenter;

    public BuyTokensFragment() {
        // Required empty public constructor
    }

    public static BuyTokensFragment newInstance() {
        BuyTokensFragment fragment = new BuyTokensFragment();
        Bundle args = new Bundle();
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        mContext = context;
        mListener = (HomeFragmentListener) context;
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (getArguments() != null) {

        }
        new BuyTokensPresenter(this, new DataRepository(new LocalRepository(Realm.getDefaultInstance()), new RemoteRepository()), mContext);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        // Inflate the layout for this fragment

        View view = inflater.inflate(R.layout.fragment_buy_tokens, container, false);
        ButterKnife.bind(this, view);
        return view;
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        setTextViewFont();
        setOnClickListeners();

        setupBilling();
    }

    @Override
    public void onResume() {
        super.onResume();
        TawController.setCurrentPage(TAG);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mBillingManager.destroy();
    }

    private void setTextViewFont() {
        txtBuyTokensTitle.setTypeface(FontUtils.getInstance().getDinBlack());
    }

    private void setOnClickListeners() {
        llBack.setOnClickListener(v -> mListener.goBack());
    }

    private void setupViewPager(List<SkuDetails> skuDetailsList) {

        tokensAdapter = new TokensAdapter(skuDetailsList, skuDetails -> mBillingManager.initiatePurchaseFlow(skuDetails));

        vpInAppPurchase.setAdapter(tokensAdapter);
        vpInAppPurchase.setPageTransformer(new CustomPagerTransformer(mContext));

        vpInAppPurchase.registerOnPageChangeCallback(new ViewPager2.OnPageChangeCallback() {
            @Override
            public void onPageSelected(int position) {
                super.onPageSelected(position);
                indicatorAdapter.setSelected(position);
            }
        });

    }

    private void setupPagerIndicator(int count) {
        rvPagerIndicator.setLayoutManager(new LinearLayoutManager(mContext, RecyclerView.HORIZONTAL, false));
        indicatorAdapter = new IndicatorAdapter(count);
        rvPagerIndicator.setAdapter(indicatorAdapter);
    }


    private void setupBilling() {
        mBillingManager = new BillingManager((BaseActivity) mContext, this);
    }

    private void querySkuDetails() {
        List<String> skuList = BillingConstants.getSkuList(BillingClient.SkuType.INAPP);
        SkuDetailsParams.Builder params = SkuDetailsParams.newBuilder();
        params.setSkusList(skuList).setType(BillingClient.SkuType.INAPP);

        mBillingManager.querySkuDetailsAsync(BillingClient.SkuType.INAPP, skuList, this);
    }

    @Override
    public void onBillingClientSetupFinished() {
        querySkuDetails();
    }

    @Override
    public void onConsumeFinished(String token, int result) {
        Log.d("CONSUMED IAP TOKEN", token);
        Log.d("CONSUMED RESULT CODE", "" + result);

        if (result == BillingClient.BillingResponseCode.OK) {
            //It was fine
            Log.d("Consume token", "Token was consumed");
        } else {
            Log.d("Consume token", "Consume token failed result: " + result);

        }

    }

    @Override
    public void onPurchasesUpdated(List<Purchase> purchases) {
        for (Purchase purchase : purchases) {

            if (!verifyValidSignature(purchase.getOriginalJson(), purchase.getSignature())) {
                Log.i(TAG, "Got a purchase: " + purchase + "; but signature is bad. Skipping...");
                return;
            }

            Log.d(TAG, "Got a verified purchase: " + purchase);
            for (String skId : purchase.getSkus()) {
                mPresenter.creditAccount(BillingConstants.getValueForSku(skId),
                        BillingConstants.getTitleForSku(skId),
                        purchase.getPurchaseToken());
            }
        }
    }

    private boolean verifyValidSignature(String signedData, String signature) {
        try {
            return Security.verifyPurchase(BillingManager.BASE_64_ENCODED_PUBLIC_KEY, signedData, signature);
        } catch (IOException e) {
            Log.e(TAG, "Got an exception trying to validate a purchase: " + e);
            return false;
        }
    }

    @Override
    public void setPresenter(BuyTokensContract.Presenter presenter) {
        if (presenter != null) {
            this.mPresenter = presenter;
        } else {
            throw new RuntimeException("Presenter is null");
        }
    }

    @Override
    public void goBack() {
        if (mListener != null && isAdded()) {
            mListener.goBack();
        }
    }

    @Override
    public void showLoader(String text) {
        if (mListener != null && isAdded()) {
            mListener.showLoader(text);
        }
    }

    @Override
    public void hideLoader() {
        if (mListener != null && isAdded()) {
            mListener.hideLoader();
        }
    }

    @Override
    public void onUnknownError(String error) {
        if (mListener != null && isAdded()) {
            mListener.showSingleButtonDialog(getString(R.string.error), error, null);
        }
    }

    @Override
    public void onTimeout() {
        if (mListener != null && isAdded()) {
            mListener.showSingleButtonDialog(getString(R.string.timeout_title), getString(R.string.timeout_body), null);
        }
    }

    @Override
    public void onNetworkError() {
        if (mListener != null && isAdded()) {
            mListener.showSingleButtonDialog(getString(R.string.network_error_title), getString(R.string.network_error_body), null);
        }
    }

    @Override
    public void onConnectionError() {
        if (mListener != null && isAdded()) {
            mListener.showNoInternetHeader();
        }
    }

    @Override
    public void logoutUser() {
        //Not needed, should be caught on wallet page
    }

    @Override
    public boolean checkConnection() {
        return ConnectionUtils.userHasConnection(mContext);
    }

    @Override
    public void consumeToken(String token) {
        mBillingManager.consumeAsync(token);
    }

    @Override
    public void showSingleButtonDialog(String error, String message) {
        if (mListener != null) {
            mListener.showSingleButtonDialog(error, message, null);
        }
    }

    @Override
    public void onSkuDetailsResponse(@NonNull BillingResult billingResult, @Nullable List<SkuDetails> skuDetailsList) {
        Log.d("DETAILS SKU", "" + skuDetailsList.size());
        if (skuDetailsList.size() > 0) {
            setupViewPager(skuDetailsList);
            setupPagerIndicator(skuDetailsList.size());
        } else {
            if (mListener != null && isAdded()) {
                mListener.showSingleButtonDialog(getString(R.string.error), getString(R.string.failed_to_get_iap), null);
            }
        }
    }
}
