package co.pixelbeard.theanfieldwrap.videoDetails;

import android.content.Context;
import android.util.Log;

import co.pixelbeard.theanfieldwrap.data.responses.GenericResponse;
import co.pixelbeard.theanfieldwrap.data.responses.GetWalletResponse;
import co.pixelbeard.theanfieldwrap.data.responses.VideoByIdResponse;
import co.pixelbeard.theanfieldwrap.data.source.DataRepository;
import co.pixelbeard.theanfieldwrap.networking.CallbackWrapper;
import co.pixelbeard.theanfieldwrap.utils.PrefKey;
import co.pixelbeard.theanfieldwrap.utils.PrefUtils;
import io.reactivex.Observer;
import io.reactivex.disposables.Disposable;

public class VideoDetailsPresenter implements VideoDetailContract.Presenter {

    private VideoDetailContract.View mView;
    private DataRepository mRepo;

    public VideoDetailsPresenter(VideoDetailContract.View view, DataRepository repo) {
        if (view != null && repo != null) {
            this.mView = view;
            this.mRepo = repo;

            mView.setPresenter(this);
        }
    }

    @Override
    public void recordAnalytic(String type, String variable, String secondVar) {
        mRepo.recordAnalytic(type, variable, secondVar).subscribe(new Observer<GenericResponse>() {
            @Override
            public void onSubscribe(Disposable d) {
            }

            @Override
            public void onNext(GenericResponse genericResponse) {
                if (genericResponse != null) {
                    Log.d("ANALYTIC: " + type, genericResponse.getMessage());
                }
            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onComplete() {

            }
        });
    }

    @Override
    public void getVideoById(String id) {
        if (mView != null && mView.checkConnection()) {
            mView.showLoader("Loading");
            mRepo.getVideoById(id).subscribe(new CallbackWrapper<VideoByIdResponse>(mView) {
                @Override
                protected void onSuccess(VideoByIdResponse videoByIdResponse) {
                    if (mView != null) {
                        mView.setVideo(videoByIdResponse.getVideo());
                    }
                }

                @Override
                protected void onSuccessFalse(VideoByIdResponse podcastByIdResponse) {
                    if (mView != null) {
                        mView.goBack();
                    }
                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void purchaseItem(String id) {
        if (mView != null && mView.checkConnection()) {
            mView.showLoader("Purchasing");
            mRepo.purchaseItem(id).subscribe(new CallbackWrapper<GetWalletResponse>(mView) {
                @Override
                protected void onSuccess(GetWalletResponse getWalletResponse) {
                    PrefUtils.getInstance().putIntPref(PrefKey.TOKEN_BALANCE, getWalletResponse.getWallet().getBalance());
                    if (mView != null) {
                        mView.setVideoPurchased(true);
                        mView.updateWalletBalance(getWalletResponse.getWallet().getBalance());
                    }
                }

                @Override
                protected void onSuccessFalse(GetWalletResponse getWalletResponse) {
                    if (mView != null) {
                        if (getWalletResponse.getMessage().contains("balance was too low")) {
                            mView.showBalanceTooLowDialog();
                        } else {
                            mView.showErrorDialog(getWalletResponse.getMessage());
                        }
                    }
                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void checkWalletBalance() {
        if (mView != null && mView.checkConnection()) {
            mRepo.getWallet().subscribe(new CallbackWrapper<GetWalletResponse>(mView) {
                @Override
                protected void onSuccess(GetWalletResponse getWalletResponse) {
                    if (mView != null) {
                        mView.updateWalletBalance(getWalletResponse.getWallet().getBalance());
                    }
                    PrefUtils.getInstance().putIntPref(PrefKey.TOKEN_BALANCE, getWalletResponse.getWallet().getBalance());
                }

                @Override
                protected void onSuccessFalse(GetWalletResponse getWalletResponse) {
                    if (mView != null) {
                        mView.updateWalletBalance(PrefUtils.getInstance().getIntPref(PrefKey.TOKEN_BALANCE));
                    }
                }
            });
        } else {
            if (mView != null) {
                mView.onConnectionError();
            }
        }
    }

    @Override
    public void subscribe() {

    }

    @Override
    public void unsubscribe() {
        mRepo.onDestroyRepo();
    }

    @Override
    public void clearUserData(Context context) {
        mRepo.logout().subscribe(new Observer<GenericResponse>() {
            @Override
            public void onSubscribe(Disposable d) {

            }

            @Override
            public void onNext(GenericResponse genericResponse) {
                Log.d("LOGOUT", genericResponse.getMessage());
            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onComplete() {

            }
        });
        //TODO test this
//        mRepo.clearAllData(context);
    }
}
