package co.pixelbeard.theanfieldwrap.playerService;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.util.Base64;

import com.google.gson.Gson;

import java.io.ByteArrayOutputStream;

import co.pixelbeard.theanfieldwrap.R;
import co.pixelbeard.theanfieldwrap.data.Podcast;

public class PodcastStorageUtil {


    private static final String PODCAST = "PODCAST";
    private static final String AUTO_PLAY = "AUTO_PLAY";
    private static final String ALBUM_ART = "ALBUM_ART";
    private static final String PLAY_POSITION = "PLAY_POSITION";


    private static final String TAG = PodcastStorageUtil.class.getSimpleName();
    private static PodcastStorageUtil instance = null;
    private final String PREF_NAME = "co.pixelbeard.theanfieldwrap.PODCAST_STORAGE_UTIL";
    private SharedPreferences mPrefs;
    private SharedPreferences.Editor mEditor;

    public PodcastStorageUtil(Context context) {
        mPrefs = context.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
    }


    public static synchronized PodcastStorageUtil getInstance(Context context) {
        if (instance == null) {
            instance = new PodcastStorageUtil(context);
        }
        return instance;
    }

    public static synchronized PodcastStorageUtil getInstance() {
        if (instance == null) {
            throw new IllegalStateException(TAG +
                    "is not initialized, call getInstance(Context) first");
        }
        return instance;
    }

    public void storeAudio(Podcast podcast) {
        doEdit();
        mEditor.putString(PODCAST, new Gson().toJson(podcast));
        doCommit();
    }

    public Podcast loadAudio() {
        String json = mPrefs.getString(PODCAST, null);

        if (json != null) {
            return new Gson().fromJson(json, Podcast.class);
        } else {
            return null;
        }
    }

    public void storeAlbumArt(Bitmap bitmap) {

        if (bitmap != null) {
            doEdit();

            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, baos); //bm is the bitmap object
            byte[] b = baos.toByteArray();

            mEditor.putString(ALBUM_ART, Base64.encodeToString(b, Base64.DEFAULT));
            doCommit();
        }
    }

    public Bitmap getAlbumArt(Context context) {
        String artString = mPrefs.getString(ALBUM_ART, null);

        Bitmap bitmap;

        if (artString == null) {
            bitmap = BitmapFactory.decodeResource(context.getResources(), R.drawable.ic_placeholder_podcast_small);
        } else {

            byte[] imageAsBytes = Base64.decode(artString.getBytes(), Base64.DEFAULT);
            bitmap = BitmapFactory.decodeByteArray(imageAsBytes, 0, imageAsBytes.length);
        }
        return bitmap;
    }

//    public void storeAudioPosition(Long position) {
//        doEdit();
//        mEditor.putLong(PLAY_POSITION, position);
//        doCommit();
//    }

//    public Long getAudioPosition() {
//        return mPrefs.getLong(PLAY_POSITION, 0);
//    }

    public void clearStoredPodcast() {
        doEdit();
        mEditor.remove(PODCAST);
        mEditor.remove(ALBUM_ART);
        mEditor.remove(PLAY_POSITION);
        mEditor.remove(AUTO_PLAY);
        mEditor.clear();
        doCommit();
    }

    public void setShouldAutoPlay(boolean shouldAutoPlay) {
        doEdit();
        mEditor.putBoolean(AUTO_PLAY, shouldAutoPlay);
        doCommit();
    }

    public boolean shouldAutoPlay() {
        return mPrefs.getBoolean(AUTO_PLAY, false);
    }

    @SuppressLint("CommitPrefEdits")
    private void doEdit() {
        if (mEditor == null) {
            mEditor = mPrefs.edit();
        }
    }

    private void doCommit() {
        mEditor.apply();
        mEditor = null;
    }
}
