//
//  TAWSignUpViewController.swift
//  TAW
//
//  Created by Andrew Steven on 17/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Crashlytics
import TransitionButton

class TAWSignUpViewController: PBAnimatedViewController {
    
    // MARK: - Interface Outlets -

    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var stepLabel: UILabel!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var nextButton: TransitionButton!
    
    @IBOutlet weak var scrollView: UIScrollView!
    @IBOutlet weak var contentView: UIView!
    
    @IBOutlet weak var step1: UIView!
    @IBOutlet weak var descriptionLabel: UILabel!
    @IBOutlet weak var firstNameTextField: PBTextField!
    @IBOutlet weak var lastNameTextField: PBTextField!
    @IBOutlet weak var usernameTextField: PBTextField!
    
    @IBOutlet weak var step2: UIView!
    @IBOutlet weak var emailTextField: PBTextField!
    @IBOutlet weak var passwordTextField: PBTextField!
    @IBOutlet weak var countryTextField: PBTextField!
    
    @IBOutlet weak var step3: UIView!
    @IBOutlet weak var authCodeTextField: PBTextField!
    @IBOutlet weak var resendCodeButton: UIButton!
    
    @IBOutlet weak var nextButtonHeight: NSLayoutConstraint!
    
    // MARK: - Variables -
    
    var currentPage: CGFloat = 0
    var convertingGuest: Bool = false
    
    // Countries
    var countries: [String] = []
    var countryPickerView: UIPickerView = UIPickerView(frame: CGRect.zero)
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        descriptionLabel.isHidden = UIDevice.isSmallDevice

        firstNameTextField.theTextFieldLabel.text = "FIRST NAME"
        firstNameTextField.setType(type: .standard, validation_type: .standard)
        firstNameTextField.theTextField.addTarget(self,
                                                  action: #selector(textFieldDidChange(textField:)),
                                                  for: .editingChanged)
        
        lastNameTextField.theTextFieldLabel.text = "LAST NAME"
        lastNameTextField.setType(type: .standard, validation_type: .standard)
        lastNameTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)
        
        usernameTextField.theTextFieldLabel.text = "USERNAME"
        usernameTextField.setType(type: .standard, validation_type: .standard)
        usernameTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)
        
        emailTextField.theTextFieldLabel.text = "EMAIL"
        emailTextField.setType(type: .email, validation_type: .email)
        emailTextField.theTextField.addTarget(self,
                                              action: #selector(textFieldDidChange(textField:)),
                                              for: .editingChanged)
        
        passwordTextField.theTextFieldLabel.text = "PASSWORD"
        passwordTextField.setType(type: .password, validation_type: .password)
        passwordTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)
        
        countryTextField.theTextFieldLabel.text = "COUNTRY"
        countryTextField.theTextField.text = "United Kingdom"
        countryTextField.setType(type: .standard, validation_type: .standard)
        countryTextField.theTextField.addTarget(self,
                                                action: #selector(textFieldDidChange(textField:)),
                                                for: .editingChanged)
        countryPickerView.delegate = self
        countryPickerView.backgroundColor = UIColor.white
        countryTextField.theTextField.inputView = countryPickerView
        
        authCodeTextField.theTextFieldLabel.text = "AUTHORISATION CODE"
        authCodeTextField.setType(type: .code, validation_type: .code)
        authCodeTextField.theTextField.addTarget(self,
                                                 action: #selector(textFieldDidChange(textField:)),
                                                 for: .editingChanged)
        
        resendCodeButton.isHidden = true
        
        let tapGesture = UITapGestureRecognizer(target: self, action: #selector(dismissKeyboard))
        tapGesture.delegate = self
        self.view.addGestureRecognizer(tapGesture)
        
        self.nextButton.backgroundColor = TAW.Colors.greyButtonColor
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        self.animations = [.slide(.left, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        self.setupView()
        self.getCountries()
    }
    
    private func setupView() {
        if UIDevice.isSmallDevice {
            self.view.layoutIfNeeded()
            self.nextButtonHeight.constant = 45
            self.view.layoutIfNeeded()
        }
    }
    
    private func getCountries() {
        
        for code in NSLocale.isoCountryCodes as [String] {
            let id = NSLocale.localeIdentifier(fromComponents: [NSLocale.Key.countryCode.rawValue: code])
            let name = NSLocale(localeIdentifier: "en_UK").displayName(forKey: NSLocale.Key.identifier, value: id) ?? "Country not found for code: \(code)"
            self.countries.append(name)
        }
        
        self.countries = self.countries.sorted()
        self.countryPickerView.reloadAllComponents()
    }
    
    // MARK: - UITextField -
    
    @objc func textFieldDidChange(textField: UITextField){
        self.invisibleCheck()
    }
    
    @objc func dismissKeyboard() {
        self.view.endEditing(true)
    }
    
    // MARK: - UIScrollView -
    
    private func scrollToStep() {
        let pageWidth: CGFloat = scrollView.frame.width
        let slideToX = currentPage * pageWidth
        self.scrollView.scrollRectToVisible(CGRect(x: slideToX,
                                                   y: 0, width: pageWidth,
                                                   height: self.scrollView.frame.height),
                                            animated: true)
        
        self.nextButton.setTitle(self.currentPage < 2 ? "NEXT" : "FINISH", for: .normal)
        self.stepLabel.text = "STEP \(Int(self.currentPage + 1))/3"
        self.resendCodeButton.isHidden = self.currentPage != 2
        self.backButton.isHidden = self.currentPage == 2
        self.invisibleCheck()
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        
        if currentPage == 0 {
            if self.isModal {
                self.navigationController?.dismiss(animated: true, completion: nil)
            } else {
                self.navigationController?.popViewController(animated: true)
            }
            
        } else {
            currentPage -= 1
            scrollToStep()
        }
    }
    
    @IBAction func goNext() {
        if valid() {
            if currentPage == 0 {
                self.checkUsernameIsAvailable()
            } else if currentPage == 1 {
                self.checkEmailIsAvailable()
            } else if currentPage == 2 {
                self.activateUser()
            }
        }
    }
    
    // MARK: - Validation functions -
    
    private func valid() -> Bool {
        if currentPage == 0 {
            let validFirstName = self.firstNameTextField.checkValidation(required: true, type: .standard)
            if !validFirstName { return false }
            
            let validLastName = self.lastNameTextField.checkValidation(required: true, type: .standard)
            if !validLastName { return false }
            
            let validUsername = self.usernameTextField.checkValidation(required: true, type: .standard)
            if !validUsername { return false }
            
            return true
        } else if currentPage == 1 {
            let validEmail = self.emailTextField.checkValidation(required: true, type: .email)
            if !validEmail { return false }
            
            let validPassword = self.passwordTextField.checkValidation(required: true, type: .password)
            if !validPassword { return false }
            
            let validCountry = self.countryTextField.checkValidation(required: true, type: .standard)
            if !validCountry { return false }
            
            return true
        } else {
            let validAuthCode = self.authCodeTextField.checkValidation(required: true, type: .code)
            if !validAuthCode { return false }
            
            return true
        }
    }
    
    private func invisibleCheck() {
        
        var isEnabled = false
        if currentPage == 0 {
            let validFirstName = self.firstNameTextField.invisibleCheckValidation(type: .standard)
            let validLastName = self.lastNameTextField.invisibleCheckValidation(type: .standard)
            let validUsername = self.usernameTextField.invisibleCheckValidation(type: .standard)
            
            if validFirstName && validLastName && validUsername {
                isEnabled = true
            }
        } else if currentPage == 1 {
            let validEmail = self.emailTextField.invisibleCheckValidation(type: .email)
            let validPassword = self.passwordTextField.invisibleCheckValidation(type: .password)
            let validCountry = self.countryTextField.invisibleCheckValidation(type: .standard)
            
            if validEmail && validPassword && validCountry {
                isEnabled = true
            }
        } else {
            let validAuthCode = self.authCodeTextField.invisibleCheckValidation(type: .code)
            
            if validAuthCode {
                isEnabled = true
            }
        }
        
        if isEnabled {
            self.nextButton.backgroundColor = TAW.Colors.redButtonColor
        } else {
            self.nextButton.backgroundColor = TAW.Colors.greyButtonColor
        }
    }
    
    // MARK: - API functions -
    
    private func checkUsernameIsAvailable() {
        if Connectivity.isConnectedToInternet {
            self.nextButton.startAnimation()

            let params = [
                "username": self.usernameTextField.theTextField.text!
            ]
            
            APIClient.checkUsernameIsAvailable(parameters: params) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    self.nextButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                        self.currentPage += 1
                        self.scrollToStep()
                    })
                } else {
                    self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                        if jsonResponse["message"] as? String ?? "" == "Username already exists in the database" {
                            self.usernameTextField.overrideFailValidation(message: "Username already taken")
                        } else {
                            self.showAlertWithTitle("Error", message: "An error occurred. Try again later")
                        }
                    })
                }
            }
        } else {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        }
    }
    
    private func checkEmailIsAvailable() {
        if Connectivity.isConnectedToInternet {
            self.nextButton.startAnimation()

            let params = [
                "email": self.emailTextField.theTextField.text!
            ]
            
            APIClient.checkEmailIsAvailable(parameters: params) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    self.registerUser()
                } else {
                    self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                        self.emailTextField.overrideFailValidation(message: "Email already registered")
                    })
                }
            }
        } else {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        }
    }
    
    private func registerUser() {
        
        if self.convertingGuest {
            self.convertGuestToFullUser()
            return
        }
        
        if !Connectivity.isConnectedToInternet {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        } else {
            let params: [String: Any] = [
                "firstname": self.firstNameTextField.theTextField.text!,
                "lastname": self.lastNameTextField.theTextField.text!,
                "username": self.usernameTextField.theTextField.text!,
                "email": self.emailTextField.theTextField.text!,
                "password": self.passwordTextField.theTextField.text!,
                "country": self.countryTextField.theTextField.text!,
                "opt_in": "true"
            ]
            
            APIClient.registerUser(parameters: params) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    self.nextButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                        self.currentPage += 1
                        self.scrollToStep()
                    })
                } else {
                    self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    })
                }
            }
        }
    }
    
    private func convertGuestToFullUser() {
        
        if !Connectivity.isConnectedToInternet {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        } else {
            let params: [String: Any] = [
                "firstname": self.firstNameTextField.theTextField.text!,
                "lastname": self.lastNameTextField.theTextField.text!,
                "username": self.usernameTextField.theTextField.text!,
                "email": self.emailTextField.theTextField.text!,
                "password": self.passwordTextField.theTextField.text!,
                "country": self.countryTextField.theTextField.text!,
                "opt_in": "true"
            ]
            
            APIClient.convertToFullUser(parameters: params) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    self.nextButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                        self.currentPage += 1
                        self.scrollToStep()
                    })
                } else {
                    self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    })
                }
            }
        }
    }
    
    private func activateUser() {
        
        if !Connectivity.isConnectedToInternet {
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!,
                                           withTitle: "No internet connection",
                                           style: .noInternet)
        } else {
            self.nextButton.startAnimation()
            
            let params = [
                "activation_code": self.authCodeTextField.theTextField.text!,
                "email": self.emailTextField.theTextField.text!
            ]
            
            APIClient.activateUser(parameters: params) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    self.login()
                } else {
                    self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                        self.authCodeTextField.overrideFailValidation(message: "Invalid code")
                    })
                }
            }
        }
    }
    
    @IBAction func resendActivationCode() {
        self.showAlertWithTitle("Success", message: "We have sent you an email containing your activation code.")
        let params = [
            "email": self.emailTextField.theTextField.text!
        ]
        
        APIClient.resendActivationEmail(parameters: params) { (jsonResponse, error) in

        }
    }
    
    private func login() {
        
        let params = [
            "email": self.emailTextField.theTextField.text!,
            "passwd": self.passwordTextField.theTextField.text!,
            "device_token": "",
            "device_type": "iOS"
        ]
        
        APIClient.loginUser(parameters: params) { (jsonResponse, error) in
            if let userJson = jsonResponse["user"] as? [String: Any] {
                let user = User(JSON: userJson)
                user?.authToken = jsonResponse["auth_token"] as? String
                user?.guest = false
                LocalStorage.shared.user = user
                LocalStorage.shared.loggedIn = true
                
                Crashlytics.sharedInstance().setUserName(user!.username)
                Crashlytics.sharedInstance().setUserIdentifier("Auth Token: \(user!.authToken!)")
                Crashlytics.sharedInstance().setUserEmail(user!.guest ? "Guest" : user!.email)
                
                Utilities().deleteOtherUsersPodcastPositions()

                self.nextButton.stopAnimation(animationStyle: .normal, revertAfterDelay: 0.3, completion: {
                    let alertview = TAWAlertView.shared
                    alertview.delegate = self
                    alertview.showAlertView(inView: self.navigationController!.view, alertStyle: .newAccount)
                })
            } else {
                self.nextButton.stopAnimation(animationStyle: .shake, revertAfterDelay: 0.3, completion: {
                    self.showAlertForAPIError(jsonResponse["message"] as! String)
                })
            }
        }
    }
}

extension TAWSignUpViewController: UIPickerViewDelegate, UIPickerViewDataSource {
    
    func numberOfComponents(in pickerView: UIPickerView) -> Int {
        return 1
    }
    
    func pickerView(_ pickerView: UIPickerView,
                    numberOfRowsInComponent component: Int) -> Int {
        
        return countries.count
    }
    
    func pickerView(_ pickerView: UIPickerView,
                    viewForRow row: Int,
                    forComponent component: Int,
                    reusing view: UIView?) -> UIView {
        
        let pickerLabel = UILabel()
        pickerLabel.font = TAW.Fonts.dinAlternate21
        pickerLabel.textColor = UIColor.black
        pickerLabel.textAlignment = .center
        pickerLabel.text = self.countries[row]
        return pickerLabel
    }
    
    func pickerView(_ pickerView: UIPickerView,
                    didSelectRow row: Int,
                    inComponent  component: Int) {
        
        self.countryTextField.theTextField.text = self.countries[row]
        self.countryTextField.layoutTheTextFieldLabel()
    }
    
    func pickerView(_ pickerView: UIPickerView,
                    widthForComponent component: Int) -> CGFloat {
        
        return UIScreen.main.bounds.width
    }
    
    func pickerView(_ pickerView: UIPickerView,
                    rowHeightForComponent component: Int) -> CGFloat {
        
        return 40
    }
}

extension TAWSignUpViewController: TAWAlertViewDelegate {
    func didTapLogout() {
        // Do nothing
    }

    func acceptedNotifications() {
        // Do nothing
    }
    
    func unlinkAccount() {
        // Do nothing
    }
    
    func accountCreated() {
        isSubscriptionPopupClose = false
        let notificationCenter = NotificationCenter.default
        notificationCenter.post(name: Notification.Name(rawValue: CheckFreeSubscription),
                                object: self)
        self.navigationController?.dismiss(animated: true, completion: nil)
    }
    
    
}
