//
//  PBAudioPlayer.swift
//  TAW
//
//  Created by Andrew Steven on 30/01/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import Foundation
import AVFoundation
import MediaPlayer
import Kingfisher
import RealmSwift
import Crashlytics

let PBAudioPlayerOnTrackChanged = "PBAudioPlayerOnTrackChanged"
let PBAudioPlayerOnPlaybackStateChanged = "PBAudioPlayerOnPlaybackStateChanged"
let PBAudioPlayerOnTrackDidFinishPlaying = "PBAudioPlayerOnTrackDidFinishPlaying"
let PBAudioPlayerOnShowBuffering = "PBAudioPlayerOnShowBuffering"
let PBAudioPlayerOnHideBuffering = "PBAudioPlayerOnHideBuffering"
let PBAudioPlayerOnPreviewFinished = "PBAudioPlayerOnPreviewFinished"

enum DurationFormat {
    case full // 2 hours, 32 minutes, 22 seconds
    case short // 2 hr, 32 min, 22 sec
    case brief // 2hr 32min 22sec
    case abbreviated // 2h 32m 22s
    case positional // 2:32:22
}

open class PBAudioPlayer: NSObject {
    
    static let shared = PBAudioPlayer(dependencies: (AVAudioSession.sharedInstance(),
                                                     MPRemoteCommandCenter.shared(),
                                                     MPNowPlayingInfoCenter.default(),
                                                     NotificationCenter.default))
    
    // MARK: - Variables -
    
    var audioPlayer: AVPlayer?
    var podcast: Podcast?
    var isPreview: Bool = false
    var timeObserver: Any?
    var periodicObserver: Any?
    
    var nowPlayingInfo: [String: Any]?
    
    open var elapsedTime: Double {
        guard let currentPodcast = self.audioPlayer?.currentItem else {
            return 0
        }
        return currentPodcast.currentTime().seconds
    }
    
    open var totalDuration: Double {
        guard let currentPodcast = self.audioPlayer?.currentItem else {
            return 0
        }
        return currentPodcast.duration.seconds
    }
    
    private func formatDurationInSeconds(_ timeInterval: TimeInterval) -> String {
        
        let formatter = DateComponentsFormatter()
        formatter.allowedUnits = [.second]
        formatter.unitsStyle = .positional
        formatter.zeroFormattingBehavior = DateComponentsFormatter.ZeroFormattingBehavior.pad
        
        return formatter.string(from: TimeInterval(timeInterval))!
    }
    
    func getElapsedTime(_ format: DurationFormat) -> String {
        guard let currentPodcast = self.audioPlayer?.currentItem else {
            return "0:00:00"
        }
        
        let elapsedDuration = TimeInterval(floatLiteral: currentPodcast.currentTime().seconds)
        return formatTimeInterval(elapsedDuration, format: format)
    }
    
    func getDuration(_ format: DurationFormat) -> String {
        guard let currentPodcast = self.audioPlayer?.currentItem else {
            return "0:00:00"
        }
        
        let totalDuration: TimeInterval = TimeInterval(floatLiteral: currentPodcast.duration.seconds)
        return formatTimeInterval(totalDuration, format: format)
    }
    
    private func formatTimeInterval(_ timeInterval: TimeInterval, format: DurationFormat) -> String {
        let formatter = DateComponentsFormatter()
        formatter.allowedUnits = [.hour, .minute, .second]
        
        switch format {
        case .full:
            formatter.unitsStyle = .full
        case .short:
            formatter.unitsStyle = .short
        case .brief:
            formatter.unitsStyle = .brief
        case .abbreviated:
            formatter.unitsStyle = .abbreviated
        case .positional:
            formatter.unitsStyle = .positional
        }
        
        formatter.zeroFormattingBehavior = DateComponentsFormatter.ZeroFormattingBehavior.pad
        Crashlytics.sharedInstance().setObjectValue(self.podcast, forKey: "podcast")
        Crashlytics.sharedInstance().setObjectValue(timeInterval, forKey: "timeInterval")
        
        if timeInterval.isNaN {
            return "0:00:00"
        }
        return formatter.string(from: timeInterval) ?? "0:00:00"
    }
    
    open var isPlaying: Bool {
        return self.audioPlayer?.rate == 1 ? true : false
    }
    
    // MARK: - Dependencies -
    
    let audioSession: AVAudioSession
    let commandCenter: MPRemoteCommandCenter
    let nowPlayingInfoCenter: MPNowPlayingInfoCenter
    let notificationCenter: NotificationCenter
    
    // MARK: - Initialisers -
    
    typealias PBAudioPlayerDependencies = (audioSession: AVAudioSession, commandCenter: MPRemoteCommandCenter, nowPlayingInfoCenter: MPNowPlayingInfoCenter, notificationCenter: NotificationCenter)
    
    init(dependencies: PBAudioPlayerDependencies) {
        self.audioSession = dependencies.audioSession
        self.commandCenter = dependencies.commandCenter
        self.nowPlayingInfoCenter = dependencies.nowPlayingInfoCenter
        self.notificationCenter = dependencies.notificationCenter
        
        super.init()
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(endPlayback),
                                            name: NSNotification.Name.AVPlayerItemDidPlayToEndTime,
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(handleInterruption),
                                            name: AVAudioSession.interruptionNotification,
                                            object: nil)
        
        self.configureCommandCenter()
        
//        if UserDefaults.standard.value(forKey: "restorePodcast") != nil {
//            if let storedData = UserDefaults.standard.value(forKey: "restorePodcast") as? [String: Any?] {
//                if let podcastId = storedData["podcast_id"] as? String,
//                    let elapsedTime = storedData["elapsed"] as? Double,
//                    let isPreview = storedData["preview"] as? Bool {
//
//                    do {
//                        let realm = try Realm()
//                        let podcastResults = realm.objects(Podcast.self).filter("postId == %@", podcastId)
//                        if let podcast = podcastResults.first {
//                            self.restorePodcast(podcast, elapsedTime: elapsedTime, isPreview: isPreview)
//                            UserDefaults.standard.removeObject(forKey: "restorePodcast")
//                        }
//
//                    } catch let error {
//                        print("Failed to retrieve podcast: \(error.localizedDescription)")
//                    }
//                }
//            }
//        }
    }

    deinit {
        self.notificationCenter.removeObserver(self, name: NSNotification.Name.AVPlayerItemDidPlayToEndTime, object: nil)
        self.notificationCenter.removeObserver(self, name: AVAudioSession.interruptionNotification, object: nil)
    }
    
    @objc func handleInterruption(notification: Notification) {
        
        guard let userInfo = notification.userInfo,
            let typeValue = userInfo[AVAudioSessionInterruptionTypeKey] as? UInt,
            let type = AVAudioSession.InterruptionType(rawValue: typeValue) else {
                return
        }
        if type == .began {
            // Interruption began, take appropriate actions
        } else if type == .ended {
            if let optionsValue = userInfo[AVAudioSessionInterruptionOptionKey] as? UInt {
                let options = AVAudioSession.InterruptionOptions(rawValue: optionsValue)
                if options.contains(.shouldResume) {
                    // Interruption Ended - playback should resume
                    self.play()
                } else {
                    // Interruption Ended - playback should NOT resume
                }
            }
        }
    }
    
    // MARK: - PBAudio Player Commands -
    
//    func restorePodcast(_ podcast: Podcast, elapsedTime: Double, isPreview: Bool) {
//
//        do {
//            try self.audioSession.setCategory(AVAudioSession.Category.playback, mode: .spokenAudio, options: [])
//            try self.audioSession.setActive(true)
//            UIApplication.shared.beginReceivingRemoteControlEvents()
//
//            // Check if podcast is downloaded before playing
//            let url: URL = podcast.isDownloaded ? podcast.getSaveFileUrl() : podcast.safeURL
//
//            // Set the AVPlayerItem using the url provided
//            let playerItem = AVPlayerItem(url: url)
//            playerItem.addObserver(self, forKeyPath: "playbackBufferEmpty", options: .new, context: nil)
//            playerItem.addObserver(self, forKeyPath: "playbackLikelyToKeepUp", options: .new, context: nil)
//            playerItem.addObserver(self, forKeyPath: "playbackBufferFull", options: .new, context: nil)
//
//            self.audioPlayer = AVPlayer(playerItem: playerItem)
//
//            self.seekTo(elapsedTime)
//
//            if isPreview {
//                self.timeObserver = self.audioPlayer?.addBoundaryTimeObserver(forTimes: [NSValue(time: CMTime(seconds: 120,
//                                                                                          preferredTimescale: 1000))],
//                                                          queue: DispatchQueue.main, using: {
//                                                            // Stop podcast from continuing
//                                                            self.endPlayback()
//                                                            self.audioPlayer?.seek(to: CMTime(seconds: 0, preferredTimescale: 1000))
//
//                                                            // Post notification to inform Podcast Player view
//                                                            self.notifyOnPreviewFinished()
//                })
//            }
//
//            self.isPreview = isPreview
//            self.podcast = podcast
//
//            // Send a notification to various views observing podcast changes/updates
//            self.updateNowPlayingInfoForCurrentPlaybackItem()
//            self.notifyOnTrackChanged()
//        }
//        catch let error {
//            print("Error restoring podcast: \(error.localizedDescription)")
//        }
//    }
    
    func playPodcast(_ podcast: Podcast, isPreview: Bool) {
        
        try! self.audioSession.setCategory(AVAudioSession.Category.playback,
                                           mode: .spokenAudio, options: [])
        try! self.audioSession.setActive(true)
        UIApplication.shared.beginReceivingRemoteControlEvents()
        
        // Check if podcast is downloaded before playing
        let url: URL = podcast.isDownloaded ? podcast.getSaveFileUrl() : podcast.safeURL
        
        // Set the AVPlayerItem using the url provided
        let playerItem = AVPlayerItem(url: url)

        // Remove previous buffer time observer before setting new audioPlayer
        if let observer = self.periodicObserver {
            // Remove current instance
            self.audioPlayer?.removeTimeObserver(observer)
            self.periodicObserver = nil
        }

        // Set new podcast
        self.audioPlayer = AVPlayer(playerItem: playerItem)
        // Set preview and podcast variables
        self.isPreview = isPreview
        self.podcast = podcast

        // Add time observer for buffer state
        self.addPeriodicObserver()

        // Add time observer for previews
        if isPreview {
            self.addPreviewTimeObserver()
        }

        // Get current position from realm for current podcast
        let realm = try! Realm()
        let podcastPosition = realm.objects(PodcastPosition.self).filter("postId == %@", podcast.postId)
        var currentPosition = podcastPosition.first?.position ?? 0.0
        if isPreview {
            currentPosition = 0.0
        }

        

        // Update position of podcast
        self.seekTo(currentPosition)
        self.audioPlayer?.play()

        // Record analytic for playing podcast
        let analyticType: AppAnalytic =
            self.podcast!.isDownloaded ? .playedDownloadedPodcast : .playedOnlinePodcast
        APIClient.recordAnalytic(analyticType,
                                 variable: podcast.postId,
                                 secondaryVariable: "1")
        
        // Send a notification to various views observing podcast changes/updates
        self.updateNowPlayingInfoForCurrentPlaybackItem()
        self.notifyOnTrackChanged()
    }

    func addPreviewTimeObserver() {
        self.timeObserver = self.audioPlayer?.addBoundaryTimeObserver(forTimes: [NSValue(time: CMTime(seconds: 120,
                                                                                                      preferredTimescale: 1000))],
                                                                      queue: DispatchQueue.main, using: {
                                                                        // Stop podcast from continuing
                                                                        self.endPlayback()
                                                                        self.audioPlayer?.seek(to: CMTime(seconds: 0,
                                                                                                          preferredTimescale: 1000))

                                                                        // Post notification to inform Podcast Player view
                                                                        self.notifyOnPreviewFinished()
        })
    }

    func addPeriodicObserver() {

        let interval: CMTime = CMTime(seconds: 1, preferredTimescale: 10)
        self.periodicObserver = self.audioPlayer?.addPeriodicTimeObserver(forInterval: interval, queue: DispatchQueue.main, using: { [weak self] time in
            guard let self = self else {
                return
            }

            let playbackLikelyToKeepUp = self.audioPlayer?.currentItem?.isPlaybackLikelyToKeepUp
            if playbackLikelyToKeepUp == false {
                // Show loader
                self.notifyOnShowBuffer()
            } else {
                // Hide loader
                self.notifyOnHideBuffer()
            }
        })
    }
    
//    open override func observeValue(forKeyPath keyPath: String?,
//                                    of object: Any?,
//                                    change: [NSKeyValueChangeKey : Any]?,
//                                    context: UnsafeMutableRawPointer?) {
//
//        if object is AVPlayerItem {
//            switch keyPath {
//            case "playbackBufferEmpty":
//                // Show loader
//                self.notifyOnShowBuffer()
//            case "playbackLikelyToKeepUp":
//                // Hide loader
//                self.notifyOnHideBuffer()
//            case "playbackBufferFull":
//                // Hide loader
//                self.notifyOnHideBuffer()
//            default:
//                break
//            }
//        }
//    }

    open func togglePlayPause() {
        if self.isPlaying {
            self.pause()
        } else {
            self.play()
        }
    }
    
    open func play() {
        
        if self.podcast == nil {
            return
        }
        
        if self.elapsedTime + 1 >= self.totalDuration {
            self.seekTo(0.0)
        }
        
        let analyticType: AppAnalytic =
            self.podcast!.isDownloaded ? .playedDownloadedPodcast : .playedOnlinePodcast
        APIClient.recordAnalytic(analyticType,
                                 variable: self.podcast!.postId,
                                 secondaryVariable: "\(self.elapsedTime)")
        
        self.audioPlayer?.play()
        self.updateNowPlayingInfoElapsedTime()
        self.notifyOnPlaybackStateChanged()
    }
    
    open func pause() {
        
        if self.podcast == nil {
            return
        }
        
        let analyticType: AppAnalytic =
            self.podcast!.isDownloaded ? .stoppedDownloadedPodcast : .stoppedOnlinePodcast
        APIClient.recordAnalytic(analyticType,
                                 variable: self.podcast!.postId,
                                 secondaryVariable: "\(self.elapsedTime)")
        
        self.audioPlayer?.pause()
        self.updateNowPlayingInfoElapsedTime()
        self.notifyOnPlaybackStateChanged()
    }
    
    func rewind() {
        
        if self.isPreview {
            return
        }

        if let current = self.audioPlayer?.currentTime() {

            let wasPlaying = self.isPlaying
            self.audioPlayer?.pause()

            let currentTime = CMTimeGetSeconds(current)
            var newTime = currentTime - 15

            if newTime < 0 {
                newTime = 0
            }

            let time2: CMTime = CMTimeMake(value: Int64(newTime * 1000 as Float64),
                                           timescale: 1000)
            self.audioPlayer?.seek(to: time2)

            if wasPlaying {
                self.audioPlayer?.play()
            }

            APIClient.recordAnalytic(.skippedBackwardsPlayer,
                                     variable: self.podcast!.postId,
                                     secondaryVariable: String(newTime))

            self.updateNowPlayingInfoElapsedTime()
        }
    }
    
    func skip() {
        guard let duration  = self.audioPlayer?.currentItem?.asset.duration else{
            return
        }
        
        if self.isPreview {
            return
        }
        
        let wasPlaying = self.isPlaying
        self.audioPlayer?.pause()
        
        let currentTime = CMTimeGetSeconds((self.audioPlayer?.currentTime())!)
        let newTime = currentTime + 15
        
        if newTime < CMTimeGetSeconds(duration) {
            let time2: CMTime = CMTimeMake(value: Int64(newTime * 1000 as Float64),
                                           timescale: 1000)
            self.audioPlayer?.seek(to: time2)
            
            if wasPlaying {
                self.audioPlayer?.play()
            }
            
            APIClient.recordAnalytic(.skippedForwardPlayer,
                                     variable: self.podcast!.postId,
                                     secondaryVariable: String(newTime))
        }
        
        self.updateNowPlayingInfoElapsedTime()
    }
    
    func seekTo(_ point: Double) {
        DispatchQueue.main.async { [weak self] in
            let newTime: CMTime = CMTimeMake(value: Int64(point * 1000 as Float64), timescale: 1000)
            self?.audioPlayer?.seek(to: newTime)
            self?.updateNowPlayingInfoElapsedTime()
        }
    }
    
    // MARK: - Command Center -
    
    func configureCommandCenter() {
        
        self.commandCenter.skipBackwardCommand.isEnabled = true
        self.commandCenter.skipBackwardCommand.preferredIntervals = [NSNumber(value: 15)]
        self.commandCenter.skipForwardCommand.isEnabled = true
        self.commandCenter.skipForwardCommand.preferredIntervals = [NSNumber(value: 15)]
        self.commandCenter.togglePlayPauseCommand.isEnabled = true
        
        self.commandCenter.togglePlayPauseCommand.addTarget (handler: { [weak self] event -> MPRemoteCommandHandlerStatus in
            guard let sself = self else { return .commandFailed }
            sself.togglePlayPause()
            return .success
        })

        self.commandCenter.playCommand.addTarget (handler: { [weak self] event -> MPRemoteCommandHandlerStatus in
            guard let sself = self else { return .commandFailed }
            sself.play()
            return .success
        })

        self.commandCenter.pauseCommand.addTarget (handler: { [weak self] event -> MPRemoteCommandHandlerStatus in
            guard let sself = self else { return .commandFailed }
            sself.pause()
            return .success
        })
        self.commandCenter.skipBackwardCommand.addTarget (handler: { [weak self] event -> MPRemoteCommandHandlerStatus in
            guard let sself = self else { return .commandFailed }
            sself.rewind()
            return .success
        })
        self.commandCenter.skipForwardCommand.addTarget (handler: { [weak self] event -> MPRemoteCommandHandlerStatus in
            guard let sself = self else { return .commandFailed }
            sself.skip()
            return .success
        })
    }
    
    // MARK: - Now Playing Info -
    
    func updateNowPlayingInfoForCurrentPlaybackItem() {
        guard let currentPlaybackItem = self.podcast else {
            self.configureNowPlayingInfo(nil)
            return
        }
        
        var nowPlayingInfo = [MPMediaItemPropertyTitle: currentPlaybackItem.title,
                              MPMediaItemPropertyArtist: "The Anfield Wrap",
                              MPNowPlayingInfoPropertyElapsedPlaybackTime: NSNumber(value: self.elapsedTime),
                              MPMediaItemPropertyPlaybackDuration: NSNumber(value: self.totalDuration),
                              MPNowPlayingInfoPropertyPlaybackRate: NSNumber(value: 1.0 as Float),
                              MPNowPlayingInfoPropertyMediaType: NSNumber(value: MPNowPlayingInfoMediaType.audio.rawValue)] as [String : Any]
        
        if let url = URL(string: currentPlaybackItem.mediumImage) as? URL {
            KingfisherManager.shared.retrieveImage(with: url,
                                                   options: [],
                                                   progressBlock: nil) { (image, error, cache, url) in
                                                    if error == nil {
                                                        nowPlayingInfo[MPMediaItemPropertyArtwork] = MPMediaItemArtwork.init(boundsSize: image!.size,
                                                                                                                             requestHandler: { (size) -> UIImage in
                                                                                                                                return image!
                                                        })
                                                    }
            }
        }
        
        self.configureNowPlayingInfo(nowPlayingInfo)
        self.updateNowPlayingInfoElapsedTime()
    }
    
    func updateNowPlayingInfoElapsedTime() {
        guard var nowPlayingInfo = self.nowPlayingInfo else { return }
        nowPlayingInfo[MPNowPlayingInfoPropertyElapsedPlaybackTime] = NSNumber(value: self.elapsedTime)
        self.configureNowPlayingInfo(nowPlayingInfo)
    }
    
    func configureNowPlayingInfo(_ nowPlayingInfo: [String: Any]?) {
        self.nowPlayingInfoCenter.nowPlayingInfo = nowPlayingInfo
        self.nowPlayingInfo = nowPlayingInfo
    }
    
    @objc func endPlayback() {
        self.pause()
    }
    
    func removePodcast() {
        
        if self.podcast != nil {
            let analyticType: AppAnalytic =
                self.podcast!.isDownloaded ? .stoppedDownloadedPodcast : .stoppedOnlinePodcast
            APIClient.recordAnalytic(analyticType,
                                     variable: self.podcast!.postId,
                                     secondaryVariable: "\(self.elapsedTime)")
            
            if self.isPreview && timeObserver != nil {
                self.audioPlayer?.removeTimeObserver(timeObserver!)
            }

            if self.periodicObserver != nil {
                self.audioPlayer?.removeTimeObserver(self.periodicObserver!)
            }
        }
        
        self.podcast = nil
        self.audioPlayer = nil
        
        self.updateNowPlayingInfoForCurrentPlaybackItem()
        self.notifyOnTrackChanged()
    }
    
    // MARK: - Convenience
    
    func notifyOnPlaybackStateChanged() {
        self.notificationCenter.post(name: Notification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged),
                                     object: self)
    }
    
    func notifyOnTrackChanged() {
        self.notificationCenter.post(name: Notification.Name(rawValue: PBAudioPlayerOnTrackChanged),
                                     object: self)
    }
    
    func notifyOnShowBuffer() {
        self.notificationCenter.post(name: Notification.Name(rawValue: PBAudioPlayerOnShowBuffering),
                                     object: self)
    }
    
    func notifyOnHideBuffer() {
        self.notificationCenter.post(name: Notification.Name(rawValue: PBAudioPlayerOnHideBuffering),
                                     object: self)
    }
    
    func notifyOnPreviewFinished() {
        self.notificationCenter.post(name: Notification.Name(rawValue: PBAudioPlayerOnPreviewFinished),
                                     object: self)
    }
}

// MARK: - AVAudioPlayerDelegate

extension PBAudioPlayer: AVAudioPlayerDelegate {
    
    open func audioPlayerDidFinishPlaying(_ player: AVAudioPlayer, successfully flag: Bool) {
        self.endPlayback()
    }
    
    open func audioPlayerBeginInterruption(_ player: AVAudioPlayer) {
        self.notifyOnPlaybackStateChanged()
    }
    
    open func audioPlayerEndInterruption(_ player: AVAudioPlayer, withOptions flags: Int) {
        if AVAudioSession.InterruptionOptions(rawValue: UInt(flags)) == .shouldResume {
            self.play()
        }
    }
}
