//
//  TAWPodcastPlayerViewController.swift
//  TAW
//
//  Created by Andrew Steven on 14/01/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import UIKit
import AVFoundation
import MediaPlayer
import RealmSwift
import UICircularProgressRing

class TAWPodcastPlayerViewController: UIViewController, UIGestureRecognizerDelegate {
    
    // MARK: - Interface outlets -
    
    @IBOutlet weak var closeButton: UIButton!
    @IBOutlet weak var downloadButton: UIButton!
    @IBOutlet weak var podcastImageShadow: PBButton!
    @IBOutlet weak var podcastImageView: UIImageView!
    @IBOutlet weak var podcastTitleLabel: UILabel!
    @IBOutlet weak var audioSlider: PBSlider!
    @IBOutlet weak var elapsedDurationLabel: UILabel!
    @IBOutlet weak var totalDurationLabel: UILabel!
    @IBOutlet weak var playButton: UIButton!
    @IBOutlet weak var rewindButton: UIButton!
    @IBOutlet weak var skipButton: UIButton!
    @IBOutlet weak var downloadProgressRing: UICircularProgressRing!
    @IBOutlet weak var bufferingIndicator: UIActivityIndicatorView!
    
    // MARK: - Variables -
    
    var timer = Timer()
    var isPreview = false
    var newPodcast: Podcast?
    var notificationCenter = NotificationCenter.default
    let downloadService = DownloadService.shared
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        self.configureNotifications()
        
        let isSmallDevice: Bool = UIDevice.isSmallDevice
        self.podcastTitleLabel.font = isSmallDevice ? TAW.Fonts.dinAlternate21 : TAW.Fonts.dinAlternate27
        self.elapsedDurationLabel.font = isSmallDevice ? TAW.Fonts.dinAlternate14 : TAW.Fonts.dinAlternate17
        self.totalDurationLabel.font = isSmallDevice ? TAW.Fonts.dinAlternate14 : TAW.Fonts.dinAlternate17
        self.downloadProgressRing.font = TAW.Fonts.dinAlternate10!
        self.downloadProgressRing.ringStyle = .ontop
        
        let swipeGesture = UISwipeGestureRecognizer(target: self, action: #selector(didSwipeView(swipeGesture:)))
        swipeGesture.delegate = self
        swipeGesture.direction = .down
        self.view.addGestureRecognizer(swipeGesture)

        self.podcastImageView.transform = CGAffineTransform(scaleX: 0.7, y: 0.7)
        self.podcastImageShadow.transform = CGAffineTransform(scaleX: 0.7, y: 0.7)
        self.podcastImageShadow.alpha = 0.0
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        if let theNewPodcast = self.newPodcast {
            if !theNewPodcast.isInvalidated {
                PBAudioPlayer.shared.playPodcast(theNewPodcast, isPreview: self.isPreview)
                PBAppStoreReviewer.incrementKeyActivity()
                self.startTimer()
            }
        } else if PBAudioPlayer.shared.isPlaying {
            self.startTimer()
        }
        
        self.setupView()
        self.updateButtons()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        super.viewDidDisappear(animated)
        self.timer.invalidate()
    }
    
    private func setupView() {
        if self.newPodcast != nil {
            self.showBuffering()
            self.displayPodcastDetails(self.newPodcast!)
        } else if let podcast = PBAudioPlayer.shared.podcast {
            self.hideBuffering()
            self.displayPodcastDetails(podcast)
        }
        
        if self.isPreview {
            self.skipButton.isEnabled = false
            self.rewindButton.isEnabled = false
            self.audioSlider.isEnabled = false
            self.downloadButton.isHidden = true
            self.skipButton.alpha = 0.5
            self.rewindButton.alpha = 0.5
            self.audioSlider.alpha = 0.5
        }
    }
    
    private func displayPodcastDetails(_ podcast: Podcast) {

        DispatchQueue.main.async {
            self.podcastTitleLabel.text = podcast.title.htmlDecoded
        }
        
        let image = podcast.largeImage
        self.podcastImageView.kf.setImage(with: URL(string: image),
                                          placeholder: UIImage(named: "placeholder_podcasts"),
                                          options: [.transition(.fade(0.2))])
        
        self.downloadButton.isHidden = podcast.isDownloaded
        
        if LocalStorage.shared.user!.guest == true || self.isPreview {
            self.downloadButton.isHidden = true
        }
        
        let elapsed = PBAudioPlayer.shared.elapsedTime
        let total = PBAudioPlayer.shared.totalDuration
        
        if elapsed == 0 {
            self.audioSlider.value = Float(0)
            self.elapsedDurationLabel.text = "0:00:00"
        } else {
            let percentage: Double = (elapsed/total) * 100
            self.audioSlider.value = Float(percentage)
            self.elapsedDurationLabel.text = PBAudioPlayer.shared.getElapsedTime(.positional)
        }
        self.totalDurationLabel.text = PBAudioPlayer.shared.getDuration(.positional)
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        if let postId = PBAudioPlayer.shared.podcast?.postId {
            APIClient.recordAnalytic(.minimisedPlayer, variable: postId, secondaryVariable: "1")
        }
        
        if self.isPreview {
            PBAudioPlayer.shared.removePodcast()
        }
    }
    
    @objc func didSwipeView(swipeGesture: UISwipeGestureRecognizer) {
        self.close()
    }
    
    @objc private func updateButtons() {
        let isPlaying = PBAudioPlayer.shared.isPlaying
        let image = isPlaying ? UIImage(named: "player_pause_button") : UIImage(named: "player_play_button")
        playButton.setImage(image, for: .normal)

        let scale: CGFloat = isPlaying ? 1.0 : 0.7
        let alpha: CGFloat = isPlaying ? 1.0 : 0.0

        UIView.animate(withDuration: 0.3) {
            self.podcastImageView.transform = CGAffineTransform(scaleX: scale, y: scale)
            self.podcastImageShadow.transform = CGAffineTransform(scaleX: scale, y: scale)
            self.podcastImageShadow.alpha = alpha
        }
    }
    
    @objc func updateProgressView() {
        
        let elapsed = PBAudioPlayer.shared.elapsedTime
        let total = PBAudioPlayer.shared.totalDuration
        
        if elapsed == 0 {
            self.audioSlider.value = Float(0)
            self.elapsedDurationLabel.text = "0:00:00"
        } else {
            let percentage: Double = (elapsed/total) * 100
            self.audioSlider.value = Float(percentage)
            self.elapsedDurationLabel.text = PBAudioPlayer.shared.getElapsedTime(.positional)
        }
        
        self.totalDurationLabel.text = PBAudioPlayer.shared.getDuration(.positional)
        
        if let podcast = PBAudioPlayer.shared.podcast {
            let userId = LocalStorage.shared.user?.userId ?? ""
            do {
                let realm = try Realm()
                try realm.write {
                    realm.add(PodcastPosition(value: ["postId": podcast.postId, "position": elapsed, "userId": userId]), update: .all)
                }
            } catch let error {
                print("Failed to update podcast position: \(error.localizedDescription)")
            }
        }
    }
    
    @objc public func updatedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            
            if download.podcast == PBAudioPlayer.shared.podcast {
                
                let value: CGFloat = CGFloat(download.progress)
                DispatchQueue.main.async {
                    self.downloadProgressRing.startProgress(to: value, duration: 0)
                }
            }
        }
    }
    
    @objc public func completedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            if download.podcast.title == PBAudioPlayer.shared.podcast?.title {
                DispatchQueue.main.async {
                    self.downloadProgressRing.isHidden = true
                }
            }
        }
    }
    
    @objc public func failedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            if download.podcast.title == PBAudioPlayer.shared.podcast?.title {
                DispatchQueue.main.async {
                    self.downloadButton.isHidden = false
                    self.downloadProgressRing.isHidden = true
                    self.downloadProgressRing.startProgress(to: 0, duration: 0)
                    
                    self.showAlertWithTitle("Download Failed", message: "There was an error trying to download \(download.podcast.title). Try again later")
                }
            }
        }
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        
        // Play button state notifications
        self.notificationCenter.addObserver(self,
                                            selector: #selector(updateButtons),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(updateButtons),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged),
                                            object: nil)
        
        // Download podcast notifications
        self.notificationCenter.addObserver(self,
                                            selector: #selector(updatedDownload(notification:)),
                                            name: NSNotification.Name(rawValue: downloadServiceUpdatedNotification),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(completedDownload(notification:)),
                                            name: NSNotification.Name(rawValue: downloadServiceCompletedNotification),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(failedDownload(notification:)),
                                            name: NSNotification.Name(rawValue: downloadServiceFailedNotification),
                                            object: nil)
        
        // Buffering notifications
        self.notificationCenter.addObserver(self,
                                            selector: #selector(showBuffering),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnShowBuffering),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(hideBuffering),
                                            name: NSNotification.Name(rawValue: PBAudioPlayerOnHideBuffering),
                                            object: nil)
        
        // Preview Finished notification
        self.notificationCenter.addObserver(self,
                                            selector: #selector(displayUnlockToHearMore),
                                            name: NSNotification.Name(PBAudioPlayerOnPreviewFinished),
                                            object: nil)
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    // MARK: - Buffering -
    
    @objc private func showBuffering() {
        DispatchQueue.main.async {
            self.playButton.isHidden = true
            self.bufferingIndicator.startAnimating()
        }
    }
    
    @objc private func hideBuffering() {
        DispatchQueue.main.async {
            self.playButton.isHidden = false
            self.bufferingIndicator.stopAnimating()
        }
    }
    
    // MARK: - Timer -
    
    func startTimer() {
        self.timer = Timer.scheduledTimer(timeInterval: 1.0,
                                          target: self,
                                          selector: #selector(self.updateProgressView),
                                          userInfo: nil,
                                          repeats: true)
    }
    
    // MARK: - PBSlider -
    
    @IBAction func audoSliderValueChanged(_ sender: UISlider) {
        print("Slider Value: \(sender.value)")
        self.timer.invalidate()
        let total = Float(PBAudioPlayer.shared.totalDuration)
        let elapsed: Float = (sender.value/100) * total
        PBAudioPlayer.shared.seekTo(Double(exactly: elapsed)!)
        self.elapsedDurationLabel.text = PBAudioPlayer.shared.getElapsedTime(.positional)
        self.startTimer()
    }
    
    // MARK: - Button actions -
    
    @IBAction func close() {
        self.dismiss(animated: true, completion: nil)
    }
    
    @IBAction func download() {
        
        guard let podcast = PBAudioPlayer.shared.podcast else {
            return
        }
        
        self.downloadButton.isHidden = true
        self.downloadProgressRing.startProgress(to: 0, duration: 0)
        self.downloadProgressRing.isHidden = false
        self.downloadService.startDownload(podcast)
    }
    
    @IBAction func play() {
        PBAudioPlayer.shared.togglePlayPause()
        self.updateButtons()
        
        if PBAudioPlayer.shared.isPlaying {
            self.startTimer()
        } else {
            self.timer.invalidate()
        }
    }
    
    @IBAction func rewind() {
        PBAudioPlayer.shared.rewind()
        self.elapsedDurationLabel.text = PBAudioPlayer.shared.getElapsedTime(.positional)
    }
    
    @IBAction func skip() {
        PBAudioPlayer.shared.skip()
        self.elapsedDurationLabel.text = PBAudioPlayer.shared.getElapsedTime(.positional)
    }
    
    @objc private func displayUnlockToHearMore() {
        if let tawVC = instantiateVC(identifier: "unlockVC") as? TAWUnlockViewController {
            if let podcast = PBAudioPlayer.shared.podcast {
                tawVC.podcast = podcast
            }
            self.navigationController?.pushViewController(tawVC, animated: false)
        }
    }
}
