//
//  TAWPodcastsViewController.swift
//  TAW
//
//  Created by Andrew Steven on 19/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Spruce
import RealmSwift
import AVFoundation
import MediaPlayer
import OneSignal
import SkeletonView

class TAWPodcastsViewController: PBAnimatedViewController, TAWAlertViewDelegate {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var headerContainerView: PBView!
    @IBOutlet weak var headerView: UIView!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var headerViewHeight: NSLayoutConstraint!
    @IBOutlet weak var tableView: UITableView!
    @IBOutlet weak var offlineView: UIView!
    @IBOutlet weak var offlineViewHeight: NSLayoutConstraint!
    @IBOutlet weak var contentViewPaddingBottom: NSLayoutConstraint!
    @IBOutlet weak var downloadsButton: UIButton!
    
    // MARK: - Variables -
    
    var playerView: PBMiniPlayerView?
    var notificationCenter = NotificationCenter.default
    
    var podcasts: [Podcast] = []
    var isOffline = false
    private let pullToRefresh = UIRefreshControl()
    
    // Paging
    var fetchedAllPodcasts = false
    var currentPage: Int = 1
    let pageSize: Int = 20
    var isLoading: Bool = false
    
    // Downloads
    let downloadService = DownloadService.shared
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        self.tableView.refreshControl = pullToRefresh
        self.tableView.tableFooterView = UIView()
        
        pullToRefresh.addTarget(self, action: #selector(forceRefresh), for: .valueChanged)
        pullToRefresh.tintColor = TAW.Colors.pullToRefreshColor
        pullToRefresh.attributedTitle = NSAttributedString(string: "LOADING",
                                                           attributes: [
                                                            NSAttributedString.Key.foregroundColor: TAW.Colors.pullToRefreshColor,
                                                            NSAttributedString.Key.font: TAW.Fonts.dinAlternate10!
            ])
        
        let swipeGesture = UISwipeGestureRecognizer(target: self, action: #selector(forceRefresh))
        swipeGesture.direction = .down
        swipeGesture.delegate = self
        self.offlineView.addGestureRecognizer(swipeGesture)
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 60 : 0
        self.view.layoutIfNeeded()
        
        self.configureNotifications()
        
        // Check how many times the user has opened the app and prompt user to rate the App
        PBAppStoreReviewer.checkAppOpenedAndAskForReview()
        
    }
    
    override func viewWillAppear(_ animated: Bool) {

        super.viewWillAppear(animated)

        self.tableView.isSkeletonable = true
        self.downloadsButton.isHidden = LocalStorage.shared.user?.guest == true
        self.tabBarController?.tabBar.isHidden = false

        if LocalStorage.shared.loggedIn {
            self.checkUserSubscription()
            self.refreshData(forceRefresh: false)
//
           
        }
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        
        if PBAudioPlayer.shared.podcast != nil {
            if self.playerView == nil {
                let window = UIApplication.shared.keyWindow
                let bottomPadding: CGFloat = window?.safeAreaInsets.bottom ?? 0
                
                let height: CGFloat = self.navigationController?.tabBarController?.tabBar.isHidden ?? true ? 60 + bottomPadding : 60
                self.playerView = PBMiniPlayerView(frame: CGRect(x: 0, y: self.view.frame.size.height - height, width: self.view.frame.size.width, height: height))
                self.playerView!.delegate = self
                self.view.addSubview(playerView!)
            }
            self.playerView?.updateProgressView()
        }
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        if self.playerView != nil {
            self.playerView?.stopTimer()
            self.playerView?.removeFromSuperview()
            self.playerView = nil
        }
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(removeData), name: NSNotification.Name(rawValue: TAWUserHasLoggedOutNotification), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(removeData), name: NSNotification.Name(rawValue: TAWDownloadRemovedNotification), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(removeData), name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(self.checkUserSubs), name: NSNotification.Name(rawValue: CheckFreeSubscription), object: nil)
        
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    @objc func removeData() {
        self.podcasts = []
        self.currentPage = 1
        self.fetchedAllPodcasts = false
        self.tableView.reloadData()
    }
    
     @objc func checkUserSubs(){
         self.checkFreeSubs(alertDelegate: self)
    }
    
        func freeSubsExpired()
        {
           // showHowToSubscribe()
            self.showSubscriptionView(isFreeSubExpired: true)
        }
    
        func acceptedNotifications() {
    
        }
    
        func unlinkAccount() {
    
        }
    
        func accountCreated() {
    
        }
    
        func didTapLogout() {
    
        }

    
    @objc func podcastStateChanged() {
        for case let cell as TAWPodcastTableViewCell in self.tableView.visibleCells  {
            cell.updateView()
        }
        
        if PBAudioPlayer.shared.podcast == nil {
            if self.playerView != nil {
                self.playerView?.stopTimer()
                self.playerView?.removeFromSuperview()
                self.playerView = nil
            }
        }
        
        self.contentViewPaddingBottom.constant = PBAudioPlayer.shared.podcast != nil ? 60 : 0
        self.view.layoutIfNeeded()
    }
    
    // MARK: - Button actions -
    
    @IBAction func showDownloads() {
        self.showUserDownloads()
    }
    
    // MARK: - Data fetching -
    
    @objc func forceRefresh() {
        self.checkUserSubscription()
        self.refreshData(forceRefresh: true)
    }
    
    @objc func refreshData(forceRefresh: Bool) {
        
        if !Connectivity.isConnectedToInternet {
            
            // Show "No Internet Connection" banner for 3 seconds
            PBBannerView.shared.showBanner(inView: (self.navigationController?.view)!, withTitle: "No internet connection", style: .noInternet)
            self.pullToRefresh.endRefreshing()
            
            // Show downloaded podcasts when offline
            self.isOffline = true
            
            if self.podcasts.count == 0 {
                self.getDownloads()
                self.offlineViewHeight.constant = self.podcasts.count == 0 ? self.view.frame.size.height - headerContainerView.frame.size.height : 250
                self.view.layoutIfNeeded()
                self.pullToRefresh.endRefreshing()
            }
            else {
                self.pullToRefresh.endRefreshing()
            }
        }
        else {
            // Hide downloaded podcasts when connected to internet
            self.isOffline = false
            self.offlineViewHeight.constant = 0
            self.view.layoutIfNeeded()
            
            // Check whether we have already fetched podcasts in the last 15 mins
            if self.podcasts.count == 0 {
                // Fetch latest podcasts with skeleton animation
                self.getPodcasts(animated: true)
            } else if self.podcastsNeedFetching() || forceRefresh {
                // Fetch latest podcasts whilst keeping the existing podcasts
                self.getPodcasts(animated: false)
            } else {
                // Refresh the existing podcasts UI incase the user has just purchased a podcast
                self.tableView.reloadData()
                self.podcastStateChanged()
                self.pullToRefresh.endRefreshing()
            }
        }
    }
    
    func podcastsNeedFetching() -> Bool {
        
        let lastFetched = LocalStorage.shared.podcastsLastFetched
        if lastFetched == nil {
            LocalStorage.shared.podcastsLastFetched = Date()
            return true
        }
        else {
            let calendar = Calendar.current
            let now = Date()
            let components:DateComponents = (calendar as NSCalendar).components([NSCalendar.Unit.minute], from: lastFetched!, to: now, options: NSCalendar.Options())
            
            if components.minute! >= 15 {
                return true
            }
            else {
                return false
            }
        }
    }

    func isLoadingCell(for indexPath: IndexPath) -> Bool {
        return indexPath.row >= self.podcasts.count
    }
    
    // MARK: - Offline functionality -
    
    private func getDownloads() {
        
        // Fetch downloaded podcasts
        let realm = try! Realm()
        let currentDownloads = realm.objects(Podcast.self).sorted(byKeyPath: "createdDate")
        let temp = Array(currentDownloads)
        self.podcasts = temp.filter {$0.isDownloaded == true}

        DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.3) {
            self.tableView.reloadData()
            self.animateTableView(self.tableView)
            self.pullToRefresh.endRefreshing()
        }
    }
    
    // MARK: - API client -
    
    private func getPodcasts(animated: Bool) {
        
        if !self.isLoading {
            self.isLoading = true
            if animated {
                self.tableView.showAnimatedGradientSkeleton()
            }

            // Fetch the latest podcasts from API
            APIClient.getPodcasts(page: 1, number: self.pageSize) { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    if let feed = jsonResponse["feed"] as? [[String: Any]] {

                        // Reset podcasts
                        self.currentPage = 1
                        self.podcasts = []
                        self.fetchedAllPodcasts = false

//                        print("Podcast feed: \(feed)")
                        let realm = try! Realm()
                        for json in feed {
                            if let podcast = Podcast(JSON: json) {
                                // The line below is for manually testing various subscription levels
//                                podcast.subLevel = "0"
                                self.podcasts.append(podcast)
                                
                                try! realm.write {
                                    realm.add(podcast, update: .all)
                                }
                            }
                        }
                        
                        if self.podcasts.count != self.pageSize {
//                            print("Reached bottom of podcasts feed")
                            self.fetchedAllPodcasts = true
                        }
                        
                        LocalStorage.shared.podcastsLastFetched = Date()
                        
                        DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.7, execute: {
                            self.tableView.reloadData()
                            if animated {
                                self.tableView.hideSkeleton()
                            }
                            self.pullToRefresh.endRefreshing()
                            self.isLoading = false
                        })
                    }
                } else {
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.pullToRefresh.endRefreshing()
                        self.isLoading = false
                        if animated {
                            self.tableView.hideSkeleton()
                        }
                        self.invalidateSession()
                    }
                    else {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                        self.pullToRefresh.endRefreshing()
                        self.isLoading = false
                        if animated {
                            self.tableView.hideSkeleton()
                        }
                    }
                }
            }
        }
    }
    
    private func getNextBatch() {
        
        if !self.fetchedAllPodcasts && !self.isLoading {
            
            self.isLoading = true
            self.currentPage += 1
            
            APIClient.getPodcasts(page: self.currentPage, number: self.pageSize) { (jsonResponse, error) in
                
                if jsonResponse["success"] as? Bool == true {
                    
                    var newPodcasts: [Podcast] = []
                    
                    if let feed = jsonResponse["feed"] as? [[String: Any]] {
                        let realm = try! Realm()
                        for json in feed {
                            if let podcast = Podcast(JSON: json) {
                                newPodcasts.append(podcast)
                                
                                try! realm.write {
                                    realm.add(podcast, update: .all)
                                }
                            }
                        }
                        
                        if newPodcasts.count != self.pageSize {
//                            print("REACHED BOTTOM OF PODCASTS FEED")
                            self.fetchedAllPodcasts = true
                        }

                        self.tableView.performBatchUpdates({
                            
                            let currentCount = self.podcasts.count
                            self.podcasts += newPodcasts
                            
                            var indexPaths: [IndexPath] = []
                            
                            for (index, _) in newPodcasts.enumerated() {
                                let indexPath = IndexPath(row: index + currentCount, section: 0)
                                indexPaths.append(indexPath)
                            }
                            
                            self.tableView.insertRows(at: indexPaths, with: .automatic)

                        }, completion: { (finished) in
                            if finished {
                                self.isLoading = false
                            }
                        })
                    }
                }
                else {
                    self.isLoading = false
                }
            }
        }
    }
    
   
    
    private func checkUserSubscription() {
        
        if Connectivity.isConnectedToInternet {
            APIClient.checkSubscription { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    if let userJson = jsonResponse["user"] as? [String: Any] {
                        if let user = User(JSON: userJson) {
                            let needsUpdate = user.subscriptionLevel != LocalStorage.shared.user?.subscriptionLevel
                            user.authToken = LocalStorage.shared.user?.authToken
                            LocalStorage.shared.user = user
                            self.checkUserSubs()
                           // self.showFreeSubsExpiredAlert()
                            if needsUpdate {
                                NotificationCenter.default.post(name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: self)
                                self.forceRefresh()
                            }
                           
                        }
                    }
                }
                else {
                    let user = LocalStorage.shared.user
                    let needsUpdate = user?.subscriptionLevel != SubscriptionType.none
                    user?.subscriptions = []
                    LocalStorage.shared.user = user
                  //  self.showFreeSubsExpiredAlert()
                   
                    if needsUpdate {
                        NotificationCenter.default.post(name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: self)
                        self.forceRefresh()
                    }

                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    }
                    else if message == "User is not currently subscribed"
                    {
                        self.showSubscriptionView(isFreeSubExpired: false)
                    }
                }
            }
            
        }
    }
    
    
    
    var maxHeaderHeight: CGFloat = 100
    var minHeaderHeight: CGFloat = 70
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let offset = scrollView.contentOffset.y + scrollView.contentInset.top
        
        if offset > 0 && offset < maxHeaderHeight {
            
            if offset <= maxHeaderHeight/4 {
                let alpha: CGFloat = 1 - (offset / (maxHeaderHeight/4))
                for subview in headerView.subviews {
                    subview.alpha = alpha
                    titleLabel.alpha = 0
                }
            }
            else {
                let alpha: CGFloat = 1 - (offset / maxHeaderHeight)
                for subview in headerView.subviews {
                    subview.alpha = 0
                    titleLabel.alpha = 1 - alpha
                }
            }
            
            let percentage: CGFloat = 1 - (offset/maxHeaderHeight)
            headerViewHeight.constant = ((maxHeaderHeight - minHeaderHeight) * percentage) + minHeaderHeight
        }
        else if offset <= 0 {
            headerViewHeight.constant = maxHeaderHeight
            
            for subview in headerView.subviews {
                subview.alpha = 1
                titleLabel.alpha = 0
            }
        }
        else if offset >= maxHeaderHeight {
            headerViewHeight.constant = minHeaderHeight
            
            for subview in headerView.subviews {
                subview.alpha = 0
                titleLabel.alpha = 1
            }
        }
    }
}

// MARK: - UITableViewDelegate -

extension TAWPodcastsViewController: UITableViewDelegate {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        if Connectivity.isConnectedToInternet {
            return self.podcasts.count + 1
        } else {
            return self.podcasts.count
        }
    }
    
    func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> UIView? {
        let view = PBView(frame: CGRect(x: 0, y: 0, width: self.view.frame.size.width, height: 30))
        view.backgroundColor = .white
        view.cornerRadius = 0
        view.shadowColor = UIColor.black.withAlphaComponent(0.2)
        view.shadowOffset = CGSize(width: 0, height: 4)
        view.shadowRadius = 2
        
        let label = UILabel(frame: CGRect(x: 20, y: 0, width: self.view.frame.size.width - 64, height: view.frame.size.height))
        label.text = "Downloaded Podcasts:"
        label.font = TAW.Fonts.dinAlternate21
        label.textColor = .black
        view.addSubview(label)
        return view
    }
    
    func tableView(_ tableView: UITableView, heightForHeaderInSection section: Int) -> CGFloat {
        return self.isOffline ? 30 : 0
    }
    
    func tableView(_ tableView: UITableView, willDisplay cell: UITableViewCell, forRowAt indexPath: IndexPath) {
        
        if !self.isLoading && indexPath.row >= self.podcasts.count && Connectivity.isConnectedToInternet && self.podcasts.count != 0 {
            self.getNextBatch()
        }

        cell.spruce.prepare(with: self.tableViewAnimations)
        var animation = SpringAnimation(duration: 0.7)
        animation.animationOptions = [UIView.AnimationOptions.allowUserInteraction]

        cell.spruce.animate(self.tableViewAnimations, animationType: animation, sortFunction: self.tableViewSortFunction!)
        cell.layoutSubviews()
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {

        if indexPath.row >= self.podcasts.count {
            let cell = tableView.dequeueReusableCell(withIdentifier: "podcastLoadingCell", for: indexPath) as! TAWLoadingTableViewCell
            if self.fetchedAllPodcasts {
                cell.titleLabel.text = "All caught up 👍"
                cell.activityIndicatorView.stopAnimating()
                cell.activityIndicatorView.isHidden = true
            } else {
                cell.titleLabel.text = "Loading more podcasts"
                cell.activityIndicatorView.startAnimating()
                cell.activityIndicatorView.isHidden = false
            }
            return cell
        } else {
            let identifier = indexPath.row == 0 && !self.isOffline ? "featuredPodcastCell" : "podcastCell"
            let cell: TAWPodcastTableViewCell = tableView.dequeueReusableCell(withIdentifier: identifier, for: indexPath) as! TAWPodcastTableViewCell
            if indexPath.row < self.podcasts.count {
                if self.podcasts[indexPath.row].isInvalidated {
                    self.forceRefresh()
                } else {
                    cell.delegate = self
                    let podcast = self.podcasts[indexPath.row]
                    let imageQuality: PodcastImageQuality = indexPath.row == 0 ? .large : .medium

                    cell.set(podcast: podcast,
                             imageQuality: imageQuality,
                             displayMode: .feed,
                             delegate: self,
                             download: self.downloadService.activeDownloads[podcast.safeURL])
                }
            }

            return cell
        }
    }
    
    func tableView(_ tableView: UITableView, estimatedHeightForRowAt indexPath: IndexPath) -> CGFloat {
        if indexPath.row == 0 && Connectivity.isConnectedToInternet {
            return 320
        } else {
            return 114
        }
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        if indexPath.row == 0 && Connectivity.isConnectedToInternet {
            return 320
        } else {
            return 114
        }
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        if self.podcasts.count <= indexPath.row {
            return
        }
        else if self.podcasts[indexPath.row].isInvalidated {
            self.forceRefresh()
            return
        }
        
        let podcast = self.podcasts[indexPath.row]
        self.performSegue(withIdentifier: "showPodcastDetail", sender: podcast)
        DispatchQueue.main.async {}
    }
    
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "showPodcastDetail" {
            if let vc = segue.destination as? TAWPodcastDetailViewController {
                vc.podcast = sender as? Podcast
            }
        }
    }
}

// MARK: - SkeletonTableViewDataSource -

extension TAWPodcastsViewController: SkeletonTableViewDataSource {
    
    func numSections(in collectionSkeletonView: UITableView) -> Int {
        return 1
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return UIDevice.isSmallDevice ? 1 : 3
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, cellIdentifierForRowAt indexPath: IndexPath) -> ReusableCellIdentifier {
        let identifier = indexPath.row == 0 && !self.isOffline ? "featuredPodcastCell" : "podcastCell"
        return identifier
    }
}

extension TAWPodcastsViewController: PBMiniPlayerViewDelegate {

    func openPlayer() {
        if let tawVC = instantiateVC(identifier: "podcastPlayerVC") as? TAWPodcastPlayerViewController {
            self.navigationController?.present(tawVC, animated: true, completion: nil)
        }
    }
}

// MARK: - PodcastCellDelegate -
// Called by PodcastCell to identify podcast for index, then pass this to download service method
extension TAWPodcastsViewController: PodcastCellDelegate {
    func didTapDeleteButton(_ podcast: Podcast) {
        // Do nothing
    }
    
    func downloadTapped(_ podcast: Podcast) {
        self.downloadService.startDownload(podcast)
    }
    
    func downloadFailed(_ podcast: Podcast) {
        self.showAlertWithTitle("Download Failed", message: "There was an error trying to download \(podcast.title). Try again later")
    }
    
    func podcastInvalidated() {
        self.forceRefresh()
    }
}
