//
//  TAWSettingsViewController.swift
//  TAW
//
//  Created by Andrew Steven on 18/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import RealmSwift
import MessageUI
import SkeletonView

let TAWUserHasLoggedOutNotification = "TAWUserHasLoggedOutNotification"
let TAWDownloadRemovedNotification = "TAWDownloadRemovedNotification"

class TAWSettingsViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -
    
    @IBOutlet weak var tableView: UITableView!
    @IBOutlet weak var tableViewHeight: NSLayoutConstraint!
    @IBOutlet weak var versionNumberLabel: UILabel!
    @IBOutlet weak var offlineLabel: UILabel!

    // MARK: - Variables -
    
    let notificationCenter = NotificationCenter.default
    var notificationSettings: [[String: String]] = []
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        self.tableView.tableFooterView = UIView()
        self.versionNumberLabel.text = Utilities().currentAppVersion()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        self.animations = self.isMovingToParent ?
            [.slide(.left, .slightly), .fadeIn] : [.slide(.right, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        self.tableView.isSkeletonable = true
        self.getPreferences()
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    @IBAction func reviewOurApp() {
        PBAppStoreReviewer.reviewApp()
    }
    
    @IBAction func manageSubs(_ sender: Any) {
        if let url = URL(string: "itms-apps://apps.apple.com/account/subscriptions") {
            if UIApplication.shared.canOpenURL(url) {
                UIApplication.shared.open(url, options: [:])
            }
        }
    }
    
    
    @IBAction func contactUs() {
        // help@theanfieldwrap.com
        if MFMailComposeViewController.canSendMail() {
            let mail = MFMailComposeViewController()
            mail.mailComposeDelegate = self
            mail.setToRecipients(["help@theanfieldwrap.com"])
            mail.setSubject("The Anfield Wrap - App Enquiry")
            mail.setMessageBody("", isHTML: false)
            self.present(mail, animated: true, completion: nil)
        }
        else {
            // Can't send an email
            let alert = UIAlertController(title: "Error",
                                          message: "You are not signed in to any email accounts on this device",
                                          preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "Ok", style: .default, handler: nil))
            self.navigationController?.present(alert, animated: true, completion: nil)
        }
    }
    
    @IBAction func termsAndConditions() {
        
        guard let url = URL(string: TAW.Urls.termsOfService) else {
            return //be safe
        }
        
        UIApplication.shared.open(url, options: [:], completionHandler: nil)
    }
    
    @IBAction func privacyPolicy() {
        
        guard let url = URL(string: TAW.Urls.privacyPolicy) else {
            return //be safe
        }

        UIApplication.shared.open(url, options: [:], completionHandler: nil)
    }
    
    @IBAction func logout() {
        let alert = TAWAlertView.shared
        alert.delegate = self
        alert.showAlertView(inView: self.navigationController!.view, alertStyle: .logout)
    }
    
    private func logoutUser() {
        let params = [
            "device_token": LocalStorage.shared.oneSignalId ?? ""
        ]
        
        APIClient.logoutUser(parameters: params, completion: { (jsonResponse, _) in
            self.removeUserDetails()
        })
    }
    
    private func removeUserDetails() {
        // Remove current podcast
        if PBAudioPlayer.shared.podcast != nil {
            PBAudioPlayer.shared.removePodcast()
        }
        
        // Post notification to other views
        self.notifyOnLogout()

        // Set the previousUserId before removing the current User
        LocalStorage.shared.previousUserId = LocalStorage.shared.user?.userId
        
        // Remove the user details
        LocalStorage.shared.user = nil
        LocalStorage.shared.loggedIn = false
        
        self.showLogin()
    }
    
    private func showLogin() {
        self.performSegue(withIdentifier: "unwindToLogin", sender: nil)
    }
    
    // MARK: - API functions -
    
    private func getPreferences() {
        
        if Connectivity.isConnectedToInternet {

            self.tableView.isHidden = false
            self.offlineLabel.isHidden = true
            
            self.view.showAnimatedGradientSkeleton()
            self.tableView.showAnimatedGradientSkeleton()
            
            // Reset data
            self.notificationSettings = []
            
            APIClient.getPreferences { (jsonResponse, _) in
                if jsonResponse["success"] as? Bool == true {
                    
                    if let preferences = jsonResponse["preferences"] as? [[String: String]] {
                        
                        for pref in preferences {
                            self.notificationSettings.append(pref)
                        }
                        
                        DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.7, execute: {
                            
                            self.tableViewHeight.constant = CGFloat(65 * self.notificationSettings.count)
                            self.view.layoutIfNeeded()
                            self.tableView.reloadData()
                            self.view.hideSkeleton()
                            self.tableView.hideSkeleton()
                        })
                    }
                } else {
                    
                    self.view.hideSkeleton()
                    self.tableView.hideSkeleton()
                    
                    let message = jsonResponse["message"] as? String ?? ""
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    }
                    else {
                        self.showAlertForAPIError(jsonResponse["message"] as? String ?? "")
                    }
                }
            }
        } else {
            self.tableView.isHidden = true
            self.offlineLabel.isHidden = false
        }
    }
    
    // MARK: - Convenience
    
    func notifyOnLogout() {
        self.notificationCenter.post(name: Notification.Name(rawValue: TAWUserHasLoggedOutNotification),
                                     object: self)
    }
}

// MARK: - MFMailComposeViewControllerDelegate -

extension TAWSettingsViewController: MFMailComposeViewControllerDelegate {
    
    func mailComposeController(_ controller: MFMailComposeViewController,
                               didFinishWith result: MFMailComposeResult,
                               error: Error?) {
        
        switch result.rawValue {
        case MFMailComposeResult.cancelled.rawValue:
            print("Cancelled")
        case MFMailComposeResult.saved.rawValue:
            print("Saved")
        case MFMailComposeResult.sent.rawValue:
            print("Sent")
        case MFMailComposeResult.failed.rawValue:
            print("Error: \(String(describing: error?.localizedDescription))")
        default:
            break
        }
        controller.dismiss(animated: true, completion: nil)
    }
}

// MARK: - String extension -

extension String {
    func toBool() -> Bool {
        switch self {
        case "True", "true", "yes", "1":
            return true
        case "False", "false", "no", "0", nil:
            return false
        default:
            return false
        }
    }
}

// MARK: - UITableViewDelegate -

extension TAWSettingsViewController: UITableViewDelegate {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return self.notificationSettings.count
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell: TAWSettingsSwitchTableViewCell =
            tableView.dequeueReusableCell(withIdentifier: "settingsSwitchCell",
                                          for: indexPath) as! TAWSettingsSwitchTableViewCell
        if indexPath.row < self.notificationSettings.count {
            let notificationSetting = self.notificationSettings[indexPath.row]
            cell.setNotificationSetting(notificationSetting)
        }
        
        return cell
    }
    
    func tableView(_ tableView: UITableView, estimatedHeightForRowAt indexPath: IndexPath) -> CGFloat {
        return 65
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return 65
    }
}

// MARK: - SkeletonTableViewDataSource -

extension TAWSettingsViewController: SkeletonTableViewDataSource {
    
    func numSections(in collectionSkeletonView: UITableView) -> Int {
        return 1
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return 4
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView,
                                cellIdentifierForRowAt indexPath: IndexPath) -> ReusableCellIdentifier {
        return "settingsSwitchCell"
    }
}

// MARK: - TAWAlertViewDelegate -

extension TAWSettingsViewController: TAWAlertViewDelegate {
    func acceptedNotifications() {
        // Do nothing
    }

    func unlinkAccount() {
        // Do nothing
    }

    func accountCreated() {
        // Do nothing
    }

    func didTapLogout() {
        self.logoutUser()
    }


}
