//
//  TAWProfileViewController.swift
//  TAW
//
//  Created by Andrew Steven on 17/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import UserNotifications
import Spruce
import OneSignal

let UserSubscriptionLevelHasChanged = "UserSubscriptionLevelHasChanged"
let CheckFreeSubscription = "CheckFreeSubscription"

class TAWProfileViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -

    @IBOutlet weak var nameLabel: UILabel!
    @IBOutlet weak var subscriptionLabel: UILabel!
    @IBOutlet weak var completeAccountButton: PBButton!
    @IBOutlet weak var tableView: UITableView!
    @IBOutlet weak var headerView: PBView!
    @IBOutlet weak var headerViewHeight: NSLayoutConstraint!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var tableviewPaddingBottom: NSLayoutConstraint!
    @IBOutlet weak var tableViewHeight: NSLayoutConstraint!
    @IBOutlet weak var scrollView: UIScrollView!
    @IBOutlet weak var collapsedView: UIView!
    @IBOutlet weak var contentView: UIView!
    @IBOutlet weak var notificationsBannerHeight: NSLayoutConstraint!
    
    // MARK: - Variables -
    
    var subscribed = false
    var options: [String] = ["SETTINGS"]
    
    var maxHeaderHeight: CGFloat = 280
    var minHeaderHeight: CGFloat = 70
    
    // MARK: - Life cycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        headerView.shadowColor = UIColor.black.withAlphaComponent(0.3)
        self.completeAccountButton.titleLabel?.adjustsFontSizeToFitWidth = true
        self.completeAccountButton.titleLabel?.minimumScaleFactor = 0.9
        self.completeAccountButton.titleLabel?.numberOfLines = 0
    }
    
    override func viewWillAppear(_ animated: Bool) {
        
        super.viewWillAppear(animated)
        self.removeDetails()
        self.displayProfile()
        self.checkSubscription()
        self.checkNotificationsEnabled()
    }
    
    private func removeDetails() {
        self.options = ["SETTINGS"]
        self.tableView.reloadData()
        self.nameLabel.text = ""
        self.subscriptionLabel.text = ""
        self.completeAccountButton.isHidden = true
    }
    
    private func checkNotificationsEnabled() {
        
        let current = UNUserNotificationCenter.current()
        
        current.getNotificationSettings(completionHandler: { (settings) in
            if settings.authorizationStatus == .notDetermined {
                // Notification permission has not been asked yet, go for it!
            }
            else if settings.authorizationStatus == .denied {
                // Notification permission was previously denied, go to settings & privacy to re-enable
                DispatchQueue.main.async {
                    self.notificationsBannerHeight.constant = 78
                    self.view.layoutIfNeeded()
                }
                
            }
            else if settings.authorizationStatus == .authorized {
                // Notification permission was already granted
                DispatchQueue.main.async {
                    self.notificationsBannerHeight.constant = 0
                    self.view.layoutIfNeeded()
                }
            }
        })
    }
    
    private func checkSubscription() {
        
        if Connectivity.isConnectedToInternet {
            self.subscriptionLabel.text = "CHECKING SUBSCRIPTION"
            UIView.animate(withDuration: 0.7, delay: 0, options: [.repeat, .autoreverse], animations: {
                self.subscriptionLabel.alpha = 0.2
            }) { (_) in
                self.subscriptionLabel.alpha = 1.0
            }
            
            APIClient.checkSubscription { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    if let userJson = jsonResponse["user"] as? [String: Any] {
                        if let user = User(JSON: userJson) {
                            let needsUpdate = user.subscriptionLevel != LocalStorage.shared.user?.subscriptionLevel
                            
                            user.authToken = LocalStorage.shared.user?.authToken
                            LocalStorage.shared.user = user
                            
                            if needsUpdate {
                                self.displayProfile()
                                NotificationCenter.default.post(name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: self)
                            } else {
                                self.subscriptionLabel.layer.removeAllAnimations()
                                self.subscriptionLabel.text = user.subscriptionLevel.text
                            }
                        }
                    }
                } else {
                    let user = LocalStorage.shared.user
                    let needsUpdate = user?.subscriptionLevel != SubscriptionType.none
                    user?.subscriptions = []
                    LocalStorage.shared.user = user

                    if needsUpdate {
                        self.displayProfile()
                        NotificationCenter.default.post(name: NSNotification.Name(rawValue: UserSubscriptionLevelHasChanged), object: self)
                    }

                    self.subscriptionLabel.layer.removeAllAnimations()
                    self.subscriptionLabel.text = SubscriptionType.none.text
                    
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    }
                }
            }
        }
    }
    
    private func displayProfile() {
        
        let user = LocalStorage.shared.user
        nameLabel.text = user?.fullName
        subscriptionLabel.text = user?.subscriptionLevel.text
        self.subscriptionLabel.layer.removeAllAnimations()
        
        if let guest = user?.guest {
            
            if guest == true {
                
                self.options = [
                    "SETTINGS"
                ]
                
                completeAccountButton.isHidden = false
                maxHeaderHeight = 280
                headerViewHeight.constant = maxHeaderHeight
            }
            else {
                self.options = [
                    "ACCOUNT DETAILS",
                    "DOWNLOADS",
                    "SETTINGS",
                    "WALLET",
                    "MANAGE DEVICES"
                ]
                
                completeAccountButton.isHidden = true
                maxHeaderHeight = 210
                headerViewHeight.constant = maxHeaderHeight
            }
        }
                
        self.minHeaderHeight = self.collapsedView.frame.origin.y + 70
        let tableHeight: CGFloat = CGFloat(self.options.count * 65)
        self.tableviewPaddingBottom.constant = UIScreen.main.bounds.height - tableHeight
        self.tableViewHeight.constant = tableHeight
        
        self.view.layoutIfNeeded()
        self.contentView.layoutIfNeeded()
        self.tableView.reloadData()
    }
    
    // MARK: - Button actions -
    
    @IBAction func openAppSettings() {
        // Show the prompt for push notifications at this point
        OneSignal.promptForPushNotifications(userResponse: { accepted in
            print("User accepted notifications: \(accepted)")
            
            if accepted {
                self.collapseNotificationsBanner()
                
                var oneSignalId = LocalStorage.shared.oneSignalId
                if oneSignalId == nil {
                    if let existingId = OneSignal.getPermissionSubscriptionState().subscriptionStatus.userId {
                        LocalStorage.shared.oneSignalId = existingId
                        oneSignalId = existingId
                    } else {
                        oneSignalId = ""
                    }
                }
                
                if oneSignalId!.count > 0 {
                    self.associateOneSignalWithDevice(oneSignalId!)
                }
            }
            else {
                if let url = URL(string:UIApplication.openSettingsURLString) {
                    if UIApplication.shared.canOpenURL(url) {
                        self.collapseNotificationsBanner()
                        UIApplication.shared.open(url, options: [:], completionHandler: nil)
                    }
                }
            }
        })
    }
    
//    private func showHelp() {
//        guard let url = URL(string: TAW.Urls.help) else {
//            return //be safe
//        }
//
//        UIApplication.shared.open(url, options: [:], completionHandler: nil)
//    }
    
    private func associateOneSignalWithDevice(_ oneSignalId: String) {
        
        let params = [
            "device_token": oneSignalId
        ]
        
        APIClient.associateDeviceToken(parameters: params) { (jsonResponse, error) in
            print("Associate Device Token Response: \(jsonResponse["message"] as! String)")
        }
    }
    
    @IBAction func collapseNotificationsBanner() {
        
        self.notificationsBannerHeight.constant = 0
        UIView.animate(withDuration: 0.3) {
            self.view.layoutIfNeeded()
        }
    }
    
    @IBAction func unwindToProfile(segue:UIStoryboardSegue) {
        if let tabBar = self.parent?.parent as? TAWTabBarController {
            tabBar.showLogin()
        }
    }
    
    @IBAction func completeSignUp() {
        self.performSegue(withIdentifier: "showCompleteAccount", sender: nil)
    }

    private func displayWallet() {
        if Connectivity.isConnectedToInternet {
            self.performSegue(withIdentifier: "showPurchases", sender: nil)
        } else {
            let alert = UIAlertController(title: "Error", message: "Wallet is disabled whilst offline.", preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "Ok", style: .cancel, handler: nil))
            self.navigationController?.present(alert, animated: true, completion: nil)
        }
    }

    private func displayDevices() {
        if Connectivity.isConnectedToInternet {
            self.performSegue(withIdentifier: "showMyDevices", sender: nil)
        } else {
            let alert = UIAlertController(title: "Error", message: "Device management is disabled whilst offline.", preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "Ok", style: .cancel, handler: nil))
            self.navigationController?.present(alert, animated: true, completion: nil)
        }
    }
    
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        
        if segue.identifier == "showCompleteAccount" {
            if let nav = segue.destination as? TAWNavigationController {
                if let vc = nav.viewControllers.first as? TAWSignUpViewController {
                    vc.convertingGuest = true
                }
            }
        }
    }
}

// MARK: - UITableViewDelegate & UITableViewDataSource -

extension TAWProfileViewController: UITableViewDelegate, UITableViewDataSource {
    
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return options.count
    }
    
    func tableView(_ tableView: UITableView,
                   willDisplay cell: UITableViewCell,
                   forRowAt indexPath: IndexPath) {
        
        cell.spruce.prepare(with: self.tableViewAnimations)
        var animation = SpringAnimation(duration: 0.7)
        animation.animationOptions = [UIView.AnimationOptions.allowUserInteraction]
        
        cell.spruce.animate(self.tableViewAnimations,
                            animationType: animation,
                            sortFunction: self.tableViewSortFunction!)
        
        cell.layoutSubviews()
    }
    
    func tableView(_ tableView: UITableView,
                   cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell = tableView.dequeueReusableCell(withIdentifier: "profileCell",
                                                 for: indexPath) as! TAWProfileTableViewCell
        cell.titleLabel.text = options[indexPath.row]
        return cell
    }
    
    func tableView(_ tableView: UITableView,
                   estimatedHeightForRowAt indexPath: IndexPath) -> CGFloat {
        return 65
    }
    
    func tableView(_ tableView: UITableView,
                   heightForRowAt indexPath: IndexPath) -> CGFloat {
        return 65
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        switch options[indexPath.row] {
        case "ACCOUNT DETAILS":
            self.performSegue(withIdentifier: "showAccountDetails", sender: nil)
        case "DOWNLOADS":
            self.performSegue(withIdentifier: "showDownloads", sender: nil)
        case "SETTINGS":
            self.performSegue(withIdentifier: "showSettings", sender: nil)
        case "WALLET":
            self.displayWallet()
        case "MANAGE DEVICES":
            self.displayDevices()
        default:
            return
        }
    }
}

extension TAWProfileViewController: UIScrollViewDelegate {
    
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let offset = scrollView.contentOffset.y + scrollView.contentInset.top
        if offset > 0 && offset < minHeaderHeight {
            
            if offset <= 10 {
                let alpha: CGFloat = 1 - (offset / 10)
                for subview in headerView.subviews {
                    if subview.tag != 999 {
                        subview.alpha = alpha
                    }
                    titleLabel.alpha = 0
                }
            } else {
                let alpha: CGFloat = 1 - (offset / minHeaderHeight)
                for subview in headerView.subviews {
                    if subview.tag != 999 {
                        subview.alpha = 0
                    }
                    titleLabel.alpha = 1 - alpha
                }
            }
            
            let percentage: CGFloat = 1 - (offset/minHeaderHeight)
            headerViewHeight.constant = ((maxHeaderHeight - minHeaderHeight) * percentage) + minHeaderHeight
        } else if offset <= 0 {
            headerViewHeight.constant = maxHeaderHeight
            
            for subview in headerView.subviews {
                if subview.tag != 999 {
                    subview.alpha = 1
                }
                titleLabel.alpha = 0
            }
        } else if offset >= maxHeaderHeight {
            headerViewHeight.constant = minHeaderHeight
            
            for subview in headerView.subviews {
                if subview.tag != 999 {
                    subview.alpha = 0
                }
                titleLabel.alpha = 1
            }
        }
    }
}
