//
//  TAWPurchasesViewController.swift
//  TAW
//
//  Created by Andrew Steven on 18/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Spruce
import SkeletonView

class TAWPurchasesViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -
    
    @IBOutlet weak var headerView: UIView!
    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var balanceLabel: UILabel!
    @IBOutlet weak var bigTokenImageView: UIImageView!
    @IBOutlet weak var bigBalanceLabel: UILabel!
    @IBOutlet weak var bigWalletLabel: UILabel!
    @IBOutlet weak var addCreditButton: UIButton!
    @IBOutlet weak var tableView: UITableView!
    @IBOutlet weak var smallHeaderView: UIView!
    @IBOutlet weak var headerViewHeight: NSLayoutConstraint!
    @IBOutlet weak var noPurchasesLabel: UILabel!
    
    // MARK: - Variables -
    
    var purchases: [Transaction] = []
    var notificationCenter = NotificationCenter.default
    
    // MARK: - Life cycle -

    override func viewDidLoad() {
        super.viewDidLoad()
        
        self.tableView.tableFooterView = UIView()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        self.animations = [.slide(.left, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        self.configureNotifications()
        self.tableView.isSkeletonable = true
        self.getWallet()
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        self.notificationCenter.removeObserver(self)
    }
    
    //MARK: - Notifications
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self, selector: #selector(getWallet), name: NSNotification.Name(rawValue: PBIAPClientPurchasedTokens), object: nil)
    }

    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    // MARK: - API functions -
    
    @objc private func getWallet() {

        if Connectivity.isConnectedToInternet {
            self.noPurchasesLabel.isHidden = true
            self.tableView.showAnimatedGradientSkeleton()
            self.purchases = []

            APIClient.getWallet { (jsonResponse, error) in
                if jsonResponse["success"] as? Bool == true {
                    if let wallet = jsonResponse["wallet"] as? [String: Any] {

                        if let transactionsData = wallet["transactions"] as? [[String: Any]] {
                            for transactionData in transactionsData {
                                if let transaction = Transaction(JSON: transactionData) {
                                    self.purchases.append(transaction)
                                }
                            }
                        }

                        self.purchases = self.purchases.sorted(by: { $0.entryDate > $1.entryDate })

                        DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.7, execute: {
                            self.balanceLabel.text = "\(wallet["balance"] as! Int) Tokens"
                            self.bigBalanceLabel.text = "\(wallet["balance"] as! Int) Tokens"
                            self.tableView.reloadData()
                            self.tableView.hideSkeleton()
                            self.noPurchasesLabel.isHidden = self.purchases.count > 0
                        })
                    } else {
                        print("Error retrieving wallet")
                    }
                }
                else {
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    } else {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    }
                }
            }
        } else {
            self.balanceLabel.text = "Balance unavailable"
            self.bigBalanceLabel.text = "Balance unavailable"
            self.noPurchasesLabel.isHidden = true
            self.purchases = []
            self.tableView.reloadData()
        }
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    @IBAction func addCredit() {
        
        PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!, withTitle: "Making purchase")
        
        let params: [String: Any] = [
            "value": 10,
            "title": "IAP 10 Tokens"
        ]
        
        APIClient.creditAccount(parameters: params) { (jsonResponse, error) in
            if jsonResponse["success"] as? Bool == true {
                if let wallet = jsonResponse["wallet"] as? [String: Any] {
                    
                    if let transactionsData = wallet["transactions"] as? [[String: String]] {
                        self.purchases = []
                        for transactionData in transactionsData {
                            if let transaction = Transaction(JSON: transactionData) {
                                self.purchases.append(transaction)
                            }
                        }
                    }
                    
                    self.purchases = self.purchases.sorted(by: { $0.entryDate > $1.entryDate })
                    
                    PBLoadingOverlay.shared.hideOverlay {
                        self.balanceLabel.text = "\(wallet["balance"] as! Int) Tokens"
                        self.bigBalanceLabel.text = "\(wallet["balance"] as! Int) Tokens"
                        self.tableView.reloadData()
                    }
                } else {
                    PBLoadingOverlay.shared.hideOverlay {
                        print("Error retrieving wallet")
                    }
                }
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    } else {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    }
                }
            }
        }
    }
    
    var maxHeaderHeight: CGFloat = 138
    var minHeaderHeight: CGFloat = 70
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let offset = scrollView.contentOffset.y + scrollView.contentInset.top
        
        if offset > 0 && offset < maxHeaderHeight {
            
            if offset <= maxHeaderHeight/4 {
                let alpha: CGFloat = 1 - (offset / (maxHeaderHeight/4))
                self.bigTokenImageView.alpha = alpha
                self.bigWalletLabel.alpha = alpha
                self.bigBalanceLabel.alpha = alpha
                self.smallHeaderView.alpha = 0
            }
            else {
                let alpha: CGFloat = 1 - (offset / maxHeaderHeight)
                self.bigTokenImageView.alpha = 0
                self.bigWalletLabel.alpha = 0
                self.bigBalanceLabel.alpha = 0
                self.smallHeaderView.alpha = 1 - alpha
            }
            
            let percentage: CGFloat = 1 - (offset/maxHeaderHeight)
            headerViewHeight.constant = ((maxHeaderHeight - minHeaderHeight) * percentage) + minHeaderHeight
        }
        else if offset <= 0 {
            headerViewHeight.constant = maxHeaderHeight
            self.bigTokenImageView.alpha = 1
            self.bigWalletLabel.alpha = 1
            self.bigBalanceLabel.alpha = 1
            self.smallHeaderView.alpha = 0
        }
        else if offset >= maxHeaderHeight {
            headerViewHeight.constant = minHeaderHeight
            self.bigTokenImageView.alpha = 0
            self.bigWalletLabel.alpha = 0
            self.bigBalanceLabel.alpha = 0
            self.smallHeaderView.alpha = 1
        }
    }
}

// MARK: - UITableViewDelegate & UITableViewDataSource -

extension TAWPurchasesViewController: UITableViewDelegate {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return purchases.count
    }
    
    func tableView(_ tableView: UITableView, willDisplay cell: UITableViewCell, forRowAt indexPath: IndexPath) {
        
        cell.spruce.prepare(with: self.tableViewAnimations)
        var animation = SpringAnimation(duration: 0.7)
        animation.animationOptions = [UIView.AnimationOptions.allowUserInteraction]
        
        cell.spruce.animate(self.tableViewAnimations, animationType: animation, sortFunction: self.tableViewSortFunction!)
        cell.layoutSubviews()
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell: TAWPurchasesTableViewCell = tableView.dequeueReusableCell(withIdentifier: "purchasesCell", for: indexPath) as! TAWPurchasesTableViewCell
        cell.setTransaction(self.purchases[indexPath.row])
        return cell
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        
        if self.purchases.count <= indexPath.row {
            return
        }
        
        let transaction = self.purchases[indexPath.row]
        if transaction.transactionType == "purchase" {
            if let podcast = transaction.podcast {
                if let tawVC = instantiateVC(identifier: "podcastDetailVC") as? TAWPodcastDetailViewController {
                    tawVC.podcast = podcast
                    self.navigationController?.pushViewController(tawVC, animated: true)
                }
            }
            else if let video = transaction.video {
                if let tawVC = instantiateVC(identifier: "videoDetailVC") as? TAWVideoDetailViewController {
                    tawVC.video = video
                    self.navigationController?.pushViewController(tawVC, animated: true)
                }
            }
        }
    }
    
    func tableView(_ tableView: UITableView, estimatedHeightForRowAt indexPath: IndexPath) -> CGFloat {
        return UITableView.automaticDimension
    }
}

extension TAWPurchasesViewController: SkeletonTableViewDataSource {
    
    func numSections(in collectionSkeletonView: UITableView) -> Int {
        return 1
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return UIDevice.isSmallDevice ? 4 : 6
    }
    
    func collectionSkeletonView(_ skeletonView: UITableView, cellIdentifierForRowAt indexPath: IndexPath) -> ReusableCellIdentifier {
        let identifier = "purchasesCell"
        return identifier
    }
}
