//
//  TAWYoutubeVideoViewController.swift
//  TAW
//
//  Created by Andrew Steven on 04/04/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import UIKit
import YoutubeKit

let TAWStoppedVideoPlayerNotification = "TAWStoppedVideoPlayerNotification"
let TAWStoppedYoutubePlayerNotification = "TAWStoppedYoutubePlayerNotification"

class TAWYoutubeVideoViewController: UIViewController {
    
    // MARK: - Variables -
    
    var tokens: Int = 0
    var video: Video?
    private var videoPlayer: YTSwiftyPlayer!
    var notificationCenter = NotificationCenter.default

    // MARK: - Lifecycle -
    
    override func viewDidLoad() {
        super.viewDidLoad()
        self.configureNotifications()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        DispatchQueue.main.async {
            self.setupVideoPlayer()
        }
    }
    
    private func setupVideoPlayer() {
        videoPlayer = YTSwiftyPlayer(frame: CGRect(x: 0,
                                                   y: 0,
                                                   width: self.view.frame.size.width,
                                                   height: self.view.frame.size.height),
                                     playerVars: [.videoID(self.video!.postId),
                                                  .showFullScreenButton(false),
                                                  .showControls(VideoControlAppearance.hidden),
                                                  .showRelatedVideo(false),
                                                  .playsInline(false),
                                                  .showInfo(false),
                                                  .showModestbranding(true)])
        
        videoPlayer.backgroundColor = UIColor(red: 27/255, green: 27/255, blue: 27/255, alpha: 1.0)
        videoPlayer.autoplay = true
        videoPlayer.delegate = self
        videoPlayer.isHidden = true
        videoPlayer.alpha = 1
        videoPlayer.clipsToBounds = true
        
        self.view.addSubview(videoPlayer)
        videoPlayer.loadPlayer()
    }
    
    private func dismiss() {
        self.dismiss(animated: true, completion: nil)
    }
    
    // MARK: - Notifications -
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self,
                                            selector: #selector(recordAnalyticForStoppedVideo(notification:)),
                                            name: NSNotification.Name(rawValue: TAWStoppedVideoPlayerNotification),
                                            object: nil)
        
        self.notificationCenter.addObserver(self,
                                            selector: #selector(dismissedYoutubeVideo),
                                            name: NSNotification.Name(rawValue: TAWStoppedYoutubePlayerNotification),
                                            object: nil)
    }
    
    deinit {
        self.notificationCenter.removeObserver(self)
    }
    
    @objc public func recordAnalyticForStoppedVideo(notification: NSNotification) {
        if let elapsedTime: Double = notification.userInfo?["elapsedTime"] as? Double {
            APIClient.recordAnalytic(.stoppedVideo,
                                     variable: self.video!.postId,
                                     secondaryVariable: "\(elapsedTime)")
        }
    }
    
    @objc public func dismissedYoutubeVideo() {
        APIClient.recordAnalytic(.stoppedVideo,
                                 variable: self.video!.postId,
                                 secondaryVariable: "\(self.videoPlayer.currentTime)")
        
        self.videoPlayer.alpha = 0
        self.videoPlayer.isHidden = true
        DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.3) {
            self.dismiss(animated: true, completion: nil)
        }
    }
}

// MARK: - YTSwiftyPlayerDelegate -

extension TAWYoutubeVideoViewController: YTSwiftyPlayerDelegate {
    
    func player(_ player: YTSwiftyPlayer,
                didChangeState state: YTSwiftyPlayerState) {
        switch state {
            
        case .ended:
            self.dismissedYoutubeVideo()
            
//        case .paused, .ended:
//
//            APIClient.recordAnalytic(.stoppedVideo,
//                                     variable: self.video!.postId,
//                                     secondaryVariable: "\(self.videoPlayer.currentTime)")
//
//            self.videoPlayer.alpha = 0
//            self.videoPlayer.isHidden = true
//            DispatchQueue.main.asyncAfter(deadline: DispatchTime.now() + 0.3) {
//                self.dismiss(animated: true, completion: nil)
//            }
        default:
            self.videoPlayer.isHidden = false
        }
    }
    
    func player(_ player: YTSwiftyPlayer, didChangeQuality quality: YTSwiftyVideoQuality) {
//        print("didChangeQuality")
    }
    
    func player(_ player: YTSwiftyPlayer, didReceiveError error: YTSwiftyPlayerError) {
//        print("didReceiveError")
    }
    
    func player(_ player: YTSwiftyPlayer, didUpdateCurrentTime currentTime: Double) {
//        print("didUpdateCurrentTime")
    }
    
    func player(_ player: YTSwiftyPlayer, didChangePlaybackRate playbackRate: Double) {
//        print("didChangePlaybackRate")
    }
    
    func playerReady(_ player: YTSwiftyPlayer) {
//        print("playerReady")
    }
    
    func apiDidChange(_ player: YTSwiftyPlayer) {
//        print("apiDidChange")
    }
    
    func youtubeIframeAPIReady(_ player: YTSwiftyPlayer) {
//        print("youtubeIframeAPIReady")
    }
    
    func youtubeIframeAPIFailedToLoad(_ player: YTSwiftyPlayer) {
//        print("youtubeIframeAPIFailedToLoad")
    }
}
