//
//  Podcast.swift
//  TAW
//
//  Created by Andrew Steven on 18/01/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import Foundation
import ObjectMapper
import RealmSwift

enum PodcastTier: String {
    case free = "0"
    case skeleton = "1"
    case premium = "2"
}

class Podcast: Object, Mappable {
    
    @objc dynamic var purchased: Bool = false
    @objc dynamic var postId: String = ""
    @objc dynamic var title: String = ""
    @objc dynamic var content: String = ""
    @objc dynamic var createdDate: String = ""
    @objc dynamic var premium: Bool = false
    @objc dynamic var url: String = ""
    @objc dynamic var thumbnailImage: String = ""
    @objc dynamic var mediumImage: String = ""
    @objc dynamic var largeImage: String = ""
    @objc dynamic var podcastLength: String = ""
    @objc dynamic var cost: Int = 0
    
    @objc dynamic var subLevel: String = "0"

    @objc dynamic var imageData: Data? = nil
    
    override static func primaryKey() -> String? {
        return "postId"
    }
    
    required convenience init?(map: Map) {
        self.init()
    }
    
    func mapping(map: Map) {
        purchased <- map["purchased"]
        postId <- map["post_id"]
        title <- map["title"]
        content <- map["content"]
        createdDate <- map["created_date"]
        premium <- map["premium"]
        url <- map["podcast_full_url"]
        thumbnailImage <- map["thumbnail_image"]
        mediumImage <- map["medium_image"]
        largeImage <- map["large_image"]
        if thumbnailImage.count == 0 {
            thumbnailImage = largeImage
        }
        if mediumImage.count == 0 {
            mediumImage = largeImage
        }
        podcastLength <- map["podcast_length"]
        cost <- map["cost"]
        
        subLevel <- map["sub_level"]
    }
    
    var formattedCost: String {
        if cost > 1 {
            return "\(self.cost) Tokens"
        }
        return "\(self.cost) Token"
    }
    
    var formattedDate: String {
        let dateFormatter = DateFormatter()
        dateFormatter.dateFormat = "yyyy-MM-dd HH:mm:ss"
        dateFormatter.locale = Locale(identifier: NSLocale.current.identifier)
        dateFormatter.timeZone = TimeZone(abbreviation: "UTC")
        let date = dateFormatter.date(from: self.createdDate)
        
        dateFormatter.locale = Locale(identifier: NSLocale.current.identifier)
        dateFormatter.timeZone = TimeZone.current
        dateFormatter.dateFormat = "dd MMM yyyy"
        return dateFormatter.string(from: date!)
    }
    
    var duration: String {
        return "\(self.podcastLength) Mins"
    }
    
    var safeURL: URL {
        let trimmed = self.url.trimmingCharacters(in: .whitespacesAndNewlines)
        return URL(string: trimmed)!
    }
    
    func getSaveFileUrl() -> URL {
        let documentsURL = FileManager.default.urls(for: .documentDirectory, in: .userDomainMask)[0]
        let fileURL = documentsURL.appendingPathComponent((self.safeURL.lastPathComponent))
        return fileURL
    }
    
    var isDownloaded: Bool {
        let fileUrl: URL = self.getSaveFileUrl()
        if FileManager.default.fileExists(atPath: fileUrl.path) {
            return true
        }
        return false
    }
    
    var tier: PodcastTier {
        return PodcastTier(rawValue: self.subLevel) ?? PodcastTier.premium
    }
    
    var isLocked: Bool {
        
        if self.purchased {
            return false
        }
        
        switch tier {
        case .free:
            if self.premium {
                // Checks whether user has subscription level 1, 8, or 9
                if let user = LocalStorage.shared.user {
                    if user.isSubscribedToAudio {
                        return false
                    }
                    return true
                }
                return true
            } else {
                return false
            }
            
        case .skeleton:
            // Checks whether user has subscription level 1, 8, or 9
            if let user = LocalStorage.shared.user {
                if user.isSubscribedToAudio {
                    return false
                }
                return true
            }
            return true

        case .premium:
            // Checks whether user has subscription level 8 or 9
            if let user = LocalStorage.shared.user {
                if user.subscriptionLevel == .premium || user.subscriptionLevel == .audio {
                    return false
                }
                return true
            }
            return true
        }
    }
    
    func premiumImageForCell() -> UIImage? {
        
        switch tier {
        case .free:
            if self.premium {
                if let user = LocalStorage.shared.user {
                    if user.isSubscribedToAudio {
                        return nil
                    }
                    return UIImage(named: "premium_icon")
                }
            }
            return nil
            
        case .skeleton:
            if let user = LocalStorage.shared.user {
                if user.isSubscribedToAudio {
                    return nil
                } else if self.purchased {
                    return UIImage(named: "purchased_icon")
                } else {
                    return UIImage(named: "premium_icon")
                }
            }
            return UIImage(named: "premium_icon")
            
        case .premium:
            if let user = LocalStorage.shared.user {
                if user.subscriptionLevel == .premium || user.subscriptionLevel == .audio {
                    return nil
                } else if self.purchased {
                    return UIImage(named: "purchased_icon")
                } else {
                    return UIImage(named: "premium_icon")
                }
            }
            return UIImage(named: "premium_icon")
        }
    }
    
    var premiumImage: UIImage? {
        
        switch tier {
        case .free:
            if self.premium {
                if let user = LocalStorage.shared.user {
                    if user.isSubscribedToAudio {
                        return UIImage(named: "subscribed_icon")
                    }
                    return UIImage(named: "premium_icon")
                }
            }
            return nil
            
        case .skeleton:
            if let user = LocalStorage.shared.user {
                if user.isSubscribedToAudio {
                    return UIImage(named: "subscribed_icon")
                } else if self.purchased {
                    return UIImage(named: "purchased_icon")
                } else {
                    return UIImage(named: "premium_icon")
                }
            }
            return UIImage(named: "premium_icon")
            
        case .premium:
            if let user = LocalStorage.shared.user {
                if user.subscriptionLevel == .premium || user.subscriptionLevel == .audio {
                    return UIImage(named: "subscribed_icon")
                } else if self.purchased {
                    return UIImage(named: "purchased_icon")
                } else {
                    return UIImage(named: "premium_icon")
                }
            }
            return UIImage(named: "premium_icon")
        }
    }
}
