//
//  DownloadService.swift
//  TAW
//
//  Created by Andrew Steven on 20/02/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import Foundation
import UIKit
import RealmSwift
import Crashlytics

let downloadServiceUpdatedNotification = "DownloadServiceUpdatedNotification"
let downloadServiceCompletedNotification = "DownloadServiceCompletedNotification"
let downloadServiceFailedNotification = "DownloadServiceFailedNotification"

// Downloads podcasts & videos, and stores in local file.
// Allows cancel, pause, resume download.
class DownloadService: NSObject {
    
    // MARK: - Properties
    static let shared = DownloadService()
    let notificationCenter = NotificationCenter.default
    let queue: OperationQueue = OperationQueue.main
    
    // ViewControllers create downloadsSession
    lazy var downloadsSession: URLSession = {
        self.queue.qualityOfService = .utility
        let configuration = URLSessionConfiguration.background(withIdentifier: Bundle.main.bundleIdentifier!)
        configuration.httpAdditionalHeaders = ["Accept-Encoding": ""]
        return URLSession(configuration: configuration, delegate: self, delegateQueue: self.queue)
    }()
    
    var activeDownloads: [URL: Download] = [:]
    
    // MARK: - Download methods called by cell delegate methods
    
    func startDownload(_ podcast: Podcast) {

        let download = Download(podcast: podcast)
        download.task = downloadsSession.downloadTask(with: podcast.safeURL)
        download.task!.resume()
        download.isDownloading = true
        activeDownloads[podcast.safeURL] = download
    }
    
    func pauseDownload(_ podcast: Podcast) {
        guard let download = activeDownloads[podcast.safeURL] else { return }
        if download.isDownloading {
            download.task?.cancel(byProducingResumeData: { data in
                download.resumeData = data
            })
            download.isDownloading = false
        }
    }
    
    func cancelDownload(_ podcast: Podcast) {
        if let download = activeDownloads[podcast.safeURL] {
            download.task?.cancel()
            activeDownloads[podcast.safeURL] = nil
        }
    }
    
    func resumeDownload(_ podcast: Podcast) {
        guard let download = activeDownloads[podcast.safeURL] else { return }
        if let resumeData = download.resumeData {
            download.task = downloadsSession.downloadTask(withResumeData: resumeData)
        } else {
            download.task = downloadsSession.downloadTask(with: podcast.safeURL)
        }
        download.task!.resume()
        download.isDownloading = true
    }
    
    // MARK: - Convenience
    
    func notifyOnDownloadProgressUpdated(_ download: Download) {
        let dataDictionary: [String: Download] = ["download": download]
        self.notificationCenter.post(name: Notification.Name(rawValue: downloadServiceUpdatedNotification), object: nil, userInfo: dataDictionary)
    }
    
    func notifyOnDownloadProgressCompleted(_ download: Download) {
        let dataDictionary: [String: Download] = ["download": download]
        self.notificationCenter.post(name: Notification.Name(rawValue: downloadServiceCompletedNotification), object: nil, userInfo: dataDictionary)
    }
    
    func notifyOnDownloadFailed(_ download: Download) {
        let dataDictionary: [String: Download] = ["download": download]
        self.notificationCenter.post(name: Notification.Name(rawValue: downloadServiceFailedNotification), object: nil, userInfo: dataDictionary)
    }
}

extension DownloadService: URLSessionDownloadDelegate {
    
    // Stores downloaded file
    @objc func urlSession(_ session: URLSession, downloadTask: URLSessionDownloadTask, didFinishDownloadingTo location: URL) {
        
        if let sourceURL = downloadTask.originalRequest?.url {
            if let download = self.activeDownloads[sourceURL] {
                self.activeDownloads[sourceURL] = nil
                
                if download.podcast.isInvalidated {
                    return
                }
                
                let fileManager = FileManager.default
                let documentsURL = fileManager.urls(for: .documentDirectory, in: .userDomainMask)[0]
                let destinationURL = documentsURL.appendingPathComponent((sourceURL.lastPathComponent))
                //            print("Podcast destination URL: \(destinationURL)")
                
                try? fileManager.removeItem(at: destinationURL)
                
                do {
                    try fileManager.copyItem(at: location, to: destinationURL)
                    print("Podcast downloaded and stored locally")
                    DispatchQueue.main.async {
                        APIClient.recordAnalytic(.downloadedPodcast, variable: download.podcast.postId, secondaryVariable: "1")
                        self.notifyOnDownloadProgressCompleted(download)
                    }
                } catch let error {
                    print("Could not copy podcast file to disk: \(error.localizedDescription)")
                    self.notifyOnDownloadFailed(download)
                }
            }
        }
    }
    
    // Updates progress info
    @objc func urlSession(_ session: URLSession, downloadTask: URLSessionDownloadTask, didWriteData bytesWritten: Int64, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) {
        
        Crashlytics.sharedInstance().setObjectValue(bytesWritten, forKey: "bytesWritten")
        Crashlytics.sharedInstance().setObjectValue(totalBytesWritten, forKey: "totalBytesWritten")
        Crashlytics.sharedInstance().setObjectValue(totalBytesExpectedToWrite, forKey: "totalBytesExpectedToWrite")
        
        if let url = downloadTask.originalRequest?.url {
            if let download = self.activeDownloads[url] {
                download.progress = (Float(totalBytesWritten) / Float(totalBytesExpectedToWrite)) * 100
                self.notifyOnDownloadProgressUpdated(download)
            }
        }
    }
}

// MARK: - URLSessionDelegate -

extension DownloadService: URLSessionDelegate {
    
    // Standard background session handler
    public func urlSessionDidFinishEvents(forBackgroundURLSession session: URLSession) {
        DispatchQueue.main.async {
            if let appDelegate = UIApplication.shared.delegate as? AppDelegate,
                let completionHandler = appDelegate.backgroundSessionCompletionHandler {
                appDelegate.backgroundSessionCompletionHandler = nil
                completionHandler()
            }
        }
    }
}
