//
//  TAWPodcastTableViewCell.swift
//  TAW
//
//  Created by Andrew Steven on 19/12/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit
import Kingfisher
import Alamofire
import RealmSwift
import UICircularProgressRing
import ESTMusicIndicator
import Crashlytics

enum PodcastImageQuality {
    case thumbnail
    case medium
    case large
}

enum PodcastDisplayMode {
    case feed
    case download
}

protocol PodcastCellDelegate: class {
    func downloadTapped(_ podcast: Podcast)
    func downloadFailed(_ podcast: Podcast)
    func didTapDeleteButton(_ podcast: Podcast)
    func podcastInvalidated()
}

class TAWPodcastTableViewCell: UITableViewCell {

    @IBOutlet weak var podcastImageView: UIImageView!
    @IBOutlet weak var premiumImageView: UIImageView!
    @IBOutlet weak var dateLabel: UILabel!
    @IBOutlet weak var durationLabel: UILabel!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var downloadButton: UIButton!
    @IBOutlet weak var deleteButton: UIButton!
    @IBOutlet weak var downloadProgressRing: UICircularProgressRing!
    @IBOutlet weak var podcastIndicator: ESTMusicIndicatorView!
    @IBOutlet weak var progressContainer: UIView!
    @IBOutlet weak var progressBar: UIView!
    @IBOutlet weak var progressBarWidth: NSLayoutConstraint!
    
    var podcast: Podcast?
    var displayMode: PodcastDisplayMode = .feed
    var podcastPosition: PodcastPosition?
    weak var delegate: PodcastCellDelegate?
    let notificationCenter = NotificationCenter.default
    
    override func awakeFromNib() {
        super.awakeFromNib()

        self.downloadProgressRing.font = TAW.Fonts.dinAlternate10!
        
        self.notificationCenter.addObserver(self, selector: #selector(updatedDownload(notification:)), name: NSNotification.Name(rawValue: downloadServiceUpdatedNotification), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(completedDownload(notification:)), name: NSNotification.Name(rawValue: downloadServiceCompletedNotification), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(failedDownload(notification:)), name: NSNotification.Name(rawValue: downloadServiceFailedNotification), object: nil)
    }

    override func setSelected(_ selected: Bool, animated: Bool) {
        super.setSelected(selected, animated: animated)
    }
    
    override func prepareForReuse() {
        
        self.podcast = nil
        self.podcastPosition?.removeObserver(self, forKeyPath: "position", context: nil)
        self.podcastPosition = nil
        self.delegate = nil
        self.podcastImageView.image = UIImage(named: "placeholder_podcasts")
        self.premiumImageView.image = nil
        self.dateLabel.text = ""
        self.durationLabel.text = ""
        self.titleLabel.text = ""
        self.downloadButton.isHidden = true
        self.downloadProgressRing.isHidden = true
        self.podcastIndicator.isHidden = true
        self.progressContainer.isHidden = true
        self.progressContainer.layoutIfNeeded()
        self.progressBarWidth.constant = 0
        self.progressContainer.layoutIfNeeded()
    }
    
    func set(podcast: Podcast,
             imageQuality: PodcastImageQuality,
             displayMode: PodcastDisplayMode = .feed,
             delegate: PodcastCellDelegate?,
             download: Download? = nil) {

        self.podcast = podcast
        self.displayMode = displayMode
        self.delegate = delegate

        if podcast.isInvalidated {
            self.delegate?.podcastInvalidated()
            return
        }
        
        let realm = try! Realm()
        let objects = realm.objects(PodcastPosition.self).filter("postId == %@", podcast.postId)
        self.podcastPosition = objects.first
        
        var image: String? = nil
        switch imageQuality {
        case .thumbnail:
            image = self.podcast?.thumbnailImage ?? nil
        case .medium:
            image = self.podcast?.mediumImage ?? nil
        case .large:
            image = self.podcast?.largeImage ?? nil
        }

        if let theImage = image {
            self.podcastImageView.kf.setImage(with: URL(string: theImage), placeholder: UIImage(named: "placeholder_podcasts"), options: [.transition(.fade(0.2))])

            if podcast.imageData == nil {
                KingfisherManager.shared.retrieveImage(with: URL(string: theImage)!, options: nil, progressBlock: nil) { (image, error, cache, url) in
                    let realm = try! Realm()
                    try! realm.write {
                        podcast.imageData = image?.pngData()
                    }
                }
            }
        } else {
            self.podcastImageView.image = UIImage(named: "placeholder_podcasts")
        }
        
        self.updateView()
        self.updateProgressBar()

        self.premiumImageView.image = self.podcast?.premiumImageForCell()
        self.premiumImageView.isHidden = self.premiumImageView.image == nil

        self.titleLabel.text = podcast.title
        self.dateLabel.text = podcast.formattedDate
        self.durationLabel.text = podcast.duration
        
        self.downloadProgressRing.font = TAW.Fonts.dinAlternate10!
        self.downloadProgressRing.ringStyle = .ontop
        self.layoutIfNeeded()
        
        if let download = download {
            self.downloadButton.isHidden = true
            let value: CGFloat = CGFloat(download.progress)
            self.downloadProgressRing.startProgress(to: value, duration: 0)
            self.downloadProgressRing.isHidden = false
        }
        
        if let podcastPosition = self.podcastPosition {
            podcastPosition.addObserver(self, forKeyPath: "position", options: .new, context: nil)
        }
    }
    
    override func observeValue(forKeyPath keyPath: String?, of object: Any?, change: [NSKeyValueChangeKey : Any]?, context: UnsafeMutableRawPointer?) {
        if keyPath == "position" {
//            print("Date property has changed, new value is \(change![NSKeyValueChangeKey.newKey]!)")
            self.updateProgressBar()
        }
    }
    
    deinit {
        self.podcastPosition?.removeObserver(self, forKeyPath: "position", context: nil)
        self.notificationCenter.removeObserver(self, name: NSNotification.Name(rawValue: downloadServiceUpdatedNotification), object: nil)
        self.notificationCenter.removeObserver(self, name: NSNotification.Name(rawValue: downloadServiceCompletedNotification), object: nil)
        self.notificationCenter.removeObserver(self, name: NSNotification.Name(rawValue: downloadServiceFailedNotification), object: nil)
    }
    
    func updateView() {
        
        guard let podcast = self.podcast else {
            return
        }
        
        // DO NOT REMOVE THIS - When deleting objects from realm on logout, the podcasts will be invalidated, therefore we need this check to be here
        if podcast.isInvalidated { return }
        if LocalStorage.shared.user == nil { return }

        if self.displayMode == .feed {
            self.updateFeedView()
        } else {
            self.updateDownloadView()
        }
    }

    private func updateFeedView() {
        guard let podcast = self.podcast, let user = LocalStorage.shared.user else {
            return
        }

        self.deleteButton.isHidden = true
        self.downloadButton.isHidden = false

        DispatchQueue.main.async {
            if user.guest {
                self.downloadButton.isHidden = true
            } else if podcast.isLocked {
                self.downloadButton.isHidden = true
            } else if podcast.isDownloaded {
                self.downloadButton.isHidden = true
            }
        }

        if PBAudioPlayer.shared.podcast?.title == podcast.title {
            if PBAudioPlayer.shared.isPlaying {
                self.podcastIndicator.state = .playing
                self.downloadButton.isHidden = true
            } else {
                self.podcastIndicator.state = .paused
                self.downloadButton.isHidden = true
            }
        } else {
            self.podcastIndicator.state = .stopped
        }

        if let download = DownloadService.shared.activeDownloads[podcast.safeURL] {
            self.downloadButton.isHidden = true
            let value: CGFloat = CGFloat(download.progress)
            self.downloadProgressRing.startProgress(to: value, duration: 0)
            self.downloadProgressRing.isHidden = false
        }
    }

    private func updateDownloadView() {
        guard let podcast = self.podcast else {
            return
        }

        self.downloadProgressRing.isHidden = true
        self.downloadButton.isHidden = true

        if PBAudioPlayer.shared.podcast?.title == podcast.title {
            self.deleteButton.isHidden = true
            if PBAudioPlayer.shared.isPlaying {
                self.podcastIndicator.state = .playing
            } else {
                self.podcastIndicator.state = .paused
            }
        } else {
            self.deleteButton.isHidden = false
            self.podcastIndicator.state = .stopped
        }
    }
    
    func updateProgressBar() {
        
        let containerWidth: CGFloat = self.progressContainer.frame.width
        Crashlytics.sharedInstance().setObjectValue(containerWidth, forKey: "containerWidth")
        
        self.progressContainer.layoutIfNeeded()
        
        if let podcast = self.podcast {
            
            var elapsed: Double = self.podcastPosition?.position ?? 0
            if !elapsed.isValid() { elapsed = 0 }
            Crashlytics.sharedInstance().setObjectValue(elapsed, forKey: "elapsed")
            
            var total: Double = 0
            if podcast.title == PBAudioPlayer.shared.podcast?.title {
                total = PBAudioPlayer.shared.totalDuration
            } else if let podcastMinutes = Double(podcast.podcastLength) {
                total = podcastMinutes * 60
            }
            
            if !total.isValid() { total = 0 }
            Crashlytics.sharedInstance().setObjectValue(total, forKey: "total")
            
            if elapsed == 0 {
                self.progressBarWidth.constant = 0
                UIView.animate(withDuration: 0.5) {
                    self.progressContainer.layoutIfNeeded()
                }
            } else if elapsed > total {
                self.progressBarWidth.constant = containerWidth
                UIView.animate(withDuration: 0.5) {
                    self.progressContainer.layoutIfNeeded()
                }
            } else {
                if elapsed.isValid() && total.isValid() {
                    let percentage: CGFloat = CGFloat(elapsed/total) * containerWidth
                    self.progressBarWidth.constant = percentage
                    self.layoutIfNeeded()
                } else {
                    self.progressBarWidth.constant = 0
                    self.layoutIfNeeded()
                }
            }
            
            self.progressContainer.isHidden = self.progressBarWidth.constant == 0
            if podcast.isLocked {
                self.progressContainer.isHidden = true
            }
        }
    }

    @IBAction func download(_ sender: Any) {
        
        self.downloadButton.isHidden = true
        self.downloadProgressRing.startProgress(to: 0, duration: 0)
        self.downloadProgressRing.isHidden = false
        
        if let podcast = self.podcast {
            self.delegate?.downloadTapped(podcast)
        }
    }
    
    @objc public func updatedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            if download.podcast == self.podcast {
                let value: CGFloat = CGFloat(download.progress)
                DispatchQueue.main.async {
                    self.downloadProgressRing.startProgress(to: value, duration: 0)
                }
            }
        }
    }
    
    @objc public func completedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            if download.podcast == self.podcast {
                DispatchQueue.main.async {
                    self.downloadProgressRing.isHidden = true
                }
            }
        }
    }
    
    @objc public func failedDownload(notification: NSNotification) {
        
        if let download = notification.userInfo?["download"] as? Download {
            if download.podcast == self.podcast {
                DispatchQueue.main.async {
                    self.downloadButton.isHidden = false
                    self.downloadProgressRing.isHidden = true
                    self.downloadProgressRing.startProgress(to: 0, duration: 0)
                    self.delegate?.downloadFailed(download.podcast)
                }
            }
        }
    }

    @IBAction func deleteDownload() {
        if let podcast = self.podcast {
            self.delegate?.didTapDeleteButton(podcast)
        }
    }
}
