//
//  PBBannerView.swift
//  TAW
//
//  Created by Andrew Steven on 07/01/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import Foundation
import UIKit

public enum PBBannerViewStyle {
    case success // Green
    case failure // Red
    case warning // Yellow
    case noInternet // Black
    
    var bgColor: UIColor {
        switch self {
        case .success:
            return .green
        case .failure:
            return .red
        case .warning:
            return .yellow
        case .noInternet:
            return .black
        }
    }
    
    // TODO: Create reusable colour schemes for future use
    var txtColor: UIColor {
        return .white
    }
    
    var titleFont: UIFont {
        return UIFont(name: "DINAlternate-Bold", size: 15)!
    }
}

public class PBBannerView {
    
    let padding: CGFloat = 34
    let screenWidth: CGFloat = UIScreen.main.bounds.size.width
    let bannerHeight: CGFloat = 35
    
    public var autoDismisses = true
    private var timer = Timer()
    
    private var bannerView = UIView()
    private var titleLabel = UILabel()
    
    private var isVisible = false
    
    // TODO: Expand the functionality by adding other elements to the PBBannerView
//    var subtitleLabel = UILabel()
//    var leftImageView = UIImageView()
//    var rightImageView = UIImageView()
    
    class var shared: PBBannerView {
        struct Static {
            static let instance: PBBannerView = PBBannerView()
        }
        return Static.instance
    }
    
    public func showBanner(inView view: UIView, withTitle title: String, style: PBBannerViewStyle) {
        
        if !isVisible {
            
            isVisible = true
            
            let window = UIApplication.shared.keyWindow
            let topPadding = window?.safeAreaInsets.top ?? 0
            let actualHeight: CGFloat = topPadding + bannerHeight
            
            bannerView = UIView(frame: CGRect(x: 0, y: -actualHeight, width: screenWidth, height: actualHeight))
            bannerView.backgroundColor = style.bgColor
            
            titleLabel = UILabel(frame: CGRect(x: padding, y: topPadding, width: screenWidth - (2 * padding), height: bannerHeight))
            titleLabel.text = title
            titleLabel.textAlignment = .center
            titleLabel.textColor = style.txtColor
            titleLabel.font = style.titleFont
            titleLabel.isUserInteractionEnabled = false
            bannerView.addSubview(titleLabel)
            
            view.addSubview(bannerView)
            
            // Swipe to dismiss gesture
            let slideUp = UISwipeGestureRecognizer(target: self, action: #selector(dismissBanner))
            slideUp.direction = .up
            bannerView.addGestureRecognizer(slideUp)
            
            let tapGesture = UITapGestureRecognizer(target: self, action: #selector(dismissBanner))
            tapGesture.numberOfTapsRequired = 1
            bannerView.addGestureRecognizer(tapGesture)
            
            DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
                UIView.animate(withDuration: 0.3, animations: {
                    self.bannerView.frame = CGRect(x: 0, y: 0, width: self.screenWidth, height: actualHeight)
                }) { (_) in
                    if self.autoDismisses {
                        self.dismissBanner()
                    }
                }
            }
        }
    }
    
    @objc private func dismissBanner() {
        
        let window = UIApplication.shared.keyWindow
        let topPadding = window?.safeAreaInsets.top ?? 0
        let actualHeight: CGFloat = topPadding + bannerHeight
        
        if autoDismisses {
            DispatchQueue.main.asyncAfter(deadline: .now() + 3) {
                UIView.animate(withDuration: 0.3, animations: {
                    self.bannerView.frame = CGRect(x: 0, y: -actualHeight, width: self.screenWidth, height: actualHeight)
                }) { (_) in
                    self.bannerView.removeFromSuperview()
                    self.isVisible = false
                }
            }
            
        }
        else {
            UIView.animate(withDuration: 0.3, animations: {
                self.bannerView.frame = CGRect(x: 0, y: -actualHeight, width: self.screenWidth, height: actualHeight)
            }) { (_) in
                self.bannerView.removeFromSuperview()
                self.isVisible = false
            }
        }
    }
}
