//
//  PBMiniPlayerView.swift
//  TAW
//
//  Created by Andrew Steven on 14/01/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import UIKit
import RealmSwift
import Crashlytics

protocol PBMiniPlayerViewDelegate: class {
    func openPlayer()
}

class PBMiniPlayerView: UIView {

    @IBOutlet weak var contentView: UIView!
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var playButton: UIButton!
    @IBOutlet weak var closeButton: UIButton!
    @IBOutlet weak var progressView: UIView!
    @IBOutlet weak var progressViewWidth: NSLayoutConstraint!
    
    weak var delegate: PBMiniPlayerViewDelegate?
    var isPlaying: Bool = false
    var timer = Timer()
    var notificationCenter = NotificationCenter.default
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        customInit()
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        customInit()
    }
    
    private func customInit() {
        Bundle.main.loadNibNamed("PBMiniPlayerView", owner: self, options: nil)
        addSubview(contentView)
        contentView.frame = self.bounds
        contentView.autoresizingMask = [.flexibleWidth, .flexibleHeight]
        self.isPlaying = PBAudioPlayer.shared.isPlaying
        let titleText = PBAudioPlayer.shared.podcast?.title
        
        DispatchQueue.main.async {
            self.titleLabel.text = titleText?.htmlDecoded
        }
        
        let image  = self.isPlaying ? UIImage(named: "pause_icon") : UIImage(named: "play_icon")
        playButton.setImage(image, for: .normal)
        
        if self.isPlaying {
            self.startTimer()
        }
        
        self.configureNotifications()
    }
    
    //MARK: - Notifications
    
    func configureNotifications() {
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged), object: nil)
        self.notificationCenter.addObserver(self, selector: #selector(podcastStateChanged), name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged), object: nil)
    }

    deinit {
        self.notificationCenter.removeObserver(self, name: NSNotification.Name(rawValue: PBAudioPlayerOnTrackChanged), object: nil)
        self.notificationCenter.removeObserver(self, name: NSNotification.Name(rawValue: PBAudioPlayerOnPlaybackStateChanged), object: nil)
    }
    
    @objc func podcastStateChanged() {
        self.updatePlayButtonState()
    }
    
    func updatePlayButtonState() {
        self.stopTimer()
        self.isPlaying = PBAudioPlayer.shared.isPlaying
        let image  = self.isPlaying ? UIImage(named: "pause_icon") : UIImage(named: "play_icon")
        self.playButton.setImage(image, for: .normal)
        if self.isPlaying {
            self.startTimer()
        }
    }
    
    func startTimer() {
        self.timer = Timer.scheduledTimer(timeInterval: 1.0, target: self, selector: #selector(updateProgressView), userInfo: nil, repeats: true)
    }
    
    func stopTimer() {
        self.timer.invalidate()
    }
    
    @objc func updateProgressView() {
        
        if PBAudioPlayer.shared.podcast == nil {
            return
        }
        
        let elapsed: Double = PBAudioPlayer.shared.elapsedTime
        let total: Double = PBAudioPlayer.shared.totalDuration
        
        Crashlytics.sharedInstance().setObjectValue(elapsed, forKey: "elapsed")
        Crashlytics.sharedInstance().setObjectValue(total, forKey: "total")
        
        self.layoutIfNeeded()
        
        if elapsed == 0 {
            self.progressViewWidth.constant = 0
            self.layoutIfNeeded()
        } else if elapsed > total {
            self.progressViewWidth.constant = UIScreen.main.bounds.width
            self.layoutIfNeeded()
        } else {
            if elapsed.isValid() && total.isValid() {
                let percentage: CGFloat = CGFloat(elapsed/total) * UIScreen.main.bounds.width
                self.progressViewWidth.constant = percentage
                self.layoutIfNeeded()
            } else {
                self.progressViewWidth.constant = 0
                self.layoutIfNeeded()
            }
        }
        
        if let podcast = PBAudioPlayer.shared.podcast {
            let userId = LocalStorage.shared.user?.userId ?? ""
            do {
                let realm = try Realm()
                try realm.write {
                    realm.add(PodcastPosition(value: ["postId": podcast.postId, "position": elapsed, "userId": userId]), update: .all)
                }
            } catch let error {
                print("Failed to update podcast position: \(error.localizedDescription)")
            }
        }
    }
    
    @IBAction func didTapMiniPlayer() {
        if let postId = PBAudioPlayer.shared.podcast?.postId {
            APIClient.recordAnalytic(.maximisedPlayer, variable: postId, secondaryVariable: "1")
        }
        
        self.delegate?.openPlayer()
    }
    
    @IBAction func play() {
        self.isPlaying = !self.isPlaying
        PBAudioPlayer.shared.togglePlayPause()
        let image  = self.isPlaying ? UIImage(named: "pause_icon") : UIImage(named: "play_icon")
        playButton.setImage(image, for: .normal)
    }
    
    @IBAction func close() {
        self.timer.invalidate()
        PBAudioPlayer.shared.removePodcast()
        self.isPlaying = false
        let image  = self.isPlaying ? UIImage(named: "pause_icon") : UIImage(named: "play_icon")
        playButton.setImage(image, for: .normal)
        
    }
}
