//
//  PBTextField.swift
//
//  Created by Will Roberts on 20/06/2018.
//  Copyright © 2018 PixelBeard. All rights reserved.
//

import UIKit

class PBTextField: UIView, UITextFieldDelegate {

    var theFrameWidth: CGFloat!
    var theFrameHeight: CGFloat!

    var theLeftImageView: UIImageView!
    var theTextFieldLabel: UILabel!
    var theTextFieldErrorLabel: UILabel!
    var theTextField: UITextField!
    var theValidationImageView: UIImageView!
    var theBottomBorderView: UIView!

    var theTopLabelHeight: CGFloat = 15.0
    var theTextFieldHeight: CGFloat = 35.0
    var theBottomBorderHeight: CGFloat = 1.0
    var theTotalHeight: CGFloat = 0.0 // Amalgamation of all of the above float values

    var theLeftImageWidth: CGFloat = 30.0
    var theValidationImageWidth: CGFloat = 30.0

    var theTopLabelColour = TAW.Colors.textFieldTitleColor
    var theHighlightedBorderColour = TAW.Colors.textFieldHighlightedLineColor
    var theBottomBorderColour = TAW.Colors.textFieldLineColor
    var theDisabledIconColour = TAW.Colors.textFieldLineColor
    var theErrorColour = TAW.Colors.textFieldErrorColor

    enum FieldType {
        case standard
        case password
        case email
        case code
    }

    enum ValidationType {
        case standard
        case email
        case password
        case code
    }

    var type: FieldType!
    var validationType: ValidationType!
    var softValidation: Bool = false
    var errorIsInAnimation: Bool = false

    var errorShowInterval: Double = 0.3 // Duration for the error message to appear
    var errorHideInterval: Double = 2.0 // Delay for the error message to disappear
    var minimumCharacterLimit: Int = 2 // Set the minimum character limit for fields

    override init(frame:CGRect) {
        super.init(frame:frame)
    }

    required init?(coder aDecoder: NSCoder) {
        super.init(coder:aDecoder)

        self.backgroundColor = .clear
        self.clipsToBounds = true

        self.theTotalHeight = self.theTopLabelHeight + self.theTextFieldHeight + self.theBottomBorderHeight

        self.theLeftImageView = UIImageView.init()
        self.theLeftImageView.backgroundColor = .clear
        self.theLeftImageView.contentMode = .center
        self.addSubview(self.theLeftImageView)

        self.theTextField = UITextField.init()
        self.theTextField.borderStyle = .none
        self.theTextField.backgroundColor = .clear
        self.theTextField.font = TAW.Fonts.textFieldTextFont
        self.theTextField.textColor = TAW.Colors.textFieldTextColor
        self.theTextField.autocorrectionType = .no
        self.theTextField.autocapitalizationType = .none
        self.theTextField.delegate = self
        self.theTextField.addTarget(self, action: #selector(textFieldDidChange(_:)), for: .editingChanged)
        self.addSubview(self.theTextField)

        self.theTextFieldLabel = UILabel.init()
        self.theTextFieldLabel.backgroundColor = .clear
        self.theTextFieldLabel.textColor = self.theTopLabelColour
        self.theTextFieldLabel.font = TAW.Fonts.textFieldTitleFont
        self.addSubview(self.theTextFieldLabel)

        self.theTextFieldErrorLabel = UILabel.init()
        self.theTextFieldErrorLabel.backgroundColor = .clear
        self.theTextFieldErrorLabel.textColor = self.theErrorColour
        self.theTextFieldErrorLabel.font = TAW.Fonts.textFieldTitleFont
        self.addSubview(self.theTextFieldErrorLabel)

        self.theValidationImageView = UIImageView.init()
        self.theValidationImageView.backgroundColor = .clear
        self.theValidationImageView.contentMode = .center
        self.theValidationImageView.image = nil
        self.addSubview(self.theValidationImageView)

        self.theBottomBorderView = UIView.init()
        self.theBottomBorderView.backgroundColor = self.theBottomBorderColour
        self.addSubview(self.theBottomBorderView)

//        self.setNeedsValidation(validation: false)
        self.hidesLeftImageView(hidden: true)
    }

    func setCharacterLimit(limit: Int) {
        self.minimumCharacterLimit = limit
    }

    func hidesLeftImageView(hidden: Bool) {
        if hidden {
            self.theLeftImageWidth = 0.0
            self.theLeftImageView.isHidden = true
        }
    }

    func animateTheValidationIcon() {
        let animation_speed = 0.05
        UIView.animate(withDuration: animation_speed, animations: {
            self.theTextField.frame.origin.x = self.theTextField.frame.origin.x + 10
            self.theTextFieldLabel.frame.origin.x = self.theTextFieldLabel.frame.origin.x + 10
            self.theBottomBorderView.frame.origin.x = self.theBottomBorderView.frame.origin.x + 10
        }, completion: {
            (value: Bool) in
            UIView.animate(withDuration: animation_speed, animations: {
                self.theTextField.frame.origin.x = self.theTextField.frame.origin.x - 10
                self.theTextFieldLabel.frame.origin.x = self.theTextFieldLabel.frame.origin.x - 10
                self.theBottomBorderView.frame.origin.x = self.theBottomBorderView.frame.origin.x - 10
            }, completion: {
                (value: Bool) in
                UIView.animate(withDuration: animation_speed, animations: {
                    self.theTextField.frame.origin.x = self.theTextField.frame.origin.x + 10
                    self.theTextFieldLabel.frame.origin.x = self.theTextFieldLabel.frame.origin.x + 10
                    self.theBottomBorderView.frame.origin.x = self.theBottomBorderView.frame.origin.x + 10
                }, completion: {
                    (value: Bool) in
                    UIView.animate(withDuration: animation_speed, animations: {
                        self.theTextField.frame.origin.x = self.theTextField.frame.origin.x - 10
                        self.theTextFieldLabel.frame.origin.x = self.theTextFieldLabel.frame.origin.x - 10
                        self.theBottomBorderView.frame.origin.x = self.theBottomBorderView.frame.origin.x - 10
                    }, completion: {
                        (value: Bool) in
                        UIView.animate(withDuration: animation_speed, animations: {
                            self.theTextField.frame.origin.x = self.theTextField.frame.origin.x + 10
                            self.theTextFieldLabel.frame.origin.x = self.theTextFieldLabel.frame.origin.x + 10
                            self.theBottomBorderView.frame.origin.x = self.theBottomBorderView.frame.origin.x + 10
                        }, completion: {
                            (value: Bool) in
                            UIView.animate(withDuration: animation_speed, animations: {
                                self.theTextField.frame.origin.x = self.theTextField.frame.origin.x - 10
                                self.theTextFieldLabel.frame.origin.x = self.theTextFieldLabel.frame.origin.x - 10
                                self.theBottomBorderView.frame.origin.x = self.theBottomBorderView.frame.origin.x - 10
                            }, completion: {
                                (value: Bool) in

                            })
                        })
                    })
                })
            })
        })
    }

    func overrideFailValidation(message: String) {
        self.showFailedValidationIcon()
        self.animateTheValidationIcon()
        self.flashError(message: message)
    }

    func errorMessage() -> String {
        switch self.validationType {
        case .email?:
            return "Please enter a valid email address"
        case .password?:
            return "Please enter a valid password"
        case .standard?:
            return "Please enter \(self.minimumCharacterLimit) characters or more"

        default:
            break
        }
        return "Please check your input and try again"
    }

    func flashError(message: String?) {

        if (self.theTextField.text?.count)! >= self.minimumCharacterLimit {
            self.theTextFieldErrorLabel.text = "\(message ?? self.errorMessage())"
        } else {
            self.theTextFieldErrorLabel.text = "Please enter \(self.minimumCharacterLimit) characters or more"
        }
        let shouldHideTextfield: Bool = (self.theTextField.text?.count)! == 0

        if !self.errorIsInAnimation {
            self.errorIsInAnimation = true
            UIView.animate(withDuration: self.errorShowInterval, animations: {
                if shouldHideTextfield {
                    self.theTextField.alpha = 0
                    self.theTextFieldLabel.alpha = 0
                } else {
                    self.theTextFieldLabel.alpha = 0
                }
                self.theTextFieldErrorLabel.frame = CGRect(x: self.theLeftImageWidth,
                                                           y: self.theTextFieldLabel.frame.origin.y,
                                                           width: self.theFrameWidth - self.theLeftImageWidth - self.theValidationImageWidth,
                                                           height: self.theTextFieldLabel.frame.height)
                self.theBottomBorderView.backgroundColor = self.theErrorColour
            }, completion: {
                (value: Bool) in
                UIView.animate(withDuration: self.errorShowInterval, delay: self.errorHideInterval, animations: {
                    self.theTextField.alpha = 1
                    self.theTextFieldLabel.alpha = 1
                    self.theTextFieldErrorLabel.frame = CGRect(x: self.theLeftImageWidth,
                                                               y: -self.theTextFieldLabel.frame.size.height,
                                                               width: self.theFrameWidth - self.theLeftImageWidth - self.theValidationImageWidth,
                                                               height: self.theTextFieldLabel.frame.size.height)
                    self.theBottomBorderView.backgroundColor = self.theBottomBorderColour
                })
            })
            DispatchQueue.main.asyncAfter(deadline: .now() + ((self.errorShowInterval * 2) + self.errorHideInterval)) {
                self.errorIsInAnimation = false
            }
        }
    }

    func showFailedValidationIcon() {
        self.theValidationImageView.image = nil
    }

    func passValidation() {
        self.theValidationImageView.image = nil
    }

    func failValidation() {
        self.animateTheValidationIcon()
        self.showFailedValidationIcon()
        self.flashError(message: nil)
    }

    func isValidEmail() -> Bool {
//        print("isValidEmail")
        let theEmailRegEX = "[A-Z0-9a-z._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,64}"
        let email = NSPredicate(format:"SELF MATCHES %@", theEmailRegEX)
        if email.evaluate(with: self.theTextField.text) {
            self.passValidation()
            return true
        } else {
            self.failValidation()
            return false
        }
    }
    
    func isEmailValid() -> Bool {
        //        print("isValidEmail")
        let theEmailRegEX = "[A-Z0-9a-z._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,64}"
        let email = NSPredicate(format:"SELF MATCHES %@", theEmailRegEX)
        if email.evaluate(with: self.theTextField.text) {
            return true
        } else {
            return false
        }
    }

    func isValidCharacterLength() -> Bool {
//        print("isValidCharacterLength")
        if (self.theTextField.text?.count)! >= self.minimumCharacterLimit {
            self.passValidation()
            return true
        } else {
            self.failValidation()
            return false
        }
    }
    
    func isCharacterLengthValid() -> Bool {
        if (self.theTextField.text?.count)! >= self.minimumCharacterLimit {
            return true
        }
        else {
            return false
        }
    }
    
    func invisibleCheckValidation(type: ValidationType) -> Bool {
        
        if self.theTextField.text?.count == 0 {
            return false
        }
        else {
            // check validation
            switch type {
            case .email:
                return self.isEmailValid()
            case .password, .standard:
                return self.isCharacterLengthValid()
            case .code:
                if (self.theTextField.text?.count)! > 5 {
                    return true
                } else {
                    return false
                }
            }
        }
    }

    func checkValidation(required: Bool, type: ValidationType) -> Bool {
        self.validationType = type
        if required {
            if self.theTextField.text?.count == 0 {
                self.failValidation()
                return false
            } else {
                // check validation
                switch type {
                case .email:
                    return self.isValidEmail()
                case .password, .standard:
                    return self.isValidCharacterLength()
                case .code:
                    if (self.theTextField.text?.count)! > 5 {
                        self.passValidation()
                        return true
                    } else {
                        self.failValidation()
                        return false
                    }
                }

            }
        }
        self.passValidation()
        return true
    }

    func softCheckValidation(required: Bool, type: ValidationType) -> Bool {
        self.validationType = type
        if required {
            if self.theTextField.text?.count == 0 {
                self.showFailedValidationIcon()
                return false
            } else {
                // check validation
                switch type {
                case .email:
                    if isValidEmail() {
                        self.passValidation()
                        return true
                    } else {
                        self.showFailedValidationIcon()
                        return false
                    }
                case .password: break
                case .standard: break
                case .code:
                    if (self.theTextField.text?.count)! > 5 {
                        self.passValidation()
                        return true
                    } else {
                        self.animateTheValidationIcon()
                        self.showFailedValidationIcon()
                        return false
                    }

                }

            }
        }
        self.passValidation()
        return true
    }

    func setNeedsValidation(validation: Bool) {
        self.theValidationImageView.isHidden = !validation
    }

    func setType(type: FieldType, validation_type: ValidationType) {
        self.type = type
        self.validationType = validation_type
        switch type {
        case .standard:
            self.theTextField.autocapitalizationType = .sentences
        case .email:
            self.theTextField.keyboardType = .emailAddress
        case .password:
            self.theTextField.isSecureTextEntry = true
            self.theTextField.keyboardType = .alphabet
        case .code:
            self.theTextField.keyboardType = .alphabet
        }
    }

    func layoutTheTextFieldLabel() {
        UIView.animate(withDuration: 0.2, animations: {
            if (self.theTextField.text?.count)! > 0 {
                self.theTextFieldLabel.frame = CGRect(x: 0,
                                                      y: 0,
                                                      width: self.theFrameWidth,
                                                      height: self.theTopLabelHeight)
                self.theLeftImageView.image = self.theLeftImageView.image?.imageWithColor(color1: UIColor.white)
            } else {
                self.theTextFieldLabel.frame = CGRect(x: self.theLeftImageWidth,
                                                      y: self.theFrameHeight - (self.theTextFieldHeight + self.theBottomBorderHeight),
                                                      width: self.theFrameWidth - self.theLeftImageWidth - self.theValidationImageWidth,
                                                      height: self.theTextFieldHeight)
                self.theLeftImageView.image = self.theLeftImageView.image?.imageWithColor(color1: self.theDisabledIconColour)
            }
        }, completion: nil)
    }

    override func layoutSubviews() {
        super.layoutSubviews()

        self.theFrameWidth = self.frame.size.width
        self.theFrameHeight = self.frame.size.height
        let theTextFieldY: CGFloat = self.theFrameHeight - (self.theTextFieldHeight + self.theBottomBorderHeight)

        self.theLeftImageView.frame = CGRect(x: 0, y: theTextFieldY, width: self.theLeftImageWidth, height: self.theTextFieldHeight)
        if (self.theTextField.text?.count)! > 0 {
            self.theTextFieldLabel.frame = CGRect(x: 0, y: 0, width: self.theFrameWidth, height: self.theTopLabelHeight)
        } else {
           self.theTextFieldLabel.frame = CGRect(x: self.theLeftImageWidth, y: theTextFieldY, width: self.theFrameWidth - self.theLeftImageWidth - self.theValidationImageWidth, height: self.theTextFieldHeight)
        }

        self.theTextField.frame = CGRect(x: self.theLeftImageWidth, y: theTextFieldY, width: self.theFrameWidth - self.theLeftImageWidth - self.theValidationImageWidth, height: self.theTextFieldHeight)
        self.theValidationImageView.frame = CGRect(x: self.theLeftImageWidth + self.theTextField.frame.size.width, y: theTextFieldY, width: self.theLeftImageWidth, height: self.theTextFieldHeight)
        self.theBottomBorderView.frame = CGRect(x: 0, y: self.theFrameHeight - self.theBottomBorderHeight, width: self.theFrameWidth, height: self.theBottomBorderHeight)
        self.theTextFieldErrorLabel.frame = CGRect(x: self.theLeftImageWidth, y: -self.theTextFieldLabel.frame.size.height, width: self.theFrameWidth - self.theLeftImageWidth - self.theValidationImageWidth, height: self.theTextFieldLabel.frame.size.height)
    }

    @objc func textFieldDidBeginEditing(_ textField: UITextField) {
        UIView.animate(withDuration: 0.2, animations: {
            self.theBottomBorderView.backgroundColor = self.theHighlightedBorderColour
        }, completion: nil)
    }

    @objc func textFieldDidEndEditing(_ textField: UITextField) {
        UIView.animate(withDuration: 0.2, animations: {
            self.theBottomBorderView.backgroundColor = self.theBottomBorderColour
        }, completion: nil)
    }

    @objc func textFieldDidChange(_ textField: UITextField) {
//        print("textFieldDidChange \(textField.text ?? "")")
        self.layoutTheTextFieldLabel()
        if (self.softValidation) {
            _ = self.softCheckValidation(required: true, type: self.validationType)
        }
    }

    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        self.endEditing(true)
//        print("textFieldShouldReturn \(textField.text ?? "")")
        return true
    }

}

extension UIImage {
    func imageWithColor(color1: UIColor) -> UIImage {
        UIGraphicsBeginImageContextWithOptions(self.size, false, self.scale)
        color1.setFill()
        
        let context = UIGraphicsGetCurrentContext()
        context?.translateBy(x: 0, y: self.size.height)
        context?.scaleBy(x: 1.0, y: -1.0)
        context?.setBlendMode(CGBlendMode.normal)
        
        let rect = CGRect(origin: .zero, size: CGSize(width: self.size.width, height: self.size.height))
        context?.clip(to: rect, mask: self.cgImage!)
        context?.fill(rect)
        
        let newImage = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        
        return newImage!
    }
}
