//
//  TAWAlertView.swift
//  TAW
//
//  Created by Andrew Steven on 11/02/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import Foundation
import UIKit

protocol TAWAlertViewDelegate: class {
    func acceptedNotifications()
    func unlinkAccount()
    func accountCreated()
    func didTapLogout()
    func freeSubsExpired()
}
extension TAWAlertViewDelegate
{
    func freeSubsExpired(){
        
    }
}

public enum TAWAlertStyle {
    case notifications
    case unlinkAccount
    case newAccount
    case logout
    case freeSubsExpired
}

public class TAWAlertView {
    
    var overlay = UIView()
    var alertView = UIView()
    var titleLabel = UILabel()
    var descriptionLabel = UILabel()
    
    
    weak var delegate: TAWAlertViewDelegate?
    
    class var shared: TAWAlertView {
        struct Static {
            static let instance: TAWAlertView = TAWAlertView()
        }
        return Static.instance
    }
    
    public func showAlertView(inView view: UIView, alertStyle style: TAWAlertStyle) {
        
        overlay = UIView(frame: UIScreen.main.bounds)
        overlay.backgroundColor = UIColor.black.withAlphaComponent(0.6)
        
        alertView = UIView(frame: CGRect(x: 0,
                                         y: overlay.frame.size.height,
                                         width: overlay.frame.size.width,
                                         height: 360))
        
        alertView.backgroundColor = UIColor.white
        overlay.addSubview(alertView)
        
        switch style {
        case .notifications:
            setupNotificationsAlert()
        case .unlinkAccount:
            setupUnlinkAccountAlert()
        case .newAccount:
            setupNewAccountAlert()
        case .logout:
            setupLogoutAlert()
        case .freeSubsExpired:
            setupFreeSubsExpireAlert()
        }
        
        view.addSubview(overlay)
        
        UIView.animate(withDuration: 0.3, delay: 0.1, options: .curveEaseIn, animations: {
            
            self.alertView.frame = CGRect(x: 0,
                                          y: self.overlay.frame.size.height - 360,
                                          width: self.overlay.frame.size.width,
                                          height: 360)
            
        }, completion: nil)
    }
    
    @IBAction func dismiss() {
        DispatchQueue.main.async {
            UIView.animate(withDuration: 0.3, delay: 0.1, options: .curveEaseOut, animations: {
                
                self.alertView.frame = CGRect(x: 0,
                                              y: self.overlay.frame.size.height,
                                              width: self.overlay.frame.size.width,
                                              height: 360)
                
            }, completion: { (_) in
                self.overlay.removeFromSuperview()
            })
        }
    }
    
    // MARK: - Notifications Alert -
    
    private func setupNotificationsAlert() {
        
        titleLabel = UILabel(frame: CGRect(x: 40,
                                           y: 30,
                                           width: alertView.frame.size.width - 80,
                                           height: 30))
        
        titleLabel.text = "ENABLE NOTIFICATIONS"
        titleLabel.font = TAW.Fonts.dinPro22
        
        titleLabel.textColor = UIColor(red: 20/255,
                                       green: 20/255,
                                       blue: 20/255,
                                       alpha: 1.0)
        
        titleLabel.textAlignment = .center
        titleLabel.numberOfLines = 0
        titleLabel.adjustsFontSizeToFitWidth = true
        titleLabel.minimumScaleFactor = 0.5
        alertView.addSubview(titleLabel)
        
        
        let paragraphStyle = NSMutableParagraphStyle()
        paragraphStyle.lineSpacing = 3
        let attributedText =
            NSAttributedString(string: "We will notify you when new content is added within the podcasts, videos, and writings feeds.",
                                                attributes: [NSAttributedString.Key.paragraphStyle: paragraphStyle])
        
        descriptionLabel = UILabel(frame: CGRect(x: 38,
                                                 y: titleLabel.frame.maxY,
                                                 width: alertView.frame.size.width - 76,
                                                 height: 80))
        
        descriptionLabel.attributedText = attributedText
        descriptionLabel.font = TAW.Fonts.dinAlternate16
        
        descriptionLabel.textColor = UIColor(red: 160/255,
                                             green: 162/255,
                                             blue: 171/255,
                                             alpha: 1.0)
        
        descriptionLabel.textAlignment = .center
        descriptionLabel.numberOfLines = 0
        alertView.addSubview(descriptionLabel)
        
        let allowButton = PBButton(frame: CGRect(x: 40,
                                                 y: descriptionLabel.frame.maxY + 68,
                                                 width: alertView.frame.size.width - 80,
                                                 height: 60))
        
        allowButton.setTitle("ENABLE NOTIFICATIONS", for: .normal)
        allowButton.addTarget(self, action: #selector(acceptedNotifications), for: .touchUpInside)
        allowButton.setTitleColor(UIColor.white, for: .normal)
        allowButton.titleLabel?.font = TAW.Fonts.dinAlternate16
        allowButton.buttonBackgroundColor = TAW.Colors.redButtonColor
        
        allowButton.shadowColor = UIColor(red: 237/255,
                                          green: 28/255,
                                          blue: 36/255,
                                          alpha: 0.35)
        
        allowButton.shadowOffset = CGSize(width: 0, height: 7)
        allowButton.shaddowRadius = 15
        allowButton.shaddowOppacity = 1
        alertView.addSubview(allowButton)
        
        let cancelButton = UIButton(frame: CGRect(x: 40,
                                                  y: allowButton.frame.maxY + 10,
                                                  width: alertView.frame.size.width - 80,
                                                  height: 60))
        
        cancelButton.setTitleColor(UIColor(red: 160/255,
                                           green: 162/255,
                                           blue: 171/255,
                                           alpha: 1.0),
                                   for: .normal)
        
        cancelButton.setTitle("NO THANKS", for: .normal)
        cancelButton.addTarget(self, action: #selector(dismiss), for: .touchUpInside)
        cancelButton.titleLabel?.font = TAW.Fonts.unitedSans13
        alertView.addSubview(cancelButton)
    }
    
    @IBAction func acceptedNotifications() {
        self.delegate?.acceptedNotifications()
        self.dismiss()
    }
    
    // MARK: - Unlink Account Alert -
    
    private func setupUnlinkAccountAlert() {
        
        titleLabel = UILabel(frame: CGRect(x: 40,
                                           y: 16,
                                           width: alertView.frame.size.width - 80,
                                           height: 40))
        
        titleLabel.text = "UNLINK ACCOUNT"
        titleLabel.font = TAW.Fonts.dinPro22
        titleLabel.textColor = UIColor(red: 20/255,
                                       green: 20/255,
                                       blue: 20/255,
                                       alpha: 1.0)
        
        titleLabel.textAlignment = .center
        titleLabel.numberOfLines = 0
        titleLabel.adjustsFontSizeToFitWidth = true
        titleLabel.minimumScaleFactor = 0.5
        alertView.addSubview(titleLabel)
        
        descriptionLabel = UILabel(frame: CGRect(x: 50,
                                                 y: titleLabel.frame.maxY,
                                                 width: alertView.frame.size.width - 100,
                                                 height: 100))
        
        descriptionLabel.text =
        "Are you sure you want to unlink your A Member account? This will remove any subscriptions you have within the app."
        descriptionLabel.font = TAW.Fonts.dinAlternate16
        descriptionLabel.textColor = UIColor(red: 160/255,
                                             green: 162/255,
                                             blue: 171/255,
                                             alpha: 1.0)
        descriptionLabel.textAlignment = .center
        descriptionLabel.numberOfLines = 0
        alertView.addSubview(descriptionLabel)
        
        let allowButton = PBButton(frame: CGRect(x: 40,
                                                 y: descriptionLabel.frame.maxY + 10,
                                                 width: alertView.frame.size.width - 80,
                                                 height: 60))
        
        allowButton.setTitle("UNLINK ACCOUNT", for: .normal)
        
        allowButton.addTarget(self,
                              action: #selector(unlinkAccount),
                              for: .touchUpInside)
        
        allowButton.setTitleColor(UIColor.white, for: .normal)
        allowButton.titleLabel?.font = TAW.Fonts.dinAlternate16
        allowButton.buttonBackgroundColor = TAW.Colors.redButtonColor
        
        allowButton.shadowColor = UIColor(red: 237/255,
                                          green: 28/255,
                                          blue: 36/255,
                                          alpha: 0.35)
        
        allowButton.shadowOffset = CGSize(width: 0, height: 7)
        allowButton.shaddowRadius = 15
        allowButton.shaddowOppacity = 1
        alertView.addSubview(allowButton)
        
        let cancelButton = UIButton(frame: CGRect(x: 40,
                                                  y: allowButton.frame.maxY + 10,
                                                  width: alertView.frame.size.width - 80,
                                                  height: 60))
        
        cancelButton.setTitleColor(UIColor(red: 160/255,
                                           green: 162/255,
                                           blue: 171/255,
                                           alpha: 1.0),
                                   for: .normal)
        
        cancelButton.setTitle("NO THANKS", for: .normal)
        cancelButton.addTarget(self, action: #selector(dismiss), for: .touchUpInside)
        cancelButton.titleLabel?.font = TAW.Fonts.unitedSans13
        alertView.addSubview(cancelButton)
    }
    
    @IBAction func unlinkAccount() {
        self.delegate?.unlinkAccount()
        self.dismiss()
    }
    
    // MARK: - New Account Alert -
    
    private func setupNewAccountAlert() {
        
        titleLabel = UILabel(frame: CGRect(x: 40,
                                           y: 16,
                                           width: alertView.frame.size.width - 80,
                                           height: 40))
        
        titleLabel.text = "WELCOME BONUS"
        titleLabel.font = TAW.Fonts.dinPro22
        titleLabel.textColor = UIColor(red: 20/255,
                                       green: 20/255,
                                       blue: 20/255,
                                       alpha: 1.0)
        
        titleLabel.textAlignment = .center
        titleLabel.numberOfLines = 0
        titleLabel.adjustsFontSizeToFitWidth = true
        titleLabel.minimumScaleFactor = 0.5
        alertView.addSubview(titleLabel)
        
        descriptionLabel = UILabel(frame: CGRect(x: 50,
                                                 y: titleLabel.frame.maxY,
                                                 width: alertView.frame.size.width - 100,
                                                 height: 100))
        
        descriptionLabel.text =
        "As a thank you for being an Anfield Wrap member, we have given you 250 free tokens to spend on any content you like."
        descriptionLabel.font = TAW.Fonts.dinAlternate16
        descriptionLabel.textColor = UIColor(red: 160/255,
                                             green: 162/255,
                                             blue: 171/255,
                                             alpha: 1.0)
        descriptionLabel.textAlignment = .center
        descriptionLabel.numberOfLines = 0
        alertView.addSubview(descriptionLabel)
        
        let allowButton = PBButton(frame: CGRect(x: 40,
                                                 y: descriptionLabel.frame.maxY + 10,
                                                 width: alertView.frame.size.width - 80,
                                                 height: 60))
        
        allowButton.setTitle("GOT IT", for: .normal)
        
        allowButton.addTarget(self,
                              action: #selector(newAccountCreated),
                              for: .touchUpInside)
        
        allowButton.setTitleColor(UIColor.white, for: .normal)
        allowButton.titleLabel?.font = TAW.Fonts.dinAlternate16
        allowButton.buttonBackgroundColor = TAW.Colors.redButtonColor
        
        allowButton.shadowColor = UIColor(red: 237/255,
                                          green: 28/255,
                                          blue: 36/255,
                                          alpha: 0.35)
        
        allowButton.shadowOffset = CGSize(width: 0, height: 7)
        allowButton.shaddowRadius = 15
        allowButton.shaddowOppacity = 1
        alertView.addSubview(allowButton)
    }
    
    private func setupFreeSubsExpireAlert() {
        
        titleLabel = UILabel(frame: CGRect(x: 40,
                                           y: 16,
                                           width: alertView.frame.size.width - 80,
                                           height: 40))
        
        titleLabel.text = "ALERT!!"
        titleLabel.font = TAW.Fonts.dinPro22
        titleLabel.textColor = UIColor(red: 20/255,
                                       green: 20/255,
                                       blue: 20/255,
                                       alpha: 1.0)
        
        titleLabel.textAlignment = .center
        titleLabel.numberOfLines = 0
        titleLabel.adjustsFontSizeToFitWidth = true
        titleLabel.minimumScaleFactor = 0.5
        alertView.addSubview(titleLabel)
        
        descriptionLabel = UILabel(frame: CGRect(x: 50,
                                                 y: titleLabel.frame.maxY,
                                                 width: alertView.frame.size.width - 100,
                                                 height: 100))
        
        descriptionLabel.text =
        "Your free subscription plan is expired.\nSubscribe Now!"
        descriptionLabel.font = TAW.Fonts.dinAlternate16
        descriptionLabel.textColor = UIColor(red: 160/255,
                                             green: 162/255,
                                             blue: 171/255,
                                             alpha: 1.0)
        descriptionLabel.textAlignment = .center
        descriptionLabel.numberOfLines = 0
        alertView.addSubview(descriptionLabel)
        
        let allowButton = PBButton(frame: CGRect(x: 40,
                                                 y: descriptionLabel.frame.maxY + 10,
                                                 width: alertView.frame.size.width - 80,
                                                 height: 60))
        
        allowButton.setTitle("OK", for: .normal)
        
        allowButton.addTarget(self,
                              action: #selector(freeSubsExpired),
                              for: .touchUpInside)
        
        allowButton.setTitleColor(UIColor.white, for: .normal)
        allowButton.titleLabel?.font = TAW.Fonts.dinAlternate16
        allowButton.buttonBackgroundColor = TAW.Colors.redButtonColor
        
        allowButton.shadowColor = UIColor(red: 237/255,
                                          green: 28/255,
                                          blue: 36/255,
                                          alpha: 0.35)
        
        allowButton.shadowOffset = CGSize(width: 0, height: 7)
        allowButton.shaddowRadius = 15
        allowButton.shaddowOppacity = 1
        alertView.addSubview(allowButton)
    }
    
    @IBAction func newAccountCreated() {
        self.delegate?.accountCreated()
        self.dismiss()
    }
    
    @IBAction func freeSubsExpired() {
        self.delegate?.freeSubsExpired()
        self.dismiss()
    }

    // MARK: - Logout Alert -

    private func setupLogoutAlert() {

        titleLabel = UILabel(frame: CGRect(x: 40,
                                           y: 16,
                                           width: alertView.frame.size.width - 80,
                                           height: 40))

        titleLabel.text = "LOGOUT"
        titleLabel.font = TAW.Fonts.dinPro22
        titleLabel.textColor = UIColor(red: 20/255,
                                       green: 20/255,
                                       blue: 20/255,
                                       alpha: 1.0)

        titleLabel.textAlignment = .center
        titleLabel.numberOfLines = 0
        titleLabel.adjustsFontSizeToFitWidth = true
        titleLabel.minimumScaleFactor = 0.5
        alertView.addSubview(titleLabel)

        descriptionLabel = UILabel(frame: CGRect(x: 50,
                                                 y: titleLabel.frame.maxY,
                                                 width: alertView.frame.size.width - 100,
                                                 height: 100))

        descriptionLabel.text =
        "Are you sure you want to logout?"
        descriptionLabel.font = TAW.Fonts.dinAlternate16
        descriptionLabel.textColor = UIColor(red: 160/255,
                                             green: 162/255,
                                             blue: 171/255,
                                             alpha: 1.0)
        descriptionLabel.textAlignment = .center
        descriptionLabel.numberOfLines = 0
        alertView.addSubview(descriptionLabel)

        let allowButton = PBButton(frame: CGRect(x: 40,
                                                 y: descriptionLabel.frame.maxY + 10,
                                                 width: alertView.frame.size.width - 80,
                                                 height: 60))

        allowButton.setTitle("LOGOUT", for: .normal)

        allowButton.addTarget(self,
                              action: #selector(logout),
                              for: .touchUpInside)

        allowButton.setTitleColor(UIColor.white, for: .normal)
        allowButton.titleLabel?.font = TAW.Fonts.dinAlternate16
        allowButton.buttonBackgroundColor = TAW.Colors.redButtonColor

        allowButton.shadowColor = UIColor(red: 237/255,
                                          green: 28/255,
                                          blue: 36/255,
                                          alpha: 0.35)

        allowButton.shadowOffset = CGSize(width: 0, height: 7)
        allowButton.shaddowRadius = 15
        allowButton.shaddowOppacity = 1
        alertView.addSubview(allowButton)

        let cancelButton = UIButton(frame: CGRect(x: 40,
                                                  y: allowButton.frame.maxY + 10,
                                                  width: alertView.frame.size.width - 80,
                                                  height: 60))

        cancelButton.setTitleColor(UIColor(red: 160/255,
                                           green: 162/255,
                                           blue: 171/255,
                                           alpha: 1.0),
                                   for: .normal)

        cancelButton.setTitle("CANCEL", for: .normal)
        cancelButton.addTarget(self, action: #selector(dismiss), for: .touchUpInside)
        cancelButton.titleLabel?.font = TAW.Fonts.unitedSans13
        alertView.addSubview(cancelButton)
    }

    @IBAction func logout() {
        self.delegate?.didTapLogout()
        self.dismiss()
    }
}
