package co.pixelbeard.theanfieldwrap.utils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Handler;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.util.ArrayList;
import java.util.List;

import co.pixelbeard.theanfieldwrap.data.NotificationSetting;
import co.pixelbeard.theanfieldwrap.data.Subscription;

/**
 * @author Michael Stoddart
 * Created on 04/01/2018.
 */

public class PrefUtils {

    private static final String TAG = PrefUtils.class.getSimpleName();
    private static final String PREF_NAME = "TAW-defaults";
    private static PrefUtils instance = null;

    private SharedPreferences mPrefs;
    private SharedPreferences.Editor mEditor;

    private PrefUtils(Context context) {
        mPrefs = context.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
    }

    public static synchronized PrefUtils getInstance(Context context) {
        if (instance == null) {
            instance = new PrefUtils(context);
        }
        return instance;
    }

    public static synchronized PrefUtils getInstance() {
        if (instance == null) {
            throw new IllegalStateException(TAG +
                    "is not initialized, call getInstance(Context) first");
        }
        return instance;
    }

    public SharedPreferences getPrefs() {
        return mPrefs;
    }

    public void putIntPref(PrefKey key, int val) {
        doEdit();
        mEditor.putInt(key.name(), val);
        doCommit();
    }

    public int getIntPref(PrefKey key) {
        int theInt = mPrefs.getInt(key.name(), 0);
        return theInt;
    }

    public void putStringPref(PrefKey key, String val) {
        doEdit();
        mEditor.putString(key.name(), val);
        doCommit();
    }

    public String getStringPref(PrefKey key) {
        return mPrefs.getString(key.name(), null);
    }

    public void putBooleanPref(PrefKey key, boolean val) {
        doEdit();
        mEditor.putBoolean(key.name(), val);
        doCommit();
    }

    public boolean getBooleanPref(PrefKey key) {
        boolean bool = mPrefs.getBoolean(key.name(), false);
        return bool;
    }


    public void putLongPref(PrefKey key, Long val) {
        doEdit();
        mEditor.putLong(key.name(), val);
        doCommit();
    }

    public Long getLongPref(PrefKey key) {
        return mPrefs.getLong(key.name(), 0L);
    }

    public void storeSubs(List<Subscription> subscriptions) {

        String subsAsJson = new Gson().toJson(subscriptions);
        putStringPref(PrefKey.USER_SUBS, subsAsJson);


    }

    public List<Subscription> getSubs() {

        String subsAsJson = getStringPref(PrefKey.USER_SUBS);
        if (subsAsJson != null) {
            return new Gson().fromJson(subsAsJson, new TypeToken<List<Subscription>>() {
            }.getType());
        } else {
            return new ArrayList<>();
        }
    }

    @SuppressLint("CommitPrefEdits")
    private void doEdit() {
        if (mEditor == null) {
            mEditor = mPrefs.edit();
        }
    }

    private void doCommit() {
        mEditor.apply();
        mEditor = null;
    }

    public void clearPrefs() {
        doEdit();
        long userId = PrefUtils.getInstance().getLongPref(PrefKey.USER_ID);
        String token = getStringPref(PrefKey.AUTH_TOKEN);
        mEditor.clear();
        doCommit();
        putStringPref(PrefKey.AUTH_TOKEN, token);

        if (userId != 0) {
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    putLongPref(PrefKey.PREV_LOGGED_IN_USER_ID, userId);
                }
            }, 2000);

        }
    }

    public List<NotificationSetting> getSettings() {
        String settingsAsJson = getStringPref(PrefKey.USER_SETTINGS);
        if (settingsAsJson != null) {
            return new Gson().fromJson(settingsAsJson, new TypeToken<List<NotificationSetting>>() {
            }.getType());
        } else {
            return new ArrayList<>();
        }
    }

    public void storeSettings(List<NotificationSetting> settings) {
        String settingsAsJson = new Gson().toJson(settings);
        putStringPref(PrefKey.USER_SETTINGS, settingsAsJson);
    }
}
