<?php
/**
 * The admin-options page of the plugin.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    PP_Pro
 * @subpackage PP_Pro/admin
 */

namespace PP_Pro\Admin;

use Podcast_Player\Helper\Functions\Getters as Get_Fn;
use Podcast_Player\Helper\Functions\Utility as Utility_Fn;
use Podcast_Player\Helper\Functions\Validation as Validation_Fn;

/**
 * The admin-options page of the plugin.
 *
 * @package    PP_Pro
 * @subpackage PP_Pro/admin
 * @author     vedathemes <contact@vedathemes.com>
 */
class Options {

	/**
	 * Holds the instance of this class.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    object
	 */
	protected static $instance = null;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {}

	/**
	 * Register hooked functions.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		$inst = self::get_instance();
		add_action( 'admin_enqueue_scripts', array( $inst, 'enqueue_admin_scripts' ) );
		add_action( 'wp_ajax_pp_feed_editor_edit', array( $inst, 'feed_editor' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_editor_edit', array( $inst, 'feed_editor' ) );
		add_action( 'wp_ajax_pp_feed_import', array( $inst, 'feed_import' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_import', array( $inst, 'feed_import' ) );
		add_action( 'wp_ajax_pp_feed_editor_sticky', array( $inst, 'sticky_player' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_editor_sticky', array( $inst, 'sticky_player' ) );
		add_action( 'wp_ajax_pp_add_bulk_players', array( $inst, 'bulk_player' ) );
		add_action( 'wp_ajax_nopriv_pp_add_bulk_players', array( $inst, 'bulk_player' ) );
		add_action( 'wp_ajax_pp_remove_bulk_players', array( $inst, 'remove_bulk_player' ) );
		add_action( 'wp_ajax_nopriv_pp_remove_bulk_players', array( $inst, 'remove_bulk_player' ) );
		add_action( 'wp_ajax_pp_feed_remove_sticky', array( $inst, 'remove_sticky_player' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_remove_sticky', array( $inst, 'remove_sticky_player' ) );
		add_action( 'wp_ajax_pp_feed_editor_save', array( $inst, 'save_custom_data' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_editor_save', array( $inst, 'save_custom_data' ) );
		add_action( 'wp_ajax_pp_feed_import_episodes', array( $inst, 'feed_import_episodes' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_import_episodes', array( $inst, 'feed_import_episodes' ) );
		add_action( 'wp_ajax_pp_feed_delete_imported', array( $inst, 'delete_imported' ) );
		add_action( 'wp_ajax_nopriv_pp_feed_delete_imported', array( $inst, 'delete_imported' ) );
		add_filter( 'podcast_player_setting_fields', array( $inst, 'add_pro_settings' ) );
	}

	/**
	 * Register the admin options scripts.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_admin_scripts() {
		$current_screen = get_current_screen();
		if ( $current_screen && 'toplevel_page_pp-options' === $current_screen->id ) {

			/**
			 * Enqueue admin scripts.
			 */
			wp_enqueue_script(
				'ppproadminoptions',
				PP_PRO_URL . 'admin/js/admin-options.build.js',
				array( 'ppadminoptions' ),
				PP_PRO_VERSION,
				true
			);

			/**
			 * Enqueue admin stylesheet.
			 */
			wp_enqueue_style(
				'ppproadminoptions',
				PP_PRO_URL . 'admin/css/admin-options.css',
				array(),
				PP_PRO_VERSION,
				'all'
			);
		}
	}

	/**
	 * Edit feed from using admin inputs via Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function feed_editor() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$feed_url = isset( $_POST['feedUrl'] ) ? wp_unslash( $_POST['feedUrl'] ) : '';
		$from     = isset( $_POST['rFrom'] ) ? sanitize_text_field( wp_unslash( $_POST['rFrom'] ) ) : '';

		if ( $from && 'indexKey' === $from ) {
			$fprn = sanitize_text_field( $feed_url );
			if ( ! $fprn ) {
				$output = array(
					'error' => esc_html__( 'Invalid feed key provided.', 'podcast-player' ),
				);
				echo wp_json_encode( $output );
				wp_die();
			}
		} else {
			$feed_url = esc_url_raw( $feed_url );
			if ( ! $feed_url ) {
				$output = array(
					'error' => esc_html__( 'Please provide a valid feed URL', 'podcast-player' ),
				);
				echo wp_json_encode( $output );
				wp_die();
			}

			$fprn = $feed_url;
		}

		$data_fields = array( 'title', 'author', 'featured', 'featured_id', 'season', 'categories' );
		$fdata       = Get_Fn::get_feed_data( $fprn, array(), $data_fields );

		if ( is_wp_error( $fdata ) ) {
			$output = array(
				'error' => esc_html( $fdata->get_error_message() ),
			);
			echo wp_json_encode( $output );
			wp_die();
		}

		// Ajax output to be returened.
		$output = array( 'data' => $fdata );
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Edit feed from using admin inputs via Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function feed_import() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$feed_url = isset( $_POST['feedUrl'] ) ? wp_unslash( $_POST['feedUrl'] ) : '';
		$from     = isset( $_POST['rFrom'] ) ? sanitize_text_field( wp_unslash( $_POST['rFrom'] ) ) : '';
		$podkey   = false;

		if ( $from && 'indexKey' === $from ) {
			$fprn = sanitize_text_field( $feed_url );
			if ( ! $fprn ) {
				$output = array(
					'error' => esc_html__( 'Invalid feed key provided.', 'podcast-player' ),
				);
				echo wp_json_encode( $output );
				wp_die();
			}
			$podkey = $fprn;
		} else {
			$feed_url = esc_url_raw( $feed_url );
			if ( ! $feed_url ) {
				$output = array(
					'error' => esc_html__( 'Please provide a valid feed URL', 'podcast-player' ),
				);
				echo wp_json_encode( $output );
				wp_die();
			}

			$fprn   = $feed_url;
			$podkey = md5( $fprn );
		}

		set_time_limit( 120 ); // Give it 2 minutes.

		$data_fields = array( 'title', 'post_id' );
		$fdata       = Get_Fn::get_feed_data( $fprn, array(), $data_fields );

		if ( is_wp_error( $fdata ) ) {
			$output = array(
				'error' => esc_html( $fdata->get_error_message() ),
			);
			echo wp_json_encode( $output );
			wp_die();
		}

		$findex = get_option( 'pp_feed_index' );
		if ( $findex && is_array( $findex ) && isset( $findex[ $podkey ] ) ) {
			$findex = $findex[ $podkey ];
			if ( isset( $findex['import'] ) ) {
				$findex = $findex['import'];
			} else {
				$findex = false;
			}
		}

		// Ajax output to be returened.
		$output = array(
			'idata' => array(
				'data'  => $fdata,
				'index' => $findex,
			),
		);
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Setup sticky player from admin inputs via Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function sticky_player() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$fmethod  = isset( $_POST['fmethod'] ) ? wp_unslash( $_POST['fmethod'] ) : '';
		$ptype    = isset( $_POST['ptype'] ) ? wp_unslash( $_POST['ptype'] ) : '';
		$taxon    = isset( $_POST['taxon'] ) ? wp_unslash( $_POST['taxon'] ) : '';
		$terms    = isset( $_POST['terms'] ) ? wp_unslash( $_POST['terms'] ) : array();
		$audio    = isset( $_POST['audio'] ) ? wp_unslash( $_POST['audio'] ) : '';
		$atitle   = isset( $_POST['atitle'] ) ? wp_unslash( $_POST['atitle'] ) : '';
		$feed_url = isset( $_POST['feedUrl'] ) ? wp_unslash( $_POST['feedUrl'] ) : '';
		$bg_color = isset( $_POST['bg'] ) ? wp_unslash( $_POST['bg'] ) : '';
		$ac_color = isset( $_POST['accent'] ) ? wp_unslash( $_POST['accent'] ) : '';
		$h_share  = isset( $_POST['hs'] ) ? wp_unslash( $_POST['hs'] ) : '';
		$h_down   = isset( $_POST['hd'] ) ? wp_unslash( $_POST['hd'] ) : '';
		$from     = isset( $_POST['rFrom'] ) ? sanitize_text_field( wp_unslash( $_POST['rFrom'] ) ) : '';

		if ( 'feed' === $fmethod && ! $feed_url ) {
			$output = array(
				'error' => esc_html__( 'Please provide a valid feed URL', 'podcast-player' ),
			);
			echo wp_json_encode( $output );
			wp_die();
		}

		if ( 'audio' === $fmethod && ! $audio ) {
			$output = array(
				'error' => esc_html__( 'Please provide a valid audio URL', 'podcast-player' ),
			);
			echo wp_json_encode( $output );
			wp_die();
		}

		if ( $feed_url && Validation_Fn::is_valid_url( $feed_url ) ) {
			$feed_url = esc_url_raw( $feed_url );
		} else {
			$feed_url = sanitize_text_field( $feed_url );
		}

		update_option(
			'pp_sticky_player_feed',
			array(
				'fmethod'    => sanitize_text_field( $fmethod ),
				'audio'      => esc_url_raw( $audio ),
				'atitle'     => sanitize_text_field( $atitle ),
				'posttype'   => sanitize_text_field( $ptype ),
				'taxonomies' => sanitize_text_field( $taxon ),
				'terms'      => array_map( 'sanitize_text_field', $terms ),
				'url'        => $feed_url,
				'bg'         => sanitize_hex_color( $bg_color ),
				'accent'     => sanitize_hex_color( $ac_color ),
				'share'      => sanitize_text_field( $h_share ),
				'dl'         => sanitize_text_field( $h_down ),
			),
			'no'
		);

		$output = array(
			'message' => esc_html__( 'Player has been added successfully.', 'podcast-player' ),
		);
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Setup bulk player from admin inputs via Ajax calls.
	 *
	 * @since 4.3.0
	 */
	public function bulk_player() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$audio      = isset( $_POST['audio'] ) ? wp_unslash( $_POST['audio'] ) : '';
		$cf         = isset( $_POST['cf'] ) ? wp_unslash( $_POST['cf'] ) : '';
		$loc        = isset( $_POST['location'] ) ? wp_unslash( $_POST['location'] ) : '';
		$style      = isset( $_POST['style'] ) ? wp_unslash( $_POST['style'] ) : '';
		$h_title    = isset( $_POST['hT'] ) ? wp_unslash( $_POST['hT'] ) : '';
		$h_featured = isset( $_POST['hF'] ) ? wp_unslash( $_POST['hF'] ) : '';
		$bg_color   = isset( $_POST['bg'] ) ? wp_unslash( $_POST['bg'] ) : '';
		$ac_color   = isset( $_POST['accent'] ) ? wp_unslash( $_POST['accent'] ) : '';
		$h_share    = isset( $_POST['hs'] ) ? wp_unslash( $_POST['hs'] ) : '';
		$h_down     = isset( $_POST['hd'] ) ? wp_unslash( $_POST['hd'] ) : '';

		if ( ! $audio ) {
			$output = array(
				'error' => esc_html__( 'Please provide an audio source.', 'podcast-player' ),
			);
			echo wp_json_encode( $output );
			wp_die();
		}

		update_option(
			'pp_bulk_player',
			array(
				'audio'     => sanitize_text_field( $audio ),
				'cf'        => sanitize_text_field( $cf ),
				'loc'       => sanitize_text_field( $loc ),
				'style'     => sanitize_text_field( $style ),
				'htitle'    => sanitize_text_field( $h_title ),
				'hfeatured' => sanitize_text_field( $h_featured ),
				'bg'        => sanitize_hex_color( $bg_color ),
				'accent'    => sanitize_hex_color( $ac_color ),
				'share'     => sanitize_text_field( $h_share ),
				'dl'        => sanitize_text_field( $h_down ),
			),
			'no'
		);

		$output = array(
			'message' => esc_html__( 'Players have been added successfully.', 'podcast-player' ),
		);
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Setup sticky player from admin inputs via Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function remove_sticky_player() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$option = get_option( 'pp_sticky_player_feed' );
		if ( $option ) {
			delete_option( 'pp_sticky_player_feed' );
			$output = array(
				'message' => esc_html__( 'Sticky player has been deleted.', 'podcast-player' ),
			);
		} else {
			$output = array(
				'error' => esc_html__( 'No Sticky player was set.', 'podcast-player' ),
			);
		}
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Remove Bulk player from admin inputs via Ajax calls.
	 *
	 * @since 4.3.0
	 */
	public function remove_bulk_player() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$option = get_option( 'pp_bulk_player' );
		if ( $option ) {
			delete_option( 'pp_bulk_player' );
			$output = array(
				'message' => esc_html__( 'Bulk player has been deleted.', 'podcast-player' ),
			);
		} else {
			$output = array(
				'error' => esc_html__( 'No Bulk player was set.', 'podcast-player' ),
			);
		}
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Edit feed from using admin inputs via Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function save_custom_data() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$podcast  = isset( $_POST['podcast'] ) ? sanitize_text_field( wp_unslash( $_POST['podcast'] ) ) : '';
		$episode  = isset( $_POST['episode'] ) ? sanitize_text_field( wp_unslash( $_POST['episode'] ) ) : '';
		$title    = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : false;
		$author   = isset( $_POST['author'] ) ? sanitize_text_field( wp_unslash( $_POST['author'] ) ) : false;
		$season   = isset( $_POST['season'] ) && 'false' !== $_POST['season'] ? absint( wp_unslash( $_POST['season'] ) ) : false;
		$featured = isset( $_POST['featured'] )  && 'false' !== $_POST['featured'] ? absint( wp_unslash( $_POST['featured'] ) ) : false;
		$cats     = isset( $_POST['cats'] ) ? sanitize_text_field( wp_unslash( $_POST['cats'] ) ) : '';
		$cats     = false !== $cats && 'false' !== $cats ? explode( ',', $cats ) : false;
		if ( false !== $cats ) {
			$fcats = array();
			foreach ( $cats as $cat ) {
				$cat           = trim( $cat );
				$key           = strtolower( str_replace( ' ', '', $cat ) );
				$fcats[ $key ] = $cat;
			}
		} else {
			$fcats = false;
		}

		if ( ! $podcast || ! $episode ) {
			$output = array(
				'error' => esc_html__( 'Insufficient data provided.', 'podcast-player' ),
			);
			echo wp_json_encode( $output );
			wp_die();
		}

		$fimg = false;
		if (false !== $featured) {
			if ( $featured ) {
				$fimg = wp_get_attachment_image_src( $featured, 'large' );
				if ( $fimg ) {
					$fimg = esc_url( $fimg[0] );
				}
			} else {
				$fimg = '';
			}
			
			$fimg = $fimg ? $fimg : '';
		}

		$mod_data = array(
			'title'       => $title,
			'author'      => $author,
			'season'      => $season,
			'categories'  => $fcats,
			'featured'    => $fimg,
			'featured_id' => $featured,
		);
		$mod_data = array_filter(
			$mod_data,
			function( $val ) {
				return false !== $val && 'false' !== $val;
			}
		);

		if ( empty( $mod_data ) ) {
			$output = array(
				'error' => esc_html__( 'Nothing to update.', 'podcast-player' ),
			);
			echo wp_json_encode( $output );
			wp_die();
		}

		$data_key    = 'pp_feed_data_custom_' . $podcast;
		$custom_data = get_option( $data_key );

		if ( false === $custom_data ) {
			$custom_data = array( $episode => array_filter( $mod_data ) );
		} elseif ( isset( $custom_data[ $episode ] ) && is_array( $custom_data[ $episode ] ) ) {
			$custom_data[ $episode ] = array_filter( array_merge( $custom_data[ $episode ], $mod_data ) );
		} else {
			$custom_data[ $episode ] = array_filter( $mod_data );
		}

		update_option( 'pp_feed_data_custom_' . $podcast, $custom_data, 'no' );

		$output = array(
			'message' => esc_html__( 'Data saved successfully.', 'podcast-player' ),
		);
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Import podcast episodes from feed to WordPress.
	 *
	 * @since 5.1.0
	 */
	public function feed_import_episodes() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$data  = isset( $_POST['idata'] ) ? wp_unslash( $_POST['idata'] ) : '';
		$pcall = isset( $_POST['pcall'] ) ? wp_unslash( $_POST['pcall'] ) : false;

		// Get relevant data.
		$podcast = isset( $data['podkey'] ) ? sanitize_text_field( $data['podkey'] ) : false;
		$elist   = isset( $data['episodes'] ) && is_array( $data['episodes'] ) ? array_map( 'sanitize_text_field', $data['episodes'] ) : false;
		$pdata   = isset( $data['post'] ) ? $data['post'] : false;
		$player  = isset( $data['player'] ) ? $data['player'] : false;

		// Return if sufficient data is not available.
		if ( ! $podcast || ! $elist || ! $pdata || ! $player ) {
			echo wp_json_encode( array( 'error' => esc_html__( 'Insufficient data provided.', 'podcast-player' ) ) );
			wp_die();
		}

		// Get post specific data.
		$post_type   = isset( $pdata['type'] ) ? sanitize_text_field( $pdata['type'] ) : 'post';
		$post_status = isset( $pdata['status'] ) ? sanitize_text_field( $pdata['status'] ) : 'draft';
		$author_id   = isset( $pdata['author'] ) ? absint( $pdata['author'] ) : 0;
		$is_get_img  = isset( $pdata['featured'] ) && 'true' === $pdata['featured'] ? true : false;

		if ( $pcall && 'false' !== $pcall ) {
			// Get player specific data.
			$location = isset( $player['location'] ) ? sanitize_text_field( $player['location'] ) : '';
			$style    = isset( $player['style'] ) ? sanitize_text_field( $player['style'] ) : '';
			$bgcolor  = isset( $player['bgcolor'] ) ? sanitize_hex_color( $player['bgcolor'] ) : '';
			$accent   = isset( $player['acccolor'] ) ? sanitize_hex_color( $player['acccolor'] ) : '';
			$hsocial  = isset( $player['hideSocial'] ) && $player['hideSocial'] && 'true' === $player['hideSocial'] ? true : false;
			$hdown    = isset( $player['hideDownload'] ) && $player['hideDownload'] && 'true' === $player['hideDownload'] ? true : false;
			$htitle   = isset( $player['hideTitle'] ) && $player['hideTitle'] && 'true' === $player['hideTitle'] ? true : false;

			// Create data array.
			$pp_data = array(
				'post_type'     => $post_type,
				'post_status'   => $post_status,
				'author'        => $author_id,
				'is_get_img'    => $is_get_img,
				'location'      => $location,
				'style'         => $style,
				'bgcolor'       => $bgcolor,
				'accent'        => $accent,
				'hide_social'   => $hsocial,
				'hide_download' => $hdown,
				'hide_title'    => $htitle,
			);

			$option = get_option( 'pp_feed_index' );

			if ( ! $option || ! is_array( $option ) ) {
				$option = array( $podcast => array( 'import' => $pp_data ) );
			} elseif ( ! isset( $option[ $podcast ] ) ) {
				$option[ $podcast ] = array( 'import' => $pp_data );
			} else {
				$option[ $podcast ]['import'] = $pp_data;
			}

			if ( $option ) {
				update_option( 'pp_feed_index', $option, 'no' );
			}
		}

		// Episode data to be fetched from the feed.
		$data_fields = array( 'title', 'description', 'date', 'src', 'featured', 'mediatype' );

		// Get required episodes data from the feed.
		$fdata = Get_Fn::get_feed_data( $podcast, array( 'elist' => $elist ), $data_fields );

		// Return error message if feed data is not proper.
		if ( is_wp_error( $fdata ) ) {
			echo wp_json_encode( array( 'error' => esc_html( $fdata->get_error_message() ) ) );
			wp_die();
		}

		$data_key    = 'pp_feed_data_custom_' . $podcast;
		$custom_data = get_option( $data_key );
		if ( false === $custom_data ) {
			$custom_data = array();
		}

		// Creating WP posts / post types from feed episodes.
		$post_items = $fdata['items'];
		foreach ( $post_items as $key => $items ) {
			$date    = strtotime( (string) $items['date'] );
			$date    = date( 'Y-m-d H:i:s', $date );
			$post_id = wp_insert_post(
				array(
					'post_author'  => $author_id,
					'post_content' => $items['description'],
					'post_date'    => $date,
					'post_status'  => $post_status,
					'post_title'   => $items['title'],
					'post_type'    => $post_type,
				)
			);

			// Return error message if the import generate errors.
			if ( is_wp_error( $post_id ) ) {
				echo wp_json_encode( array( 'error' => esc_html( $post_id->get_error_message() ) ) );
				wp_die();
			}

			// Add post specific information.
			add_post_meta(
				$post_id,
				'pp_import_data',
				array(
					'podkey'  => sanitize_text_field( $podcast ),
					'episode' => sanitize_text_field( $key ),
					'src'     => esc_url_raw( $items['src'] ),
					'type'    => sanitize_text_field( $items['mediatype'] ),
				)
			);

			// Conditionally import and set post features image.
			if ( $is_get_img && isset( $items['featured'] ) && $items['featured'] ) {
				$img_id = Utility_Fn::upload_image( $items['featured'], $items['title'] );
				if ( $img_id ) {
					set_post_thumbnail( $post_id, $img_id );
				}
			}

			if ( isset( $custom_data[ $key ] ) && is_array( $custom_data[ $key ] ) ) {
				$custom_data[ $key ]['post_id'] = $post_id;
			} else {
				$custom_data[ $key ] = array( 'post_id' => $post_id );
			}
		}

		// Saving post_ids against feed episodes.
		update_option( 'pp_feed_data_custom_' . $podcast, $custom_data, 'no' );

		// Success Message.
		$output = array(
			'data'    => $custom_data,
			'message' => esc_html__( 'Episodes Imported successfully.', 'podcast-player' ),
		);
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Delete imported episodes.
	 *
	 * @since 5.1.0
	 */
	public function delete_imported() {
		check_ajax_referer( 'podcast-player-admin-options-ajax-nonce', 'security' );

		$data = isset( $_POST['idata'] ) ? wp_unslash( $_POST['idata'] ) : '';

		// Get relevant data.
		$podcast = isset( $data['podkey'] ) ? sanitize_text_field( $data['podkey'] ) : false;
		$id      = isset( $data['epiId'] ) && is_array( $data['epiId'] ) ? array_map( 'absint', $data['epiId'] ) : false;
		$key     = isset( $data['epiKey'] ) && is_array( $data['epiKey'] ) ? array_map( 'sanitize_text_field', $data['epiKey'] ) : false;

		// Return if sufficient data is not available.
		if ( ! $podcast || ! $id || ! $key || count( $id ) !== count( $key ) ) {
			echo wp_json_encode( array( 'error' => esc_html__( 'Insufficient data provided.', 'podcast-player' ) ) );
			wp_die();
		}

		$post_items  = array_combine( $key, $id );
		$data_key    = 'pp_feed_data_custom_' . $podcast;
		$custom_data = get_option( $data_key );
		if ( false === $custom_data ) {
			echo wp_json_encode( array( 'error' => esc_html__( 'Podcast import data is not available.', 'podcast-player' ) ) );
			wp_die();
		}

		foreach ( $post_items as $k => $i ) {
			if ( isset( $custom_data[ $k ] ) && is_array( $custom_data[ $k ] ) && isset( $custom_data[ $k ]['post_id'] ) ) {
				wp_delete_post( $i, true );
				unset( $custom_data[ $k ]['post_id'] );
			}
		}

		// Saving post_ids against feed episodes.
		update_option( 'pp_feed_data_custom_' . $podcast, $custom_data, 'no' );

		// Success Message.
		$output = array(
			'message' => esc_html__( 'Episodes Deleted successfully.', 'podcast-player' ),
		);
		echo wp_json_encode( $output );
		wp_die();
	}

	/**
	 * Add pro specific admin settings.
	 *
	 * @since 5.2.0
	 *
	 * @param array $settings Admin settings.
	 */
	public function add_pro_settings( $settings ) {
		return array_merge(
			$settings,
			array(
				'sticky_on_scroll' => array(
					'name'        => esc_html__( 'Make player sticky on scroll', 'podcast-player' ),
					'id'          => 'sticky_on_scroll',
					'description' => esc_html__( 'If currently playing podcast player (having default or legacy display style) move out of visible area while scrolling, it will be converted to bottom sticky player.', 'podcast-player' ),
					'link'        => '',
					'type'        => 'checkbox',
					'default'     => 'yes',
					'section'     => 'general',
				),
			)
		);
	}

	/**
	 * Returns the instance of this class.
	 *
	 * @since  1.0.0
	 *
	 * @return object Instance of this class.
	 */
	public static function get_instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}
}
